
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.util.arl;

import java.io.StringReader ;
import org.apache.xml.utils.WrappedRuntimeException;

/**
 * ArlUtil class utility (static methods).
 * <p>
 * "ARL" stands for "ARray Language". It is a very basic language that allows
 * to set elements of an <code>int</code> array to one of three flag values:
 * <code>F_ACCEPT</code>, <code>F_REMOVE</code> or <code>F_CROSS</code>.
 * <p>
 * An ARL sequence looks like <code>"~23;+34;-45;-[46,56];..."</code> The
 * meaning of this expression is: "set the 23th element of the array to
 * <code>F_CROSS</code>; set the 34th element to <code>F_ACCEPT</code>; set
 * the 45th to <code>F_REMOVE</code>; set all elements between and including
 * the 46th and the 56th to <code>F_REMOVE</code>;..."
 * <p>
 * Order is important: <code>"-[23,56];~34"</code> means "set all elements of
 * the array between and including 23 and 56 to <code>F_REMOVE</code>; <i>then</i>
 * set the 34th element to <code>F_CROSS</code>". On the other hand,
 * <code>"~34;-[23,56]"</code> is strictly identical to <code>"-[23,56]"</code>.
 * <p>
 * The "+" sign (for <code>F_ACCEPT</code>) is optional: <code>"+45"</code> is identical
 * to <code>"45"</code>.
 * <p>
 * This language is used for parsers control
 * (see {@link org.ejen.ext.parsers.java_1_2.JavaSourceToXML#process(
 * org.apache.xalan.extensions.ExpressionContext,String,String,String)
 * JavaSourceToXML.process(ExpressionContext,String,String,String)},
 * {@link org.ejen.ext.parsers.java_1_2.SimpleNode#toNode(org.w3c.dom.Document,
 *                                                        org.w3c.dom.Node,
 *                                                        int[],
 *                                                        int[],
 *                                                        boolean)
 * SimpleNode.toNode(Document,Node,int[],int[],boolean)}
 * and {@link org.ejen.ext.parsers.java_1_2.Token#toNode(org.w3c.dom.Document,
 *                                                       org.w3c.dom.Node,
 *                                                       int[],
 *                                                       boolean)
 * Token.toNode(Document,Node,int[],boolean)}).
 * <p>
 * @author F. Wolff
 * @version 1.0
 */
public class ArlUtil implements ArlParserTreeConstants, ArlParserConstants {
    
    public static final int F_ACCEPT = 0;
    public static final int F_REMOVE = 1;
    public static final int F_CROSS = 2;

    /**
     * Prevents intanciation.
     */
    protected ArlUtil() {}

    /**
     * Returns the array parameter whose elements are set according to the
     * arl parameter.
     * @param arl an ARL sequence.
     * @param array the array to modify.
     * @return the modified array parameter.
     * @throws org.apache.xml.utils.WrappedRuntimeException parse exception,
     *         array out of bound exception...
     */
    public static int[] process(String  arl, int[] array) {
        if (array == null) {
            return null;
        }
        StringReader  sr = null;

        try {
            sr = new StringReader (arl);
            if (ArlParser.token_source == null) {
                new ArlParser(sr);
            } else {
                ArlParser.ReInit(sr);
            }
            SimpleNode sn = ArlParser.ArlSequence();

            for (int i = 0; i < sn.jjtGetNumChildren(); i++) {
                int flag = F_ACCEPT;
                SimpleNode exp = (SimpleNode) (sn.jjtGetChild(i));
                Token t = exp.getFirstToken();

                if (t.kind >= ACCEPT && t.kind <= CROSS) {
                    if (t.kind == REMOVE) {
                        flag = F_REMOVE;
                    } else if (t.kind == CROSS) {
                        flag = F_CROSS;
                    }
                    t = t.next;
                }
                if (exp.jjtGetNumChildren() == 0) {
                    array[Integer.parseInt(t.image)] = flag;
                } else {
                    exp = (SimpleNode) (exp.jjtGetChild(0));
                    t = exp.getFirstToken().next;
                    int i1 = Integer.parseInt(t.image);
                    int i2 = Integer.parseInt(t.next.next.image);

                    for (int j = i1; j <= i2; j++) {
                        array[j] = flag;
                    }
                }
            }
            return array;
        } catch (Exception  e) {
            throw new WrappedRuntimeException("Bad Arl sequence: " + arl, e);
        }
        finally {
            if (sr != null) {
                try {
                    sr.close();
                } catch (Exception  e) {}
                finally {
                    sr = null;
                }
            }
        }
    }
}
