
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.util;

import org.ejen.EjenConstants;
import java.util.Properties ;
import java.io.InputStream ;
import java.io.FileInputStream ;
import java.io.ByteArrayInputStream ;
import javax.xml.parsers.DocumentBuilderFactory ;
import javax.xml.parsers.DocumentBuilder ;
import org.xml.sax.InputSource ;
import org.w3c.dom.Node ;
import org.w3c.dom.Element ;
import org.w3c.dom.NamedNodeMap ;
import org.w3c.dom.Document ;
import org.w3c.dom.DOMException ;
import org.apache.xml.utils.WrappedRuntimeException;

/**
 * DOM utility (static methods used in java code).
 * @author F. Wolff
 * @version 1.0
 */
public class DOMUtil {
    public static final String  S_PY_PROPERTY_NODE_NAME = "property";
    public static final String  S_PY_NAME = "name";
    public static final String  S_PY_VALUE = "value";

    /**
     * Returns a Node attribute value.
     * @param n the <code>Node</code> from which the attribute must be retreived.
     * @param name the name of the attribute.
     * @return the attribute value or <code>null</code> if there is no such attribute.
     */
    public static String  getAttribute(Node  n, String  name) {
        if (n != null && name != null) {
            NamedNodeMap  nnm = n.getAttributes();

            if (nnm != null) {
                Node  attr = nnm.getNamedItem(name);

                if (attr != null) {
                    return attr.getNodeValue();
                }
            }
        }
        return null;
    }

    /**
     * Returns a property value.
     * <p>
     * The <code>n</code> parameter must have the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;property name="..." value="..."&gt;
     * </pre></td></tr></table>
     * <p>
     * @param n the <code>Node</code> from which the property must be retreived.
     * @param name the name of the property (<i>ie: the value of the "name" attribute</i>).
     * @return the property value (<i>ie: the value of the "value" attribute</i>)
     *         or <code>null</code> if there is no such property.
     */
    public static String  getProperty(Node  n, String  name) {
        if (n != null && name != null) {
            String [] prop = getProperty(n);

            if (prop != null && name.equals(prop[0])) {
                return prop[1];
            }
        }
        return null;
    }

    /**
     * Returns a property as an array of two Strings.
     * <p>
     * The <code>n</code> parameter must have the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;property name="..." value="..."&gt;
     * </pre></td></tr></table>
     * <p>
     * @param n the <code>Node</code> from which the property must be retreived.
     * @return the property name/value pair (<i>ie: the value of the "name" attribute and
     *         the value of the "value" attribute</i>)
     *         or <code>null</code> if there is no such property.
     */
    public static String [] getProperty(Node  n) {
        if (n != null) {
            if (S_PY_PROPERTY_NODE_NAME.equals(n.getNodeName())) {
                NamedNodeMap  nnm = n.getAttributes();

                if (nnm != null) {
                    Node  name = nnm.getNamedItem(S_PY_NAME);
                    Node  value = nnm.getNamedItem(S_PY_VALUE);

                    if (name != null && value != null) {
                        return new String [] {name.getNodeValue(),
                            value.getNodeValue()};
                    }
                }
            }
        }
        return null;
    }

    /**
     * Returns a set of properties from a given parent <code>Node</code>.
     * <p>
     * Each child property must have the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;property name="..." value="..."&gt;
     * </pre></td></tr></table>
     * <p>
     * Other kind of child nodes are ignored.
     * <p>
     * @param n the <code>Node</code> from which the child properties must be retreived.
     * @return the set of child properties (empty if there is no child property).
     */
    public static Properties  getChildProperties(Node  n) {
        return getChildProperties(n, new Properties ());
    }

    /**
     * Appends to an existing set of properties those found in a given parent <code>Node</code>.
     * <p>
     * Each child property must have the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;property name="..." value="..."&gt;
     * </pre></td></tr></table>
     * <p>
     * Other kind of child nodes are ignored.
     * <p>
     * @param n the <code>Node</code> from which the child properties must be retreived.
     * @param props the existing properties.
     * @return the <code>props</code> parameter (with appended child properties).
     */
    public static Properties  getChildProperties(Node  parent, Properties  props) {
        if (props != null) {
            for (Node  n = parent.getFirstChild(); n != null; n = n.getNextSibling()) {
                String [] prop = getProperty(n);

                if (prop != null) {
                    props.setProperty(prop[0], prop[1]);
                }
            }
        }
        return props;
    }

    /**
     * Create a new <code>Document</code> object.
     * @return the new <code>Document</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException creation error.
     */
    public static Document  newDocument() {
        try {
            DocumentBuilderFactory  dbf = DocumentBuilderFactory.newInstance();

            dbf.setNamespaceAware(true);
            DocumentBuilder  db = dbf.newDocumentBuilder();

            return db.newDocument();
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Parse a XML document file.
     * @param name the XML file name.
     * @return the parsed <code>Document</code> object.
     * @throws org.apache.xml.utils.WrappedRuntimeException error (no such file, parse error).
     */
    public static Document  parseXMLFile(String  name) {
        InputStream  inputs = null;

        try {
            inputs = new FileInputStream (name);
            DocumentBuilderFactory  dbf = DocumentBuilderFactory.newInstance();

            dbf.setNamespaceAware(true);
            return dbf.newDocumentBuilder().parse(new InputSource (inputs));
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        finally {
            if (inputs != null) {
                try {
                    inputs.close();
                } catch (Exception  e) {}
                finally {
                    inputs = null;
                }
            }
        }
    }

    /**
     * Parse a XML document <code>String</code> (using <code>DEFAULT_XML_DATA_ENCODING</code>
     * encoding).
     * @param xmlString the XML content <code>String</code>. If <code>xmlString</code>
     *        is <code>null</code>, the <code>DEFAULT_XML_DATA</code> is used.
     * @return the parsed <code>Document</code> object.
     * @throws org.apache.xml.utils.WrappedRuntimeException error (parse error).
     */
    public static Document  parseXMLString(String  xmlString) {
        InputStream  inputs = null;

        try {
            if (xmlString == null) {
                inputs = new ByteArrayInputStream (EjenConstants.DEFAULT_XML_DATA.getBytes(EjenConstants.DEFAULT_XML_DATA_ENCODING));
            } else {
                inputs = new ByteArrayInputStream (xmlString.getBytes(EjenConstants.DEFAULT_XML_DATA_ENCODING));
            }
            DocumentBuilderFactory  dbf = DocumentBuilderFactory.newInstance();

            dbf.setNamespaceAware(true);
            return dbf.newDocumentBuilder().parse(new InputSource (inputs));
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Creates a new Node of name 'nodeName', in the 'doc' Document, child of the 'parent' Node.
     * @param doc the owning Document of the new Node.
     * @param parent the parent Node of the new Node.
     * @param nodeName name of the <code>Node</code> to be created.
     * @return the created Node.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM errors.
     */
    public static Node  createNode(Document  doc, Node  parent, String  nodeName) {
        try {
            Element  elt = doc.createElement(nodeName);

            parent.appendChild(elt);
            return elt;
        } catch (DOMException  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Creates a new Node of name 'nodeName', in the 'doc' Document, child of the 'parent' Node,
     * and fills it with a CDATA Node whose content is 'nodeValue'.
     * @param doc the owning Document of the new Node.
     * @param parent the parent Node of the new Node.
     * @param nodeName name of the <code>Node</code> to be created.
     * @param nodeValue value of the <code>CDATA Node</code> to be created.
     * @return the created Node (parent of the CDATA Node).
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM errors.
     */
    public static Node  createCDATANode(Document  doc, Node  parent, String  nodeName, String  nodeValue) {
        try {
            Element  elt = doc.createElement(nodeName);

            elt.appendChild(doc.createCDATASection(nodeValue));
            parent.appendChild(elt);
            return elt;
        } catch (DOMException  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Creates a new Node of name 'nodeName', in the 'doc' Document,
     * and fills it with a CDATA Node whose content is 'nodeValue'.
     * @param doc the owning Document of the new Node.
     * @param nodeName name of the <code>Node</code> to be created.
     * @param nodeValue value of the <code>CDATA Node</code> to be created.
     * @return the created Node (parent of the CDATA Node).
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM errors.
     */
    public static Node  createCDATANode(Document  doc, String  nodeName, String  nodeValue) {
        try {
            Element  elt = doc.createElement(nodeName);

            elt.appendChild(doc.createCDATASection(nodeValue));
            return elt;
        } catch (DOMException  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Creates a new Node of name 'nodeName', in the 'doc' Document,
     * and fills it with a text Node whose content is 'nodeValue'.
     * @param doc the owning Document of the new Node.
     * @param nodeName name of the <code>Node</code> to be created.
     * @param nodeValue value of the text <code>Node</code> to be created.
     * @return the created Node (parent of the text Node).
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM errors.
     */
    public static Node  createTextNode(Document  doc, String  nodeName, String  nodeValue) {
        try {
            Element  elt = doc.createElement(nodeName);

            elt.appendChild(doc.createTextNode(nodeValue));
            return elt;
        } catch (DOMException  e) {
            throw new WrappedRuntimeException(e);
        }
    }
}
