
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext.db;

import org.ejen.util.XSLUtil;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.Node ;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xpath.NodeSet;
import org.apache.xml.utils.WrappedRuntimeException;

/**
 * Table metadata <b>abstract</b> class
 * (see {@link org.ejen.ext.db.BasicMetaDataConnection} for usage).
 * @author F. Wolff
 * @version 1.0
 */
public abstract class TableMetaDataNodeBuilder extends MetaDataNodeBuilder {
    public static final String  S_TABLE_METADATA_NODE_NAME = "metadata";

    /**
     * Gets table metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getTableMetaData('ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned <code>Node</code> has the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;metadata&gt;
     *    &lt;primary-key column-name="ID"
     *                 key-seq="1"
     *                 pk-name="SYSTEM_PK"
     *                 table-cat="#NULL"
     *                 table-name="ADDRESS"
     *                 table-shem="#NULL"/>
     *    &lt;index asc-or-desc="A"
     *           cardinality="#NULL"
     *           column-name="ID"
     *           filter-condition="#NULL"
     *           index-name="SYSTEM_PK"
     *           index-qualifier="#NULL"
     *           non-unique="false"
     *           ordinal-position="1"
     *           pages="#NULL"
     *           table-cat="#NULL"
     *           table-name="ADDRESS"
     *           table-shem="#NULL" type="3"/>
     *    [... Other index nodes]
     *    &lt;column auto-increment="false"
     *            case-sensitive="true"
     *            catalog-name=""
     *            column-class-name="#NOT_SUPPORTED"
     *            column-display-size="0"
     *            column-label="ID"
     *            column-name="ID"
     *            column-no-nulls="0"
     *            column-nullable="1"
     *            column-nullable-unknown="2"
     *            column-type="4"
     *            column-type-name="INTEGER"
     *            currency="false"
     *            definitely-writable="true"
     *            nullable="1"
     *            precision="0"
     *            read-only="false"
     *            scale="0"
     *            schema-name=""
     *            searchable="true"
     *            sequence="true"
     *            signed="true"
     *            table-name="ADDRESS"
     *            writable="true"/>
     *    [... Other column nodes]
     *  &lt;/metadata&gt;
     * </pre></td></tr></table>
     * <p>
     * Any <code>null</code> value returned by a
     * {@link java.sql.ResultSetMetaData}.get*() or a
     * {@link java.sql.ResultSet}.get*(int columnIndex) is marked as
     * <code>#NULL</code>.
     * <p>
     * Any exception thrown by a {@link java.sql.ResultSetMetaData}.get*() method
     * is marked as <code>#NOT_SUPPORTED</code>.
     * <p>
     * Important database errors (table does not exists, ...) are not thrown. Instead,
     * an errors <code>NodeSet</code> is built and can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * See {@link MetaDataNodeBuilder#getPrimaryKeys(ExpressionContext,String)},
     * {@link MetaDataNodeBuilder#getIndexInfo(ExpressionContext,String)} and
     * {@link MetaDataNodeBuilder#getResultSetMetaData(ExpressionContext,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param table table name from which metadata are to be retreived.
     * @return a <code>Node</code> containing the table metadata informations.
     */
    public static Node  getTableMetaData(ExpressionContext context, String  table) {
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getTableMetaData(null, null, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Gets table metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getTableMetaData('CAT','SCH','ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link #getTableMetaData(ExpressionContext,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the catalog.
     *   <dd><b>[Mandatory/AVT]</b> name of the schema.
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param catalog the table catalog name.
     * @param schema the table schema name.
     * @param table the table name from which metadata are to be retreived.
     * @return a <code>Node</code> containing the table metadata informations.
     */
    public static Node  getTableMetaData(ExpressionContext context,
            String  catalog,
            String  schema,
            String  table) {
        catalog = XSLUtil.evaluate(context, catalog);
        schema = XSLUtil.evaluate(context, schema);
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getTableMetaData(catalog, schema, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Gets table metadata.
     * <p>
     * See {@link #getTableMetaData(ExpressionContext,String)}.
     * <p>
     * @param catalog the table catalog name.
     * @param schema the table schema name.
     * @param table the table name from which metadata are to be retreived.
     * @param doc a <code>Document</code> object used to build the
     *        <code>metadata Node</code>.
     * @return a <code>Node</code> containing the table metadata informations.
     * @throws org.apache.xml.utils.WrappedRuntimeException if something goes
     *         wrong (except database errors).
     */
    protected static Node  getTableMetaData(String  catalog,
            String  schema,
            String  table,
            Document  doc) {
        NodeSet pks = getPrimaryKeys(catalog, schema, table, doc);
        NodeSet ii = getIndexInfo(catalog, schema, table, doc);
        NodeSet cls = getResultSetMetaData(table, "*", S_RS_COLUMN_NODE_NAME,
                doc);

        if (pks == null || ii == null || cls == null) {
            return null;
        }
        try {
            Element  root = doc.createElement(S_TABLE_METADATA_NODE_NAME);

            for (int i = 0; i < pks.getLength(); i++) {
                root.appendChild(pks.item(i));
            }
            for (int i = 0; i < ii.getLength(); i++) {
                root.appendChild(ii.item(i));
            }
            for (int i = 0; i < cls.getLength(); i++) {
                root.appendChild(cls.item(i));
            }
            return root;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }
}
