
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext.db;

import org.ejen.util.DOMUtil;
import org.ejen.util.XSLUtil;
import java.sql.Statement ;
import java.sql.ResultSetMetaData ;
import java.sql.ResultSet ;
import java.sql.Connection ;
import java.sql.DatabaseMetaData ;
import java.sql.SQLException ;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.Node ;
import org.apache.xpath.NodeSet;
import org.apache.xml.utils.WrappedRuntimeException;
import org.apache.xalan.extensions.ExpressionContext;

/**
 * Database/table metadata <b>abstract</b> class.
 * (see {@link org.ejen.ext.db.BasicMetaDataConnection} for usage).
 * @author F. Wolff
 * @version 1.0
 */
public abstract class MetaDataNodeBuilder {
    // DatabaseMetaData.*
 public static final String  S_DI_NODE_NAME = "database-information";
    public static final String  S_DI_DATABASE_PRODUCT_NAME = "database-product-name";
    public static final String  S_DI_DATABASE_PRODUCT_VERSION = "database-product-version";
    // ResultSetMetaData.*
 public static final String  S_RS_COLUMN_NODE_NAME = "column";
    public static final String  S_RS_CATALOG_NAME = "catalog-name";
    public static final String  S_RS_COLUMN_CLASS_NAME = "column-class-name";
    public static final String  S_RS_COLUMN_DISPLAY_SIZE = "column-display-size";
    public static final String  S_RS_COLUMN_LABEL = "column-label";
    public static final String  S_RS_COLUMN_NAME = "column-name";
    public static final String  S_RS_COLUMN_TYPE = "column-type";
    public static final String  S_RS_COLUMN_TYPE_NAME = "column-type-name";
    public static final String  S_RS_PRECISION = "precision";
    public static final String  S_RS_SCALE = "scale";
    public static final String  S_RS_SCHEMA_NAME = "schema-name";
    public static final String  S_RS_TABLE_NAME = "table-name";
    public static final String  S_RS_AUTO_INCREMENT = "auto-increment";
    public static final String  S_RS_CASE_SENSITIVE = "case-sensitive";
    public static final String  S_RS_CURRENCY = "currency";
    public static final String  S_RS_DEFINITELY_WRITABLE = "definitely-writable";
    public static final String  S_RS_NULLABLE = "nullable";
    public static final String  S_RS_READ_ONLY = "read-only";
    public static final String  S_RS_SEARCHABLE = "searchable";
    public static final String  S_RS_SIGNED = "signed";
    public static final String  S_RS_WRITABLE = "writable";
    public static final String  S_RS_COLUMN_NO_NULLS = "column-no-nulls";
    public static final String  S_RS_COLUMN_NULLABLE = "column-nullable";
    public static final String  S_RS_COLUMN_NULLABLE_UNKNOWN = "column-nullable-unknown";
    // DatabaseMetaData.getPrimaryKeys
 public static final String  S_PK_NODE_NAME = "primary-key";
    public static final String  S_PK_TABLE_CAT = "table-cat";
    public static final String  S_PK_TABLE_SCHEM = "table-shem";
    public static final String  S_PK_TABLE_NAME = "table-name";
    public static final String  S_PK_COLUMN_NAME = "column-name";
    public static final String  S_PK_KEY_SEQ = "key-seq";
    public static final String  S_PK_PK_NAME = "pk-name";
    // DatabaseMetaData.getExportedKeys / DatabaseMetaData.getImportedKeys
 public static final String  S_EIK_EK_NODE_NAME = "exported-key";
    public static final String  S_EIK_IK_NODE_NAME = "imported-key";
    public static final String  S_EIK_PKTABLE_CAT = "pktable-cat";
    public static final String  S_EIK_PKTABLE_SCHEM = "pktable-shem";
    public static final String  S_EIK_PKTABLE_NAME = "pktable-name";
    public static final String  S_EIK_PKCOLUMN_NAME = "pkcolumn-name";
    public static final String  S_EIK_FKTABLE_CAT = "fktable-cat";
    public static final String  S_EIK_FKTABLE_SCHEM = "fktable-shem";
    public static final String  S_EIK_FKTABLE_NAME = "fktable-name";
    public static final String  S_EIK_FKCOLUMN_NAME = "fkcolumn-name";
    public static final String  S_EIK_KEY_SEQ = "key-seq";
    public static final String  S_EIK_UPDATE_RULE = "update-rule";
    public static final String  S_EIK_DELETE_RULE = "delete-rule";
    public static final String  S_EIK_FK_NAME = "fk-name";
    public static final String  S_EIK_PK_NAME = "pk-name";
    public static final String  S_EIK_DEFERRABILITY = "deferrability";
    // DatabaseMetaData.getIndexInfo
 public static final String  S_II_NODE_NAME = "index";
    public static final String  S_II_TABLE_CAT = "table-cat";
    public static final String  S_II_TABLE_SCHEM = "table-shem";
    public static final String  S_II_TABLE_NAME = "table-name";
    public static final String  S_II_NON_UNIQUE = "non-unique";
    public static final String  S_II_INDEX_QUALIFIER = "index-qualifier";
    public static final String  S_II_INDEX_NAME = "index-name";
    public static final String  S_II_TYPE = "type";
    public static final String  S_II_ORDINAL_POSITION = "ordinal-position";
    public static final String  S_II_COLUMN_NAME = "column-name";
    public static final String  S_II_ASC_OR_DESC = "asc-or-desc";
    public static final String  S_II_CARDINALITY = "cardinality";
    public static final String  S_II_PAGES = "pages";
    public static final String  S_II_FILTER_CONDITION = "filter-condition";
    // Errors
 public static final String  S_EXCEPTION_NODE_NAME = "exception";
    public static final String  S_EXCEPTION_CLASS_NAME = "class-name";
    public static final String  S_EXCEPTION_CODE = "code";
    // Default values
 public static final String  S_NOT_SUPPORTED = "#NOT_SUPPORTED";
    public static final String  S_NULL = "#NULL";

    /** <code>NodeSet</code> containing errors informations */
    protected static NodeSet _errors = null;

    /** Current active connection */
    protected static Connection  _activeConn = null;

    /**
     * Gets the current active JDBC connection.
     * @return the connection.
     * @throws java.sql.SQLException if there is no active connection
     *         (ie: no connections at all).
     */
    protected static Connection  getConnection() throws SQLException  {
        if (_activeConn == null) {
            throw new SQLException ("Not connected");
        }
        return _activeConn;
    }

    /**
     * Returns a <code>NodeSet</code> that contains errors informations.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="errors" select="mta:getErrors()"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned <code>NodeSet</code> has the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  [... other 'exception' nodes with earlier errors]
     *  &lt;exception class-name="java.sql.SQLException" code="0"&gt;
     *    Table not found: ADRESS in statement [SELECT * FROM ADRESS]
     *  &lt;/exception&gt;
     * </pre></td></tr></table>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the errors <code>NodeSet</code>.
     */
    public static NodeSet getErrors(ExpressionContext context) {
        return _errors;
    }

    /**
     * Returns a <code>Node</code> that contains basic database informations.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getDatabaseInformation()/@*"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned <code>Node</code> has the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;database-information database-product-name="..."
     *                        database-product-version="..."/&gt;
     * </pre></td></tr></table>
     * <p>
     * Database errors (SQLException) are not thrown. Instead, an errors
     * <code>NodeSet</code> is built that can be retreived by the
     * {@link #getErrors(ExpressionContext)} method.
     * <p>
     * See {@link java.sql.DatabaseMetaData#getDatabaseProductName()} and
     * {@link java.sql.DatabaseMetaData#getDatabaseProductVersion()}.
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the database informations <code>Node</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static Node  getDatabaseInformation(ExpressionContext context) {
        Document  doc = XSLUtil.getContextDocument(context);

        try {
            DatabaseMetaData  dbmd = getConnection().getMetaData();
            Element  elt = doc.createElement(S_DI_NODE_NAME);

            elt.setAttribute(S_DI_DATABASE_PRODUCT_NAME,
                    noNull(dbmd.getDatabaseProductName()));
            elt.setAttribute(S_DI_DATABASE_PRODUCT_VERSION,
                    noNull(dbmd.getDatabaseProductVersion()));
            return elt;
        } catch (SQLException  e) {
            appendErrorNode(e);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        return null;
    }

    /**
     * Returns a <code>NodeSet</code> that contains <code>ResultSet</code>
     * metadata (for all columns).
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getResultSetMetaData('ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned <code>NodeSet</code> has the following format
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;column auto-increment="false"
     *          case-sensitive="true"
     *          catalog-name=""
     *          column-class-name="#NOT_SUPPORTED"
     *          column-display-size="0"
     *          column-label="ID"
     *          column-name="ID"
     *          column-no-nulls="0"
     *          column-nullable="1"
     *          column-nullable-unknown="2"
     *          column-type="4"
     *          column-type-name="INTEGER"
     *          currency="false"
     *          definitely-writable="true"
     *          nullable="1"
     *          precision="0"
     *          read-only="false"
     *          scale="0"
     *          schema-name=""
     *          searchable="true"
     *          sequence="true"
     *          signed="true"
     *          table-name="ADDRESS"
     *          writable="true"/>
     *  [... Other column nodes]
     * </pre></td></tr></table>
     * <p>
     * Note: <code>column-no-nulls</code>, <code>column-nullable</code> and
     * <code>column-nullable-unknown</code> are just constant values used by
     * <code>nullable</code> attribute.
     * <p>
     * Any <code>null</code> value returned by a
     * {@link java.sql.ResultSetMetaData}.get*()
     * or a {@link java.sql.ResultSet}.get*(int columnIndex) is marked
     * as <code>#NULL</code>.
     * <p>
     * Any exception thrown by a {@link java.sql.ResultSetMetaData}.get*() method
     * is marked as <code>#NOT_SUPPORTED</code>.
     * <p>
     * Important database errors (table does not exists, ...) are not thrown. Instead,
     * an errors <code>NodeSet</code> is built and can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * See {@link java.sql.ResultSetMetaData}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param table name of the table.
     * @return the <code>ResultSet</code> metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getResultSetMetaData(ExpressionContext context,
            String  table) {
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getResultSetMetaData(table, "*", S_RS_COLUMN_NODE_NAME,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains <code>ResultSet</code> metadata
     * (for the given column names).
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getResultSetMetaData('ADDRESS','COL1,COL2')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link #getResultSetMetaData(ExpressionContext,String)} method.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     *   <dd><b>[Mandatory/AVT]</b> comma separated list column names.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param table name of the table.
     * @param columns comma separated list of columns.
     * @return the <code>ResultSet</code> metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getResultSetMetaData(ExpressionContext context,
            String  table,
            String  columns) {
        table = XSLUtil.evaluate(context, table);
        columns = XSLUtil.evaluate(context, columns);
        _errors = null;
        return getResultSetMetaData(table, columns, S_RS_COLUMN_NODE_NAME,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains <code>ResultSet</code> metadata
     * (for the given column names).
     * <p>
     * See {@link #getResultSetMetaData(ExpressionContext,String)} method.
     * <p>
     * @param table name of the table.
     * @param columns comma separated list of columns.
     * @param eltName name of each columns node name.
     * @param doc a <code>Document</code>.
     * @return the <code>ResultSet</code> metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    protected static NodeSet getResultSetMetaData(String  table,
            String  columns,
            String  eltName,
            Document  doc) {
        Statement  stmt = null;

        try {
            stmt = getConnection().createStatement();
            ResultSetMetaData  rsmd
                    = stmt.executeQuery("SELECT " + columns + " FROM " + table).getMetaData();
            String  columnNoNulls = String.valueOf(rsmd.columnNoNulls);
            String  columnNullable = String.valueOf(rsmd.columnNullable);
            String  columnNullableUnknown = String.valueOf(rsmd.columnNullableUnknown);
            int count = rsmd.getColumnCount();
            NodeSet ns = new NodeSet();

            ns.setShouldCacheNodes(true);
            for (int i = 1; i <= count; i++) {
                Element  elt = doc.createElement(eltName);

                try {
                    elt.setAttribute(S_RS_CATALOG_NAME,
                            noNull(rsmd.getCatalogName(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_CATALOG_NAME, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_COLUMN_CLASS_NAME,
                            noNull(rsmd.getColumnClassName(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_COLUMN_CLASS_NAME, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_COLUMN_DISPLAY_SIZE,
                            String.valueOf(rsmd.getColumnDisplaySize(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_COLUMN_DISPLAY_SIZE, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_COLUMN_LABEL,
                            noNull(rsmd.getColumnLabel(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_COLUMN_LABEL, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_COLUMN_NAME,
                            noNull(rsmd.getColumnName(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_COLUMN_NAME, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_COLUMN_TYPE,
                            String.valueOf(rsmd.getColumnType(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_COLUMN_TYPE, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_COLUMN_TYPE_NAME,
                            noNull(rsmd.getColumnTypeName(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_COLUMN_TYPE_NAME, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_PRECISION,
                            String.valueOf(rsmd.getPrecision(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_PRECISION, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_SCALE,
                            String.valueOf(rsmd.getScale(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_SCALE, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_SCHEMA_NAME,
                            noNull(rsmd.getSchemaName(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_SCHEMA_NAME, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_TABLE_NAME,
                            noNull(rsmd.getTableName(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_TABLE_NAME, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_AUTO_INCREMENT,
                            String.valueOf(rsmd.isAutoIncrement(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_AUTO_INCREMENT, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_CASE_SENSITIVE,
                            String.valueOf(rsmd.isCaseSensitive(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_CASE_SENSITIVE, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_CURRENCY,
                            String.valueOf(rsmd.isCurrency(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_CURRENCY, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_DEFINITELY_WRITABLE,
                            String.valueOf(rsmd.isDefinitelyWritable(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_DEFINITELY_WRITABLE, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_NULLABLE,
                            String.valueOf(rsmd.isNullable(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_NULLABLE, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_READ_ONLY,
                            String.valueOf(rsmd.isReadOnly(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_READ_ONLY, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_SEARCHABLE,
                            String.valueOf(rsmd.isSearchable(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_SEARCHABLE, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_SIGNED,
                            String.valueOf(rsmd.isSigned(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_SIGNED, S_NOT_SUPPORTED);
                }
                try {
                    elt.setAttribute(S_RS_WRITABLE,
                            String.valueOf(rsmd.isWritable(i)));
                } catch (SQLException  e) {
                    elt.setAttribute(S_RS_WRITABLE, S_NOT_SUPPORTED);
                }
                elt.setAttribute(S_RS_COLUMN_NO_NULLS, columnNoNulls);
                elt.setAttribute(S_RS_COLUMN_NULLABLE, columnNullable);
                elt.setAttribute(S_RS_COLUMN_NULLABLE_UNKNOWN,
                        columnNullableUnknown);
                ns.addElement(elt);
            }
            return ns;
        } catch (SQLException  e) {
            appendErrorNode(e);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        finally {
            if (stmt != null) {
                try {
                    stmt.close();
                } catch (SQLException  e) {
                    appendErrorNode(e);
                }
            }
        }
        return null;
    }

    /**
     * Returns a <code>NodeSet</code> that contains primary keys metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getPrimaryKeys('ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned <code>NodeSet</code> has the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;primary-key column-name="ID"
     *               key-seq="1"
     *               pk-name="SYSTEM_PK"
     *               table-cat="#NULL"
     *               table-name="ADDRESS"
     *               table-shem="#NULL"/>
     * </pre></td></tr></table>
     * <p>
     * Any <code>null</code> value returned by a
     * {@link java.sql.ResultSetMetaData}.get*()
     * or a {@link java.sql.ResultSet}.get*(int columnIndex) is marked
     * as <code>#NULL</code>.
     * <p>
     * Any exception thrown by a {@link java.sql.ResultSetMetaData}.get*() method
     * is marked as <code>#NOT_SUPPORTED</code>.
     * <p>
     * Important database errors (table does not exists, ...) are not thrown.
     * Instead, an errors <code>NodeSet</code> is built and can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * See {@link java.sql.DatabaseMetaData#getPrimaryKeys(String,String,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param table name of the table.
     * @return the primary keys metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getPrimaryKeys(ExpressionContext context, String  table) {
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getPrimaryKeys(null, null, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains primary keys metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getPrimaryKeys('CAT','SCH','ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link #getPrimaryKeys(ExpressionContext,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the catalog.
     *   <dd><b>[Mandatory/AVT]</b> name of the schema.
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param catalog name of the catalog.
     * @param schema name of the schema.
     * @param table name of the table.
     * @return the primary keys metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getPrimaryKeys(ExpressionContext context,
            String  catalog,
            String  schema,
            String  table) {
        catalog = XSLUtil.evaluate(context, catalog);
        schema = XSLUtil.evaluate(context, schema);
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getPrimaryKeys(catalog, schema, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains primary keys metadata.
     * <p>
     * See {@link #getPrimaryKeys(ExpressionContext,String)}.
     * <p>
     * @param catalog name of the catalog.
     * @param schema name of the schema.
     * @param table name of the table.
     * @param doc a <code>Document</code>.
     * @return the <code>ResultSet</code> metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    protected static NodeSet getPrimaryKeys(String  catalog,
            String  schema,
            String  table,
            Document  doc) {
        ResultSet  rs = null;

        try {
            rs = getConnection().getMetaData().getPrimaryKeys(catalog, schema,
                    table);
            NodeSet ns = new NodeSet();

            ns.setShouldCacheNodes(true);
            while (rs.next()) {
                Element  elt = doc.createElement(S_PK_NODE_NAME);

                elt.setAttribute(S_PK_TABLE_CAT, noNull(rs.getString(1)));
                elt.setAttribute(S_PK_TABLE_SCHEM, noNull(rs.getString(2)));
                elt.setAttribute(S_PK_TABLE_NAME, noNull(rs.getString(3)));
                elt.setAttribute(S_PK_COLUMN_NAME, noNull(rs.getString(4)));
                elt.setAttribute(S_PK_KEY_SEQ, String.valueOf(rs.getShort(5)));
                elt.setAttribute(S_PK_PK_NAME, noNull(rs.getString(6)));
                ns.addElement(elt);
            }
            return ns;
        } catch (SQLException  e) {
            appendErrorNode(e);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException  e) {
                    appendErrorNode(e);
                }
            }
        }
        return null;
    }

    /**
     * Returns a <code>NodeSet</code> that contains imported keys metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getImportedKeys('ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned <code>NodeSet</code> has the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;imported-key pktable-cat="..."
     *                pktable-shem="..."
     *                pktable-name="..."
     *                pkcolumn-name="..."
     *                fktable-cat="..."
     *                fktable-shem="..."
     *                fktable-name="..."
     *                fkcolumn-name="..."
     *                key-seq="..."
     *                update-rule="..."
     *                delete-rule="..."
     *                fk-name="..."
     *                pk-name="..."
     *                deferrability="..."/>
     *  [... Other imported-key nodes]
     * </pre></td></tr></table>
     * <p>
     * Any <code>null</code> value is marked as <code>#NULL</code>.
     * <p>
     * Important database errors (table does not exists, ...) are not thrown.
     * Instead, an errors <code>NodeSet</code> is built and can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * See {@link java.sql.DatabaseMetaData#getImportedKeys(String,String,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param table name of the table.
     * @return the primary keys metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getImportedKeys(ExpressionContext context, String  table) {
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getImportedKeys(null, null, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains imported keys metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getImportedKeys('CAT','SCH','ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link #getImportedKeys(ExpressionContext,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the catalog.
     *   <dd><b>[Mandatory/AVT]</b> name of the schema.
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param catalog name of the catalog.
     * @param schema name of the schema.
     * @param table name of the table.
     * @return the primary keys metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getImportedKeys(ExpressionContext context,
            String  catalog,
            String  schema,
            String  table) {
        catalog = XSLUtil.evaluate(context, catalog);
        schema = XSLUtil.evaluate(context, schema);
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getImportedKeys(catalog, schema, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains imported keys metadata.
     * <p>
     * See {@link #getImportedKeys(ExpressionContext,String)}.
     * <p>
     * @param catalog name of the catalog.
     * @param schema name of the schema.
     * @param table name of the table.
     * @param doc a <code>Document</code>.
     * @return the <code>ResultSet</code> metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    protected static NodeSet getImportedKeys(String  catalog,
            String  schema,
            String  table,
            Document  doc) {
        ResultSet  rs = null;

        try {
            rs = getConnection().getMetaData().getImportedKeys(catalog, schema,
                    table);
            NodeSet ns = new NodeSet();

            ns.setShouldCacheNodes(true);
            getImportedOrExportedKeys(rs, S_EIK_IK_NODE_NAME, ns, doc);
            return ns;
        } catch (SQLException  e) {
            appendErrorNode(e);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException  e) {
                    appendErrorNode(e);
                }
            }
        }
        return null;
    }

    /**
     * Returns a <code>NodeSet</code> that contains exported keys metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getExportedKeys('ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned <code>NodeSet</code> has the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;exported-key pktable-cat="..."
     *                pktable-shem="..."
     *                pktable-name="..."
     *                pkcolumn-name="..."
     *                fktable-cat="..."
     *                fktable-shem="..."
     *                fktable-name="..."
     *                fkcolumn-name="..."
     *                key-seq="..."
     *                update-rule="..."
     *                delete-rule="..."
     *                fk-name="..."
     *                pk-name="..."
     *                deferrability="..."/>
     *  [... Other exported-key nodes]
     * </pre></td></tr></table>
     * <p>
     * Any <code>null</code> value is marked as <code>#NULL</code>.
     * <p>
     * Important database errors (table does not exists, ...) are not thrown.
     * Instead, an errors <code>NodeSet</code> is built and can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * See {@link java.sql.DatabaseMetaData#getExportedKeys(String,String,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param table name of the table.
     * @return the primary keys metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getExportedKeys(ExpressionContext context, String  table) {
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getExportedKeys(null, null, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains exported keys metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getExportedKeys('CAT','SCH','ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link #getExportedKeys(ExpressionContext,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the catalog.
     *   <dd><b>[Mandatory/AVT]</b> name of the schema.
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param catalog name of the catalog.
     * @param schema name of the schema.
     * @param table name of the table.
     * @return the primary keys metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getExportedKeys(ExpressionContext context,
            String  catalog,
            String  schema,
            String  table) {
        catalog = XSLUtil.evaluate(context, catalog);
        schema = XSLUtil.evaluate(context, schema);
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getExportedKeys(catalog, schema, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains exported keys metadata.
     * <p>
     * See {@link #getExportedKeys(ExpressionContext,String)}.
     * <p>
     * @param catalog name of the catalog.
     * @param schema name of the schema.
     * @param table name of the table.
     * @param doc a <code>Document</code>.
     * @return the <code>ResultSet</code> metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    protected static NodeSet getExportedKeys(String  catalog,
            String  schema,
            String  table,
            Document  doc) {
        ResultSet  rs = null;

        try {
            rs = getConnection().getMetaData().getExportedKeys(catalog, schema,
                    table);
            NodeSet ns = new NodeSet();

            ns.setShouldCacheNodes(true);
            getImportedOrExportedKeys(rs, S_EIK_EK_NODE_NAME, ns, doc);
            return ns;
        } catch (SQLException  e) {
            appendErrorNode(e);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException  e) {
                    appendErrorNode(e);
                }
            }
        }
        return null;
    }

    /**
     * Returns a <code>NodeSet</code> that contains exported or imported
     * keys metadata.
     * <p>
     * See {@link #getExportedKeys(ExpressionContext,String)} and
     * {@link #getImportedKeys(ExpressionContext,String)}.
     * <p>
     * @param rs a ResutSet with imported or exported keys metadata.
     * @param eltName name of the Nodes to be built.
     * @param ns NodeSet to fill with Nodes.
     * @param doc a <code>Document</code>.
     * @throws java.lang.Exception any...
     */
    protected static void getImportedOrExportedKeys(ResultSet  rs,
            String  eltName,
            NodeSet ns,
            Document  doc)
        throws Exception  {
        while (rs.next()) {
            Element  elt = doc.createElement(eltName);

            elt.setAttribute(S_EIK_PKTABLE_CAT, noNull(rs.getString(1)));
            elt.setAttribute(S_EIK_PKTABLE_SCHEM, noNull(rs.getString(2)));
            elt.setAttribute(S_EIK_PKTABLE_NAME, noNull(rs.getString(3)));
            elt.setAttribute(S_EIK_PKCOLUMN_NAME, noNull(rs.getString(4)));
            elt.setAttribute(S_EIK_FKTABLE_CAT, noNull(rs.getString(5)));
            elt.setAttribute(S_EIK_FKTABLE_SCHEM, noNull(rs.getString(6)));
            elt.setAttribute(S_EIK_FKTABLE_NAME, noNull(rs.getString(7)));
            elt.setAttribute(S_EIK_FKCOLUMN_NAME, noNull(rs.getString(8)));
            elt.setAttribute(S_EIK_KEY_SEQ, String.valueOf(rs.getShort(9)));
            elt.setAttribute(S_EIK_UPDATE_RULE, String.valueOf(rs.getShort(10)));
            elt.setAttribute(S_EIK_DELETE_RULE, String.valueOf(rs.getShort(11)));
            elt.setAttribute(S_EIK_FK_NAME, noNull(rs.getString(12)));
            elt.setAttribute(S_EIK_PK_NAME, noNull(rs.getString(13)));
            elt.setAttribute(S_EIK_DEFERRABILITY,
                    String.valueOf(rs.getShort(14)));
            ns.addElement(elt);
        }
    }

    /**
     * Returns a <code>NodeSet</code> that contains indexes metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getIndexInfo('ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned <code>NodeSet</code> has the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;index asc-or-desc="A"
     *         cardinality="#NULL"
     *         column-name="ID"
     *         filter-condition="#NULL"
     *         index-name="SYSTEM_PK"
     *         index-qualifier="#NULL"
     *         non-unique="false"
     *         ordinal-position="1"
     *         pages="#NULL"
     *         table-cat="#NULL"
     *         table-name="ADDRESS"
     *         table-shem="#NULL" type="3"/>
     *  [... Other index nodes]
     * </pre></td></tr></table>
     * <p>
     * Any <code>null</code> value is marked as <code>#NULL</code>.
     * <p>
     * Important database errors (table does not exists, ...) are not thrown.
     * Instead, an errors <code>NodeSet</code> is built and can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * See {@link java.sql.DatabaseMetaData#getIndexInfo(String,String,String,boolean,boolean)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param table name of the table.
     * @return the primary keys metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getIndexInfo(ExpressionContext context, String  table) {
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getIndexInfo(null, null, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains indexes metadata.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="mta:getIndexInfo('CAT','SCH','ADDRESS')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link #getIndexInfo(ExpressionContext,String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the catalog.
     *   <dd><b>[Mandatory/AVT]</b> name of the schema.
     *   <dd><b>[Mandatory/AVT]</b> name of the table.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param catalog name of the catalog.
     * @param schema name of the schema.
     * @param table name of the table.
     * @return the primary keys metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet getIndexInfo(ExpressionContext context,
            String  catalog,
            String  schema,
            String  table) {
        catalog = XSLUtil.evaluate(context, catalog);
        schema = XSLUtil.evaluate(context, schema);
        table = XSLUtil.evaluate(context, table);
        _errors = null;
        return getIndexInfo(catalog, schema, table,
                XSLUtil.getContextDocument(context));
    }

    /**
     * Returns a <code>NodeSet</code> that contains indexes metadata.
     * <p>
     * See {@link #getIndexInfo(ExpressionContext,String)}.
     * <p>
     * @param catalog name of the catalog.
     * @param schema name of the schema.
     * @param table name of the table.
     * @param doc a <code>Document</code>.
     * @return the <code>ResultSet</code> metadata <code>NodeSet</code>.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    protected static NodeSet getIndexInfo(String  catalog,
            String  schema,
            String  table,
            Document  doc) {
        ResultSet  rs = null;

        try {
            rs = getConnection().getMetaData().getIndexInfo(catalog, schema,
                    table, false, false);
            NodeSet ns = new NodeSet();

            ns.setShouldCacheNodes(true);
            while (rs.next()) {
                Element  elt = doc.createElement(S_II_NODE_NAME);

                elt.setAttribute(S_II_TABLE_CAT, noNull(rs.getString(1)));
                elt.setAttribute(S_II_TABLE_SCHEM, noNull(rs.getString(2)));
                elt.setAttribute(S_II_TABLE_NAME, noNull(rs.getString(3)));
                elt.setAttribute(S_II_NON_UNIQUE,
                        String.valueOf(rs.getBoolean(4)));
                elt.setAttribute(S_II_INDEX_QUALIFIER, noNull(rs.getString(5)));
                elt.setAttribute(S_II_INDEX_NAME, noNull(rs.getString(6)));
                elt.setAttribute(S_II_TYPE, String.valueOf(rs.getShort(7)));
                elt.setAttribute(S_II_ORDINAL_POSITION,
                        String.valueOf(rs.getShort(8)));
                elt.setAttribute(S_II_COLUMN_NAME, noNull(rs.getString(9)));
                elt.setAttribute(S_II_ASC_OR_DESC, noNull(rs.getString(10)));
                elt.setAttribute(S_II_CARDINALITY, noNull(rs.getString(11)));
                elt.setAttribute(S_II_PAGES, noNull(rs.getString(12)));
                elt.setAttribute(S_II_FILTER_CONDITION, noNull(rs.getString(13)));
                ns.addElement(elt);
            }
            return ns;
        } catch (SQLException  e) {
            appendErrorNode(e);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException  e) {
                    appendErrorNode(e);
                }
            }
        }
        return null;
    }

    /**
     * Appends an error Node to the current _errors NodeSet.
     * <p>
     * @param e the exception that must be add as an error Node.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    protected static void appendErrorNode(Exception  e) {
        if (e == null) {
            return;
        }
        int errCode = 0;

        if (e instanceof SQLException ) {
            errCode = ((SQLException ) e).getErrorCode();
        }
        Document  doc = null;

        if (_errors == null) {
            _errors = new NodeSet();
            _errors.setShouldCacheNodes(true);
            doc = DOMUtil.newDocument();
        } else {
            doc = _errors.getRoot() != null
                    ? _errors.getRoot().getOwnerDocument()
                    : DOMUtil.newDocument();
        }
        try {
            Element  elt = doc.createElement(S_EXCEPTION_NODE_NAME);

            elt.setAttribute(S_EXCEPTION_CLASS_NAME, e.getClass().getName());
            elt.setAttribute(S_EXCEPTION_CODE, String.valueOf(errCode));
            elt.appendChild(doc.createTextNode(e.getMessage()));
            _errors.addElement(elt);
        } catch (Exception  f) {
            throw new WrappedRuntimeException(f);
        }
    }

    /**
     * Returns s if s is not <code>null</code>, "#NULL" otherwise.
     * <p>
     * @param s the String to test for nullity.
     * @return the s if s is not <code>null</code>, "#NULL" otherwise.
     */
    protected static String  noNull(String  s) {
        return (s == null) ? S_NULL : s;
    }
}
