
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.w3c.dom.Node ;
import org.w3c.dom.traversal.NodeIterator;
import org.apache.xalan.extensions.XSLProcessorContext;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xalan.templates.ElemExtensionCall;

/**
 * XSL operations utility (static methods).
 * <p>
 * This class only wrappes some methods implemented in the
 * {@link org.ejen.util.XSLUtil} class.
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:xsu="org.ejen.ext.XSLUtil"
 *                  extension-element-prefixes="xsu ..."
 *                  exclude-result-prefixes="xsu ..."</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;xsu:{@link #evaluate(XSLProcessorContext,ElemExtensionCall) evaluate} avt="{./file}.xml"/&gt;
 *      &lt;xsl:variable name="avt" select="xsu:{@link #evaluate(ExpressionContext,String) evaluate}('{./file}.xml')"/&gt;
 *      &lt;xsl:if test="xsu:{@link #equals(ExpressionContext,NodeIterator,NodeIterator) equals}($nodes1,$nodes2)"&gt;
 *        ...
 *      &lt;/xsl:if&gt;
 *      &lt;xsl:if test="xsu:{@link #equals(ExpressionContext,Node,Node) equals}($node1,$node2)"&gt;
 *        ...
 *      &lt;/xsl:if&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class XSLUtil {

    /**
     * Prevents instanciation.
     */
    protected XSLUtil() {}

    /**
     * Returns an interpreted value (AVT) of a Node attribute whose name is "avt".
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsu:evaluate avt="{./file}.xml"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>avt <b>[Mandatory/AVT]</b> AVT to be evaluated.
     * </dl></dd>
     * <p>
     * @param context the XSLProcessorContext to be used.
     * @param elem the ElemExtensionCall to be used.
     * @return the evaluated result.
     * @throws java.lang.IllegalArgumentException bad context.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static String  evaluate(XSLProcessorContext context, ElemExtensionCall elem) {
        return org.ejen.util.XSLUtil.evaluate(context, elem);
    }

    /**
     * Returns an interpreted value (AVT) of a Node attribute whose name is equals
     * to the avt parameter.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="avt" select="xsu:evaluate('{./file}.xml')"&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> AVT expression to be evaluated.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param avt the name of the attribute.
     * @return the evaluated result.
     * @throws java.lang.IllegalArgumentException bad context.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static String  evaluate(ExpressionContext context, String  avt) {
        return org.ejen.util.XSLUtil.evaluate(context, avt);
    }

    /**
     * Checks ni1 and ni2 for strict equality (same names, same attributes,
     * same child nodes...).
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:if test="xsu:equals($nodes1,$nodes2)"&gt;
     *    ...
     *  &lt;/xsl:if&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory]</b> the first NodeIterator.
     *   <dd><b>[Mandatory]</b> the second NodeIterator.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param ni1 the first NodeIterator.
     * @param ni2 the second NodeIterator.
     * @return true if ni1 equals ni2, false otherwise.
     */
    public static boolean equals(ExpressionContext context, NodeIterator ni1, NodeIterator ni2) {
        return org.ejen.util.XSLUtil.equals(ni1, ni2);
    }

    /**
     * Checks n1 and n2 for strict equality (same names, same attributes,
     * same child nodes...).
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:if test="xsu:equals($node1,$node2)"&gt;
     *    ...
     *  &lt;/xsl:if&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory]</b> the first Node.
     *   <dd><b>[Mandatory]</b> the second Node.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param n1 the first Node.
     * @param n2 the second Node.
     * @return true if n1 equals n2, false otherwise.
     */
    public static boolean equals(ExpressionContext context, Node  n1, Node  n2) {
        return org.ejen.util.XSLUtil.equals(n1, n2);
    }
}
