
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.ejen.util.XSLUtil;
import org.ejen.EjenChildNode;
import java.sql.SQLException ;
import org.apache.xalan.extensions.XSLProcessorContext;
import org.apache.xalan.templates.ElemExtensionCall;
import org.apache.xml.utils.WrappedRuntimeException;

/**
 * Messages utility (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:msg="org.ejen.ext.Messenger"
 *                  extension-element-prefixes="msg ..."
 *                  exclude-result-prefixes="msg ..."</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="text" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;msg:{@link #send(XSLProcessorContext, ElemExtensionCall) send} message="Oop!"/&gt;
 *      &lt;msg:{@link #throwSQLException(XSLProcessorContext, ElemExtensionCall) throwSQLException} message="Table not found" code="0"/&gt;
 *      &lt;msg:{@link #throwRuntimeException(XSLProcessorContext, ElemExtensionCall) throwRuntimeException} message="Bug!"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class Messenger {

    /**
     * Protected constructor (prevents instanciation).
     */
    protected Messenger() {}

    /**
     * Sends a message to the current {@link org.ejen.EjenListener}.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;msg:send message="Hello {people[$i]/@name} !"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>message <b>[Mandatory/AVT]</b> - message to be sent.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws org.apache.xml.utils.WrappedRuntimeException with a XSL Exception
     *         if something goes wrong.
     */
    public static void send(XSLProcessorContext context, ElemExtensionCall elem) {
        EjenChildNode.sendXSLMessageEvent(XSLUtil.getAttribute(context, elem,
                "message", true));
    }

    /**
     * Throws a {@link java.sql.SQLException}.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;msg:throwSQLException message="Table not found" code="2"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>message <b>[Mandatory/AVT]</b> - SQLException message.
     *   <dd>code <b>[Mandatory/AVT]</b> - SQLException code.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws java.sql.SQLException with supplied message/code.
     * @throws org.apache.xml.utils.WrappedRuntimeException
     *         if one of 'message' or 'code' attributes is missing, or if 'code'
     *         cannot be parsed as an integer.
     */
    public static void throwSQLException(XSLProcessorContext context, ElemExtensionCall elem)
        throws SQLException  {
        String  message = XSLUtil.getAttribute(context, elem, "message", true);
        String  code = XSLUtil.getAttribute(context, elem, "code", true);
        int iCode;

        try {
            iCode = Integer.parseInt(code);
        } catch (NumberFormatException  e) {
            throw new WrappedRuntimeException("Invalid 'code' attribute in 'throwSQLException' node: "
                    + code + " (message: " + message + ")",
                    e);
        }
        throw new SQLException (message, null, iCode);
    }

    /**
     * Throws a {@link java.lang.RuntimeException}.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;msg:throwRuntimeException message="Didn't want to work anymore"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>message <b>[Mandatory/AVT]</b> - RuntimeException message.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws java.lang.RuntimeException with supplied message.
     * @throws org.apache.xml.utils.WrappedRuntimeException
     *         if the 'message' attribute is missing.
     */
    public static void throwRuntimeException(XSLProcessorContext context, ElemExtensionCall elem) {
        context.getTransformer().stopTransformation();
        throw new RuntimeException (XSLUtil.getAttribute(context, elem, "message",
                true));
    }
}
