
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import org.ejen.util.XSLUtil;
import java.util.Properties ;
import java.io.File ;
import java.io.OutputStream ;
import java.io.FileOutputStream ;
import java.io.StringWriter ;
import javax.xml.transform.stream.StreamResult ;
import javax.xml.transform.dom.DOMSource ;
import org.w3c.dom.NodeList ;
import org.apache.xpath.XPathAPI;
import org.apache.xalan.transformer.TransformerImpl;
import org.apache.xalan.processor.StylesheetHandler;

/**
 * Template node class.
 * <p>
 * A template node uses the current in memory DOM tree to create resulting
 * text files (XML, HTML, source code, ...).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (ant build file)</th></tr>
 * <tr><td class="usage"><pre><code>
 *  &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *
 *  &lt;project name="generate" default="build"&gt;
 *
 *    &lt;taskdef name="ejen" classname="org.ejen.EjenTask"/&gt;
 *
 *    &lt;target name="build"&gt;
 *      &lt;{@link org.ejen.EjenTask ejen} ...&gt;
 *        ...
 *        <b>&lt;template {@link org.ejen.EjenStylesheetNode#setFile(String) file}="filter.xml"
 *                 [{@link #setForeach(String) foreach}="/ejen/entity-bean"]
 *                 [{@link #setFilepattern(String) filepattern}="EJB_{&#064;table-name}Bean.java"]&gt;</b>
 *          ...
 *          [&lt;{@link org.ejen.EjenIncludeNode include} .../&gt;]
 *          [&lt;{@link org.ejen.EjenImportNode import} .../&gt;]
 *          [&lt;{@link org.ejen.EjenParamNode param} .../&gt;]
 *          ...
 *        <b>&lt;/template&gt;</b>
 *        ...
 *      &lt;/ejen&gt;
 *    &lt;/target&gt;
 *
 *  &lt;/project&gt;
 * </code></pre></td></tr></table>
 * <p>
 * <b>Parent nodes</b>:
 * <ul>
 *   <li>{@link org.ejen.EjenTask ejen}
 * </ul>
 * @author F. Wolff
 * @version 1.0
 */
public class EjenTemplateNode extends EjenStylesheetNode {
    protected String  _foreach = null;
    protected String  _filepattern = null;
    protected String  _outdated = null;

    /**
     * Returns the name of this EjenTemplateNode (always "template").
     * @return the name of this EjenTemplateNode.
     */
    public String  nodeName() {
        return "template";
    }

    /**
     * Returns all non null attributes of this EjenTemplateNode.
     * @return non null attributes of this EjenTemplateNode.
     */
    public Properties  getAttributes() {
        Properties  attrs = super.getAttributes();

        if (_foreach != null) {
            attrs.setProperty("foreach", _foreach);
        }
        if (_filepattern != null) {
            attrs.setProperty("filepattern", _filepattern);
        }
        if (_outdated != null) {
            attrs.setProperty("outdated", _outdated);
        }
        return attrs;
    }

    /**
     * <b>[optional/AVT]</b> - sets the foreach attribute. This attribute
     * allows iterative applications of this template stylesheet to a sub-nodes
     * set of the current in memory DOM tree.
     * <p>
     * Suppose you have the following DOM tree in memory:
     * <table class="usage">
     * <tr><td class="usage"><pre><code>
     *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
     *  &lt;ejen&gt;
     *    &lt;name&gt;Name1&lt;/name&gt;
     *    &lt;name&gt;Name2&lt;/name&gt;
     *    &lt;name&gt;Name3&lt;/name&gt;
     *    ...
     *  &lt;/ejen&gt;
     * </code></pre></td></tr></table>
     * You want to generate text files with the following structure:
     * <table class="usage">
     * <tr><td class="usage"><pre><code>
     *  Dear &lt;Name...&gt;,
     *    Obviously, you are reading a meaningless letter.
     *  Best regards.
     * </code></pre></td></tr></table>
     * You can use this template stylesheet (with the foreach attribute set to "/ejen/name"):
     * <table class="usage">
     * <tr><td class="usage"><pre><code>
     *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
     *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
     *                  version="1.0"&gt;
     *    &lt;xsl:output method="text" encoding="iso-8859-1"/&gt;
     *    <b>&lt;xsl:template match="name"&gt;&lt;!--
     *   --&gt;Dear &lt;xsl:value-of select="."/&gt;,
     *    Obviously, you are reading a meaningless letter.
     *  Best regards.&lt;!--
     *   --&gt;&lt;/xsl:template&gt;</b>
     *  &lt;/xsl:stylesheet&gt;
     * </code></pre></td></tr></table>
     * Note that comments in this template are only for carriage return control
     * purposes (we want neither a carriage return before "Dear", nor after "regards.").
     * <p>
     * If this attribute is used, a parameter whose name is "root" and value is the root
     * node of the current DOM tree is always and automaticaly passed to the template
     * stylesheet (you may have the line "<code>&lt;xsl:param name"root"/&gt;</code>" in
     * the stylesheet).
     * <p>
     * See the {@link #setFilepattern(String) filepattern} for output file naming.
     * <p>
     * @param foreach foreach String (default is null, meaning "apply this template
     *        stylesheet to the entire current DOM tree").
     */
    public void setForeach(String  foreach) {
        _foreach = foreach;
    }

    /**
     * <b>[optional/AVT]</b> - sets the filepattern attribute. This attribute
     * allows basic file output naming setting and is relative to the foreach
     * attribute (if foreach attribute is not used, filepattern will be relative
     * to the root node of the current in memory DOM tree).
     * <p>
     * Suppose what is supposed {@link #setForeach(String) here}. You want to
     * generate text files whose names are: Name1.txt, Name2.txt,
     * Name3.txt... Then, you can use this filepattern: "{.}.txt".
     * <p>
     * If this basic file output naming mechanism is not sufficient, you may still
     * use the Xalan
     * <a HREF="http://xml.apache.org/xalan-j/apidocs/org/apache/xalan/lib/Redirect.html">Redirect</a>
     * extension.
     * <p>
     * @param filepattern filepattern String (default is null, meaning "let the
     *        template do whatever it wants").
     */
    public void setFilepattern(String  filepattern) {
        _filepattern = filepattern;
    }

    /**
     * <b>[optional/AVT]</b> - sets the outdate attribute.
     * @param outdate if that attribute is "true" output file will generate only
     *        if source file is newer then output file, otherwise output file generate always 
     */
    public void setOutdated(String  outdate) {
        _outdated = outdate;
    }
    
    /**
     * Executes this EjenTemplateNode.
     * @throws org.ejen.EjenException if something goes wrong...
     */
    public void process() {
        super.process();
        TransformerImpl ti = null;
        DOMSource  src = null;

        try {
            ti = (TransformerImpl) (getFromContext(CTX_TRANSFORMER_IMPL));
            src = (DOMSource ) (getFromGlobalContext(CTX_DOM_SOURCE));
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
        if (ti == null) {
            throw new EjenException(this,
                    "no '" + CTX_TRANSFORMER_IMPL + "' in context");
        }
        if (src == null) {
            throw new EjenException(this,
                    "no '" + CTX_DOM_SOURCE + "' in global context");
        }
        if (_foreach != null) {
            NodeList  nl = null;

            try {
                ti.setParameter("root", src.getNode());
                nl = XPathAPI.selectNodeList(src.getNode(),
                        evaluateAVT(ti, _foreach));
            } catch (Exception  e) {
                throw new EjenException(this,
                        "invalid 'foreach' attribute: " + _foreach, e);
            }
            for (int i = 0; i < nl.getLength(); i++) {
                DOMSource  nodeSrc = null;

                try {
                    nodeSrc = new DOMSource (nl.item(i));
                } catch (Exception  e) {
                    throw new EjenException(this,
                            "invalid 'foreach' attribute: " + _foreach, e);
                }
                processTemplateStreamResult(ti, nodeSrc);
            }
        } else {
            processTemplateStreamResult(ti, src);
        }
    }

    /**
     * Creates the output file, based on the filepattern attribute.
     * @param ti the TransformerImpl to use for the transformation.
     * @param src the DOMSource to be transformed.
     * @throws org.ejen.EjenException if something goes wrong...
     */
    private void processTemplateStreamResult(TransformerImpl ti, DOMSource  src) {
        StylesheetHandler sh = null;

        try {
            sh = (StylesheetHandler) (getFromGlobalContext(CTX_STYLESHEET_HANDLER));
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
        if (sh == null) {
            throw new EjenException(this,
                    "no '" + CTX_STYLESHEET_HANDLER + "' in context");
        }
        OutputStream  outputs = null;

        try {
            StreamResult  sres = null;

            if (_filepattern != null) {
                // Cannot use evaluateAVT(ti, _filepattern) because _filepattern is
 // an AVT relative to _foreach...
 String  fileName = XSLUtil.evaluateAttribute(sh,
                        ti.getXPathContext(), src.getNode(), _filepattern);
                boolean flag = false;

                if (_outdated != null && _outdated.equals("true")) {
                    flag = true;
                }
                if (!flag || outdated(fileName)) {
                    sendMessageEvent("Creating '" + fileName + "'");
                    File  f = new File (fileName);
                    File  pf = f.getParentFile();

                    if (pf != null) {
                        pf.mkdirs();
                    }
                    outputs = new FileOutputStream (f.getPath());
                    sres = new StreamResult (outputs);
                } else {
                    return;
                }
            } else {
                sres = new StreamResult (new StringWriter ());
            }
            ti.transform(src, sres);
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
        finally {
            if (outputs != null) {
                try {
                    outputs.close();
                } catch (Exception  e) {}
                finally {
                    outputs = null;
                }
            }
        }
    }
    
    /**
     * Check if source file is newer than output file.
     * @param fileName file name of output file.
     */
    protected boolean outdated(String  fileName) {
        File  srcFile = new File ((String ) getFromGlobalContext("DOM_SOURCE_FILE"));
        File  genFile = new File (fileName);

        if (!genFile.exists() || !srcFile.exists()) { // files not exist
 return true;
        }
        long slm = srcFile.lastModified(); 

        if (slm == 0L) {  // source file not exist or I/O error
 return true;
        }
        long glm = genFile.lastModified();

        if (glm == 0L) { // generating file not exist
 return true;
        }
        return slm >= glm;  // source file is newer than generating file
    }
}
