
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import java.util.Properties ;
import org.ejen.ext.Version;
import org.ejen.util.DOMUtil;
import javax.xml.transform.dom.DOMSource ;
import org.w3c.dom.Document ;

/**
 * Source node class (child of an EjenRootNode).
 * <p>
 * A Source node always replace an existing in memory DOM tree, even
 * if it is the first sub-node of an "ejen" node (see
 * {@link org.ejen.EjenRootNode#beforeProcess()}).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (ant build file)</th></tr>
 * <tr><td class="usage"><pre><code>
 *  &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *
 *  &lt;project name="generate" default="build"&gt;
 *
 *    &lt;taskdef name="ejen" classname="org.ejen.EjenTask"/&gt;
 *
 *    &lt;target name="build"&gt;
 *      &lt;{@link org.ejen.EjenTask ejen} ...&gt;
 *        ...
 *        <b>&lt;source [{@link #setFile(String) file}="source.xml"]/&gt;</b>
 *        ...
 *      &lt;/ejen&gt;
 *    &lt;/target&gt;
 *
 *  &lt;/project&gt;
 * </code></pre></td></tr></table>
 * <p>
 * <b>Parent nodes</b>:
 * <ul>
 *   <li>{@link org.ejen.EjenTask ejen}
 * </ul>
 * @author F. Wolff
 * @version 1.0
 */
public class EjenSourceNode extends EjenChildNode {
    protected String  _file = null;

    /**
     * Returns the name of this EjenSourceNode (always "source").
     * @return the name of this EjenSourceNode.
     */
    public String  nodeName() {
        return "source";
    }

    /**
     * Returns all non null attributes of this EjenSourceNode.
     * @return non null attributes of this EjenSourceNode.
     */
    public Properties  getAttributes() {
        Properties  attrs = super.getAttributes();

        if (_file != null) {
            attrs.setProperty("file", _file);
        }
        return attrs;
    }

    /**
     * <b>[optional/AVT]</b> - sets the file attribute.
     * If this attribute is missing, the current DOM tree will be reset to
     * default ({@link org.ejen.EjenConstants#DEFAULT_XML_DATA}).
     * @param file the XML file name to load.
     */
    public void setFile(String  file) {
        _file = file;
        putInGlobalContext("DOM_SOURCE_FILE", file);
    }

    /**
     * Checks this EjenSourceNode for mandatory attributes.
     */
    public void check() {
        super.check();
    }
    
    /**
     * Executes this EjenSourceNode. Even if a file is loaded, an "ejen-version"
     * attribute, with the current Ejen version, is always added to the root
     * node of the new DOM tree.
     * @throws org.ejen.EjenException if something goes wrong...
     */
    public void process() {
        super.process();
        try {
            Document  doc = null;

            if (_file != null) {
                doc = DOMUtil.parseXMLFile(evaluateAVT(_file));
            } else {
                doc = DOMUtil.parseXMLString(null);
            }
            doc.getDocumentElement().setAttribute("ejen-version",
                    Version.toString(null));
            putInGlobalContext(CTX_DOM_SOURCE, new DOMSource (doc));
        } catch (EjenException e) {
            throw e;
        } catch (Exception  e) {
            throw new EjenException(this, "bad source file: " + _file, e);
        }
    }
}
