
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import org.ejen.ext.Version;
import org.ejen.util.DOMUtil;
import org.ejen.util.XSLUtil;
import java.util.Vector ;
import java.util.Enumeration ;
import javax.xml.transform.dom.DOMSource ;
import javax.xml.transform.TransformerFactory ;
import org.w3c.dom.Document ;
import org.apache.xalan.processor.TransformerFactoryImpl;

/**
 * Ejen root node class (see {@link org.ejen.EjenTask EjenTask}).
 * <p>
 * @author F. Wolff
 * @version 1.0
 * @see org.ejen.EjenTask
 * @see org.ejen.EjenSourceNode
 * @see org.ejen.EjenMergeNode
 * @see org.ejen.EjenSaveNode
 * @see org.ejen.EjenFilterNode
 * @see org.ejen.EjenTemplateNode
 */
public class EjenRootNode extends EjenChildNode {
    private Vector  _childNodes = new Vector ();

    /**
     * Returns the name of this node ("ejen").
     * @return the name of this node.
     */
    public String  nodeName() {
        return "ejen";
    }

    /**
     * Returns child nodes of this EjenRootNode.
     * @return child nodes of this EjenRootNode.
     */
    public Vector  getChildren() {
        Vector  children = super.getChildren();

        children.addAll(_childNodes);
        return children;
    }

    /**
     * Appends a child node to the current list of child nodes.
     * @param the child node to append.
     * @throws org.ejen.EjenException if the child node is an instance
     *         of EjenParamNode, EjenIncludeNode or EjenImportNode.
     */
    public void appendChildNode(EjenChildNode ecn) {
        if (ecn instanceof EjenParamNode || ecn instanceof EjenIncludeNode
                || ecn instanceof EjenImportNode) {
            throw new EjenException(this, "initialisation error");
        }
        _childNodes.addElement(ecn);
    }

    /**
     * Calls the check method for each child node.
     * @throws org.ejen.EjenException if check failed.
     */
    public void check() {
        super.check();
        for (Enumeration  e = _childNodes.elements(); e.hasMoreElements();) {
            ((EjenChildNode) e.nextElement()).check();
        }
    }

    /**
     * Prepares the generation process.
     * A default DOM tree is created by this method (see
     * {@link org.ejen.EjenConstants#DEFAULT_XML_DATA}).
     * @throws org.ejen.EjenException if something goes wrong...
     */
    public void beforeProcess() {
        super.beforeProcess();
        
        try {
            Document  doc = DOMUtil.parseXMLString(null);

            doc.getDocumentElement().setAttribute("ejen-version",
                    Version.toString(null));
            putInGlobalContext(CTX_DOM_SOURCE, new DOMSource (doc));
            TransformerFactoryImpl tfi = (TransformerFactoryImpl) TransformerFactory.newInstance();

            putInGlobalContext(CTX_TRANSFORMER_FACTORY_IMPL, tfi);
            putInGlobalContext(CTX_TRANSFORMER_IMPL,
                    XSLUtil.getDefaultTransformer(tfi));
            putInGlobalContext(CTX_STYLESHEET_HANDLER, tfi.newTemplatesHandler());
        } catch (EjenException e) {
            throw e;
        } catch (Exception  e) {
            throw new EjenException(this, "initialisation error", e);
        }
        
    }

    /**
     * Starts the generation process (child nodes execution loop).
     * @throws org.ejen.EjenException if something goes wrong.
     */
    public void process() {
        super.process();
        try {
            for (Enumeration  e = _childNodes.elements(); e.hasMoreElements();) {
                EjenChildNode ecn = (EjenChildNode) (e.nextElement());

                // sendMessageEvent("Processing " + ecn.toString());
                _messageIndent = LOG_INDENT_STR2;
                ecn.beforeProcess();
                ecn.process();
                ecn.afterProcess();
                ecn.idle();
                _messageIndent = "";
            }
        } catch (EjenException e) {
            throw e;
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
    }
}
