
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import org.ejen.util.XSLUtil;
import java.util.Properties ;
import java.util.Vector ;
import javax.xml.transform.dom.DOMSource ;
import org.apache.xalan.transformer.TransformerImpl;
import org.apache.xalan.processor.StylesheetHandler;

/**
 * Parent abstract class of all Ejen...Node classes.
 * @author F. Wolff
 * @version 1.0
 */
public abstract class EjenChildNode implements EjenConstants {

    /** Stack of contexts (current transformer,...) */
    private static final EjenContextsStack _ejenContextsStack = new EjenContextsStack();

    /** Current listener for message/error reporting. */
    private static EjenListener _ejenListener = null;

    /** Indentation string for message reporting. */
    protected static String  _messageIndent = "";

    /**
     * Current state of this EjenChildNode
     * ({@link org.ejen.EjenConstants#STATE_IDLE}).
     */
    private int _state = STATE_IDLE;

    /**
     * Returns the name of this EjenChildNode.
     * @return name of this EjenChildNode.
     */
    public abstract String  nodeName();

    /**
     * Returns the attributes of this EjenChildNode (empty here).
     * @return a Properties that contains all attributes of this EjenChildNode.
     */
    public Properties  getAttributes() {
        return new Properties ();
    }

    /**
     * Returns the children of this EjenChildNode (empty here).
     * @return a Vector that contains all children of this EjenChildNode.
     */
    public Vector  getChildren() {
        return new Vector ();
    }

    /**
     * Check this EjenChildNode for mandatory attributes/child nodes.
     * Only sends a state change event here ({@link org.ejen.EjenConstants#STATE_CHECK}).
     */
    public void check() {
        _state = STATE_CHECK;
        sendStateEvent();
    }
    
    /**
     * Prepares this EjenChildNode execution.
     * Only sends a state change event here ({@link org.ejen.EjenConstants#STATE_BEFORE_PROCESS}).
     */
    public void beforeProcess() {
        _state = STATE_BEFORE_PROCESS;
        sendStateEvent();
    }

    /**
     * Executes this EjenChildNode.
     * Only sends a state change event here ({@link org.ejen.EjenConstants#STATE_PROCESS}).
     */
    public void process() {
        _state = STATE_PROCESS;
        sendStateEvent();
    }

    /**
     * Does post-execution actions after this EjenChildNode execution.
     * Only sends a state change event here ({@link org.ejen.EjenConstants#STATE_AFTER_PROCESS}).
     */
    public void afterProcess() {
        _state = STATE_AFTER_PROCESS;
        sendStateEvent();
    }

    /**
     * Signals that this EjenChildNode has completly terminated its execution (including
     * any post-execution actions).
     * Only sends a state change event here ({@link org.ejen.EjenConstants#STATE_IDLE}).
     */
    public final void idle() {
        _state = STATE_IDLE;
        sendStateEvent();
    }
    
    /**
     * Returns the current state of this EjenChildNode.
     * @return the current state (see {@link org.ejen.EjenConstants}).
     */
    public final int getState() {
        return _state;
    }

    /**
     * Sets the listener for all EjenChildNode classes (the listener is static).
     * @param ejenListener the new EjenListener.
     */
    public static final void setListener(EjenListener ejenListener) {
        _ejenListener = ejenListener;
    }

    /**
     * Returns the current listener for all EjenChildNode classes (the listener is static).
     * @return the current EjenListener.
     */
    public static final EjenListener getListener() {
        return _ejenListener;
    }

    /**
     * Sends a state change event (according to the current state of this EjenChildNode).
     * Does nothing if there is no listener.
     */
    public final void sendStateEvent() {
        if (_ejenListener != null) {
            _ejenListener.stateChanged(new EjenEvent(this, STATES[_state],
                    MSG_VERBOSE));
        }
    }

    /**
     * Sends a message event from this EjenChildNode (with level
     * Does nothing if there is no listener.
     * {@link org.ejen.EjenConstants#MSG_INFO}).
     * @param msg the message to send to the listener.
     */
    public final void sendMessageEvent(String  msg) {
        sendMessageEvent(msg, MSG_INFO);
    }

    /**
     * Sends a message event from this EjenChildNode (with a specific level).
     * Does nothing if there is no listener.
     * @param msg the message to send to the listener.
     * @param level level of the message (see {@link org.ejen.EjenConstants}).
     */
    public final void sendMessageEvent(String  msg, int level) {
        if (_ejenListener != null) {
            _ejenListener.nodeMessageSent(new EjenEvent(this,
                    _messageIndent + msg, level));
        }
    }

    /**
     * Sends a message event from an XSL file (via the
     * {@link org.ejen.ext.Messenger#send(XSLProcessorContext context, ElemExtensionCall elem)}
     * extension function).
     * Does nothing if there is no listener.
     * @param msg the message to send to the listener.
     */
    public static final void sendXSLMessageEvent(String  msg) {
        sendXSLMessageEvent(msg, MSG_INFO);
    }

    /**
     * Sends a message event from an XSL file (via the
     * {@link org.ejen.ext.Messenger#send(XSLProcessorContext context, ElemExtensionCall elem)}
     * extension function), with specific level.
     * Does nothing if there is no listener.
     * @param msg the message to send to the listener.
     * @param level level of the message (see {@link org.ejen.EjenConstants}).
     */
    public static final void sendXSLMessageEvent(String  msg, int level) {
        if (_ejenListener != null) {
            _ejenListener.xslMessageSent(new EjenEvent("(xsl)",
                    _messageIndent + "[xsl] " + msg, level));
        }
    }

    /**
     * Returns the String representation of this EjenChildNode, in the form of
     * "&lt;node_name&gt;&nbsp;{[&lt;attribute_name_1&gt;=&lt;attribute_value_1&gt;[,&nbsp;&lt;attribute_name_2&gt;=&lt;attribute_value_2&gt;&nbsp;...]]}".
     * @return the String representation of this EjenChildNode.
     */
    public final String  toString() {
        return nodeName() + " " + getAttributes();
    }
    
    /**
     * Returns an Object (whose name is 'name') from the current context stack.
     * @param name name of the Object to be returned.
     * @return the Object (may be null if there is no such name).
     */
    protected final Object  getFromContext(String  name) {
        return _ejenContextsStack.peekContext().get(name);
    }
    
    /**
     * Returns an Object (whose name is 'name') from the current context stack.
     * @param name name of the value to be returned.
     * @param offset a negative value or zero (to access any context in the stack).
     * @return the Object (may be null if there is no such name).
     * @throws java.util.EmptyStackException if the stack is empty.
     */
    protected final Object  getFromContext(String  name, int offset) {
        return _ejenContextsStack.peekContext(_ejenContextsStack.size() + offset - 1).get(name);
    }
    
    /**
     * Returns an Object (whose name is 'name') from the global (shared) context.
     * @param name name of the value to be returned.
     * @return the Object (may be null if there is no such name).
     */
    protected final Object  getFromGlobalContext(String  name) {
        return _ejenContextsStack.globalGet(name);
    }
    
    /**
     * Puts an Object (whose name is 'name') in the current context (on the top
     * of the stack).
     * @param name name of the Object to be put in the context.
     * @param value the Object.
     * @return the previous Object mapped to the name (may be null).
     */
    protected final Object  putInContext(String  name, Object  value) {
        return _ejenContextsStack.peekContext().put(name, value);
    }
    
    /**
     * Puts an Object (whose name is 'name') in the global (shared) context.
     * @param name name of the Object to be put in the context.
     * @param value the Object.
     * @return the previous Object mapped to the name (may be null).
     */
    protected final Object  putInGlobalContext(String  name, Object  value) {
        return _ejenContextsStack.globalPut(name, value);
    }
    
    /**
     * Pops the current context from the contexts stack.
     * @return the current EjenContext (may not be null).
     * @throws java.util.EmptyStackException if the stack is empty.
     */
    protected final EjenContext popContext() {
        return _ejenContextsStack.popContext();
    }
    
    /**
     * Pushes a new EjenContext onto the top of the contexts stack.
     * @param ejenContext the new EjenContext to be pushed.
     * @throws java.util.EmptyStackException if the stack is empty.
     */
    protected final EjenContext pushContext(EjenContext ejenContext) {
        return _ejenContextsStack.pushContext(ejenContext);
    }
    
    /**
     * Duplicates the current context (or creates a new one if the stack is empty).
     * @return the cloned context.
     */
    protected final EjenContext cloneContext() {
        if (!_ejenContextsStack.isEmpty()) {
            return new EjenContext(_ejenContextsStack.peekContext());
        } else {
            return new EjenContext();
        }
    }
    
    /**
     * Returns the evaluation of an Attribute Value Template.
     * @param avt the Attribute Value Template.
     * @return the evaluation of the Attribute Value Template.
     * @throws org.ejen.EjenException if something goes wrong.
     */
    protected final String  evaluateAVT(String  avt) {
        TransformerImpl ti = null;

        try {
            ti = (TransformerImpl) (getFromGlobalContext(CTX_TRANSFORMER_IMPL));
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }           
        if (ti == null) {
            throw new EjenException(this,
                    "no '" + CTX_TRANSFORMER_IMPL + "' in global context");
        }
        
        return evaluateAVT(ti, avt);
    }
    
    /**
     * Returns the evaluation of an Attribute Value Template, using the provided
     * TransformerImpl object.
     * @param ti a TransformerImpl instance.
     * @param avt the Attribute Value Template.
     * @return the evaluation of the Attribute Value Template.
     * @throws org.ejen.EjenException if something goes wrong.
     */
    protected final String  evaluateAVT(TransformerImpl ti, String  avt) {
        StylesheetHandler sh = null;
        DOMSource  src = null;

        try {
            src = (DOMSource ) (getFromGlobalContext(CTX_DOM_SOURCE));
            sh = (StylesheetHandler) (getFromGlobalContext(CTX_STYLESHEET_HANDLER));
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }           
        if (sh == null) {
            throw new EjenException(this,
                    "no '" + CTX_STYLESHEET_HANDLER + "' in global context");
        }
        if (src == null) {
            throw new EjenException(this,
                    "no '" + CTX_DOM_SOURCE + "' in global context");
        }
        try {
            String  s = XSLUtil.evaluateAttribute(sh, ti.getXPathContext(),
                    src.getNode(), avt);

            if (!s.equals(avt)) {
                sendMessageEvent("Resolved '" + avt + "' into '" + s + "'");
            }
            return s;
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
    }
}
