
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 */

/*
 *
 * @author    Tanja Jovanovic, Nenad Vico
 * @version   1.0.0
 *
 */
package org.enhydra.dods.generator;

import java.io.BufferedReader ;
import java.io.File ;
import java.io.IOException ;
import java.io.InputStreamReader ;
import java.lang.reflect.InvocationTargetException ;
import java.util.ArrayList ;
import java.util.HashMap ;
import java.util.HashSet ;
import org.enhydra.dods.Common;
import org.enhydra.dods.wizard.DefaultDODSWizard;

/**
 * This class is used for generating DODS code and DODS documentation.
 */
public class DODSGenerator {
   // constants
 protected static final int HELP_PARAMETER = 0;
   protected static final int ACTION_PARAMETER = 10;
   protected static final int TEMPLATE_SET_PARAMETER = 20;
   protected static final int FORCE_PARAMETER = 30;
   protected static final int DATABASE_PARAMETER = 40;
   protected static final int CONFIGURATION_DIR_PARAMETER = 50;
   protected static final int HTML_PARAMETER = 100;
   protected static final int PDF_PARAMETER = 110;
   protected static final int XMI_PARAMETER = 120;
   protected static final int PTL_PARAMETER = 130;
   // Messages
 protected static final String  HELP_MESSAGE = "\n";
   protected static final String  INVALID_NUMBER_OF_PARAMETER_MESSAGE = "\nWrong number of input parameters.\n";
   protected static final String  INVALID_PARAMETER_MESSAGE = "\nWrong input parameter ";
   protected static final String  INVALID_ACTION_PARAMETER_MESSAGE = "\nWrong action parameter ";
   protected static final String  INVALID_TEMPLATE_SET_PARAMETER_MESSAGE = "\n Wrong template set ";
   protected static final String  INVALID_DOML_FILE = "\n Wrong .doml file ";
   protected static final String  NONEXISTING_DOML_FILE = "\n Doml file doesn't exist  ";
   protected static final String  INVALID_CONF_DIR_MESSAGE = "\nWrong input parameter ";

   public static final String  DATABASE_NOT_SET = "database_not_set";

   /**
    * Wizard that will be used.
    */
   protected DODSWizard wizard = null;

   /**
    * Doml that will be used.
    */
   protected String  doml = null;

   /**
    * Output directory that will be used.
    */
   protected String  outputDir = null;

   /**
    * Action that will be performed.
    */
   protected String  action = null;

   /**
    * Template set that will be used.
    */
   protected String  templateSet = null;

   /**
    * Template set that will be used.
    */
   protected String  configDir = null;

   /**
    * Indicator for force (overwrite) option.
    */
   protected String  force = "false";

   /**
    * Database vendor.
    */
   protected String  database = null;

   /**
    * Indicator for html option.
    */
   protected boolean html = false;

   /**
    * Indicator for pdf option.
    */
   protected boolean pdf = false;

   /**
    * Indicator for xmi option.
    */
   protected boolean xmi = false;

   /**
    * Indicator for ptl option.
    */
   protected boolean ptl = false;

   /**
    * Indicator whether dods generator should be involved during project ant rebuild.
    */
   protected boolean invoke = false;

   /**
    * Indicator for using kelp.
    */
   protected boolean kelp = false;
   protected static boolean help = false;
   protected static HashMap  parameters;
   protected static HashSet  actions;
   protected static HashSet  templateSets;
   static {
      // initialize parameter mapings
      parameters = new HashMap ();
      parameters.put("-?", new Integer (HELP_PARAMETER));
      parameters.put("-help", new Integer (HELP_PARAMETER));
      parameters.put("-a", new Integer (ACTION_PARAMETER));
      parameters.put("-t", new Integer (TEMPLATE_SET_PARAMETER));
      parameters.put("-f", new Integer (FORCE_PARAMETER));
      parameters.put("-force", new Integer (FORCE_PARAMETER));
      parameters.put("-b", new Integer (DATABASE_PARAMETER));
      parameters.put("-databse", new Integer (DATABASE_PARAMETER));
      parameters.put("-c", new Integer (CONFIGURATION_DIR_PARAMETER));
      parameters.put("-h", new Integer (HTML_PARAMETER));
      parameters.put("-html", new Integer (HTML_PARAMETER));
      parameters.put("-p", new Integer (PDF_PARAMETER));
      parameters.put("-pdf", new Integer (PDF_PARAMETER));
      parameters.put("-x", new Integer (XMI_PARAMETER));
      parameters.put("-xmi", new Integer (XMI_PARAMETER));
      parameters.put("-r", new Integer (PTL_PARAMETER));
      parameters.put("-ptl", new Integer (PTL_PARAMETER));
      // initialize allowed actions
      actions = new HashSet ();
      actions.add("dods:sql");
      actions.add("dods:java");
      actions.add("dods:javaNoCompile");
      actions.add("dods:noCompile");
      actions.add("dods:build_all_split");
      actions.add("dods:sqlsplit");
      actions.add("dods:noCompileSplit");
      actions.add("dods:build_all");
   }

   /**
    * Default constructor.
    */
   public DODSGenerator() {
      wizard = new DefaultDODSWizard(this);
      action = "dods:build_all";
      templateSet = "standard";
      database = DATABASE_NOT_SET;
   }

   /**
    * Method returns used wizard.
    *
    * @return used wizard.
    */
   public DODSWizard getWizard() {
      return wizard;
   }

   /**
    * Method sets wizard that will be used.
    *
    * @param wizard wizard that will be used.
    */
   public void setWizard(DODSWizard wizard) {
      this.wizard = wizard;
   }

   /**
    * Method gets doml file that is used.
    *
    * @return doml file that is used.
    */
   public String  getDoml() {
      return doml;
   }

   /**
    * Method sets doml file that will be used.
    *
    * @param doml doml file that will be used.
    */
   public void setDoml(String  doml) {
      this.doml = doml;
   }

   /**
    * Method gets output directory that is used.
    *
    * @return output directory that is used.
    */
   public String  getOutputDir() {
      return outputDir;
   }

   /**
    * Method sets output directory that will be used.
    *
    * @param outputDir output directory that will be used.
    */
   public void setOutputDir(String  outputDir) {
      this.outputDir = outputDir;
   }

   /**
    * Method gets action that is performed.
    *
    * @return action that is performed.
    */
   public String  getAction() {
      return action;
   }

   /**
    * Method sets action that will be performed.
    *
    * @return action action that will be performed.
    */
   public void setAction(String  action) {
      this.action = action;
   }

   /**
    * Method gets template set that is used.
    *
    * @return template set that is used.
    */
   public String  getTemplateSet() {
      return templateSet;
   }

   /**
    * Method sets template set that will be used.
    *
    * @param templateSet template set that will be used.
    */
   public void setTemplateSet(String  templateSet) {
      this.templateSet = templateSet;
   }

   /**
    * Method gets indicator for force (overwrite) option.
    *
    * @return indicator for force (overwrite) option.
    */
   public String  getForce() {
      return force;
   }

   /**
    * Method sets indicator for force (overwrite) option.
    *
    * @param force indicator for force (overwrite) option.
    */
   public void setForce(String  force) {
      this.force = force;
   }

   /**
    * Method gets database vendor for generating sql.
    *
    * @return database vendor for generating sql.
    */
   public String  getDatabase() {
      return database;
   }

   /**
    * Method sets database vendor for generating sql.
    *
    * @param database database vendor for generating sql.
    */
   public void setDatabase(String  database) {
      this.database = database;
   }

   /**
    * Method gets indicator for html option.
    *
    * @return indicator for html option.
    */
   public boolean getHtml() {
      return html;
   }

   /**
    * Method sets indicator for html option.
    *
    * @param html indicator for html option.
    */
   public void setHtml(boolean html) {
      this.html = html;
   }

   /**
    * Method gets indicator for pdf option.
    *
    * @return indicator for pdf option.
    */
   public boolean getPdf() {
      return pdf;
   }

   /**
    * Method sets indicator for pdf option.
    *
    * @param pdf indicator for pdf option.
    */
   public void setPdf(boolean pdf) {
      this.pdf = pdf;
   }

   /**
    * Method gets indicator for xmi option.
    *
    * @return indicator for xmi option.
    */
   public boolean getXmi() {
      return xmi;
   }

   /**
    * Method sets indicator for xmi option.
    *
    * @param xmi indicator for xmi option.
    */
   public void setXmi(boolean xmi) {
      this.xmi = xmi;
   }

   /**
    * Method gets indicator for ptl option.
    *
    * @return indicator for ptl option.
    */
   public boolean getPtl() {
      return ptl;
   }

   /**
    * Method sets indicator for ptl option.
    *
    * @param ptl indicator for ptl option.
    */
   public void setPtl(boolean ptl) {
      this.ptl = ptl;
   }

   /**
    * Method gets invoke indicator.
    *
    * @return invoke indicator.
    */
   public boolean getInvoke() {
      return invoke;
   }

   /**
    * Method sets invoke indicator.
    *
    * @param invoke invoke indicator.
    */
   public void setInvoke(boolean invoke) {
      this.invoke = invoke;
   }

   /**
    * Method gets indicator for kelp option.
    *
    * @return indicator for kelp option.
    */
   public boolean getKelp() {
      return kelp;
   }

   /**
    * Method sets indicator for kelp option.
    *
    * @param kelp indicator for kelp option.
    */
   public void setKelp(boolean kelp) {
      this.kelp = kelp;
   }

   /**
    * Method runs wizard's GUI .
    */
   public void runWizard() {
      wizard.startup();
   }

   /**
    * This method is called by wizard before generate action. User should
    * override this method to involve any pre-generating action.
    */
   public void preGenerate() {}

   /**
    * This method is called by wizard after generate action. User should
    * override this method to involve any post-generating action.
    */
   public void postGenerate() {}

   /**
    * This method is called before wizard closing. User should
    * override this method to involve any pre-closing action.
    */
   public void preClose() {}

   /**
    * This method is called after wizard closing. User should
    * override this method to involve any post-closing action.
    */
   public void preCancel() {}

   /**
    * Generation of DODS code means generation of ant-build, sql and java files.
    * Method produces <code>Procces</code> which will generate code.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param action type of generation.
    * @param templateSet template set which will be used for generating java code.
    * @param forceBuild "true" if code will be always generated, otherwise only
    *        changes will be regenerated.
    * @param database database vendor for generating sql files.
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *        otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException if any error occurs.
    * <p>
    * Example:
    * <blockquote><pre>
    * String s;
    * Process p = DODSGenerator.generateCode(outputDir,doml,action,template,force,false);
    * BufferedReader buffer = new BufferedReader(new InputStreamReader(p.getInputStream()));
    * while((s = buffer.readLine()) != null) {
    *         System.out.println(s);
    * }
    * </pre></blockquote>
    */



   public static Process  generateCode(String  outputDir, String  domlFile, String  action, String  templateSet,
                                      String  forceBuild, String  database, boolean ownErrorReader)
      throws DODSGenerateException {
      java.lang.Process  process = null;

      try {
         if (action == null) {
            action = "dods:build_all";
         } // default action
 if (templateSet == null) {
            templateSet = "standard";
         }  // default template set
 if (forceBuild == null) {
            forceBuild = "false";
         }      // default no force generating
 if (database == null) {
            database = DATABASE_NOT_SET;
         } // default no force generating
 // call org.enhydra.dods.generator.DODSEjenProperties
 java.util.ArrayList  argsList = new ArrayList ();

         argsList.add(domlFile);
         argsList.add(outputDir);
         argsList.add(templateSet);
         argsList.add(database);
         argsList.add(forceBuild);
         String  className = "org.enhydra.dods.generator.DODSEjenProperties";
         java.lang.reflect.Method  m = null;
         java.lang.Class  c = null;

         c = java.lang.Class.forName(className);
         m = c.getMethod("main", new java.lang.Class [] {
                  String [].class
               });
         String  args[] = (String []) argsList.toArray(((java.lang.Object []) (
                                                         new String [argsList.size()])));

         if (m != null) {
            m.invoke(((java.lang.Object ) (null)), new java.lang.Object [] {
                     args
                  });
         }

         String  as[] = null;
         String  endorsing = System.getProperty("DODS_ENDORSED", null);
         //System.err.println("DODS_ENDORSED is "+ endorsing);
         as = new String [12];
         as[0] = System.getProperty("JAVA_HOME") + File.separator + "bin"
            + File.separator + "java";
         as[1] = "-DDODS_HOME=" + System.getProperty("DODS_HOME");
         as[2] = "-DPROJECT_ROOT=" + outputDir;
         as[3] = "-DDOML_FILE=" + domlFile;
         as[4] = "-DTEMPLATESET=" + templateSet;
         as[5] = "-DDATABASE_VENDOR=" + database;
         as[6] = "-DFORCE=" + forceBuild;
         as[7] = (null != endorsing)
            ?("-Djava.endorsed.dirs=" + endorsing)
            :("-DDODS_HOME=" + System.getProperty("DODS_HOME"));
         as[8] = "org.apache.tools.ant.Main";
         as[9] = "-f";
         as[10] = new String (System.getProperty("DODS_HOME") + File.separator
                                + "build" + File.separator + "generate.xml");
         as[11] = action;
         process = Runtime.getRuntime().exec(as);
         if (!ownErrorReader) {
            BufferedReader  errorBufferedReader = new BufferedReader (new InputStreamReader (process.getErrorStream()));

            (new ErrorReader(errorBufferedReader, true)).start();
         }
      } catch (ClassNotFoundException  e) {
         throw new DODSGenerateException(e);
      } catch (NoSuchMethodException  e) {
         throw new DODSGenerateException(e);
      } catch (IllegalAccessException  e) {
         throw new DODSGenerateException(e);
      } catch (IOException  e) {
         throw new DODSGenerateException(e);
      } catch (InvocationTargetException  e) {
         throw new DODSGenerateException(e);
      }
      return process;
   }

   /**
    * Generation of DODS code means generation of ant-build, sql and java files.
    * Method produces <code>Procces</code> which will generate code.
    *
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *        otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException if any error occurs.
    * <p>
    * Example:
    * <blockquote><pre>
    * String s;
    * Process p = DODSGenerator.generateCode(false);
    * BufferedReader buffer = new BufferedReader(new InputStreamReader(p.getInputStream()));
    * while((s = buffer.readLine()) != null) {
    *         System.out.println(s);
    * }]
    * </pre></blockquote>
    */
   public Process  generateCode(boolean ownErrorReader)
      throws DODSGenerateException {
      return generateCode(outputDir, doml, action, templateSet, force,
                          database, ownErrorReader);
   }

   /**
    * Generates DODS html documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param htmlFile output html file.
    * @param forceBuild "true" if code will be always generated, otherwise only changes
    *          will be regenerated.
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException  if any error occurs.
    *
    * @see        org.enhydra.dods.generator.DODSGenerator#generateCode(String, String, String, String, String, String, boolean)
    */
   public static Process  generateHTML(String  outputDir, String  domlFile, String  htmlFile,
                                      String  forceBuild, boolean ownErrorReader)
      throws DODSGenerateException {
      return generateDocumentation(outputDir, domlFile, htmlFile, forceBuild,
                                   ownErrorReader, "html", "html");
   }

   /**
    * Generates DODS html documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    *
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException  if any error occurs.
    *
    * @see        org.enhydra.dods.generator.DODSGenerator#generateCode(String, String, String, String, String, String, boolean)
    */

   public Process  generateHTML(boolean ownErrorReader)
      throws DODSGenerateException {
      return generateDocumentation(outputDir,doml,null,force,ownErrorReader,"html","html");
   }
   /**
    * Generates DODS pdf documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param pdfFile output pdf file.
    * @param forceBuild "true" if code will be always generated, otherwise only changes
    *          will be regenerated.
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException  if any error occurs.
    *
    * @see        org.enhydra.dods.generator.DODSGenerator#generateCode(String, String, String, String, String, String, boolean)
    */
   public static Process  generatePDF(String  outputDir, String  domlFile, String  pdfFile,
                                     String  forceBuild, boolean ownErrorReader)
      throws DODSGenerateException {
      return generateDocumentation(outputDir, domlFile, pdfFile, forceBuild,
                                   ownErrorReader, "pdf", "pdf");
   }

   /**
    * Generates DODS pdf documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    *
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException  if any error occurs.
    *
    * @see        org.enhydra.dods.generator.DODSGenerator#generateCode(String, String, String, String, String, String, boolean)
    */
   public Process  generatePDF(boolean ownErrorReader)
      throws DODSGenerateException {
      return generateDocumentation(outputDir, doml, null, force,
                                   ownErrorReader, "pdf", "pdf");
   }

   /**
    * Generates DODS xmi documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param xmiFile output xmi file.
    * @param forceBuild "true" if code will be always generated, otherwise only changes
    *          will be regenerated.
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException  if any error occurs.
    *
    * @see        org.enhydra.dods.generator.DODSGenerator#generateCode(String, String, String, String, String, String, boolean)
    */
   public static Process  generateXMI(String  outputDir, String  domlFile, String  xmiFile,
                                     String  forceBuild, boolean ownErrorReader)
      throws DODSGenerateException {
      return generateDocumentation(outputDir, domlFile, xmiFile, forceBuild,
                                   ownErrorReader, "xmi", "xmi");
   }

   /**
    * Generates DODS xmi documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    *
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException  if any error occurs.
    *
    * @see        org.enhydra.dods.generator.DODSGenerator#generateCode(String, String, String, String, String, String, boolean)
    */
   public Process  generateXMI(boolean ownErrorReader)
      throws DODSGenerateException {
      return generateDocumentation(outputDir, doml, null, force,
                                   ownErrorReader, "xmi", "xmi");
   }

   /**
    * Generates DODS ptl (Rational Rose) documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param ptlFile output ptl file.
    * @param forceBuild "true" if code will be always generated, otherwise only changes
    *          will be regenerated.
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException  if any error occurs.
    *
    * @see        org.enhydra.dods.generator.DODSGenerator#generateCode(String, String, String, String, String, String, boolean)
    */
   public static Process  generatePTL(String  outputDir, String  domlFile, String  ptlFile,
                                     String  forceBuild, boolean ownErrorReader)
      throws DODSGenerateException {
      return generateDocumentation(outputDir, domlFile, ptlFile, forceBuild,
                                   ownErrorReader, "ptl", "ptl");
   }

   /**
    * Generates DODS ptl documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    *
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException  if any error occurs.
    *
    * @see        org.enhydra.dods.generator.DODSGenerator#generateCode(String, String, String, String, String, String, boolean)
    */
   public Process  generatePTL(boolean ownErrorReader)
      throws DODSGenerateException {
      return generateDocumentation(outputDir, doml, null, force,
                                   ownErrorReader, "ptl", "ptl");
   }

   /*
    * Generates DODS documentation from .doml file.
    * Method produces <code>Procces</code> which will generate documentation.
    */
   private static Process  generateDocumentation(String  outputDir, String  doml, String  outputName, String  forceBuild,
                                                boolean ownErrorReader, String  type, String  extension)
      throws DODSGenerateException {
      java.lang.Process  process = null;
      String  as[] = new String [3];

      try {
         if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
            as[0] = System.getProperty("DODS_HOME") + File.separator + "bin"
               + File.separator + "doml2" + type + ".bat";
         } else {
            as[0] = System.getProperty("DODS_HOME") + File.separator + "bin"
               + File.separator + "doml2" + type;
         }
         as[1] = doml;
         if (outputName == null) {
            int index = doml.lastIndexOf(File.separator);

            if (index != -1) {
               outputName = doml.substring(index + 1);
            } else {
               outputName = doml;
            }
            outputName = outputName.substring(0,
                                              outputName.lastIndexOf(".doml"));
         }
         as[2] = outputDir + File.separator + outputName + "." + extension;
         process = Runtime.getRuntime().exec(as);
         if (!ownErrorReader) {
            BufferedReader  errorBufferedReader = new BufferedReader (new InputStreamReader (process.getErrorStream()));

            (new ErrorReader(errorBufferedReader, true)).start();
         }
      } catch (IOException  e) {
         throw new DODSGenerateException(e);
      } catch (StringIndexOutOfBoundsException  e) {
         throw new DODSGenerateException(e);
      }
      return process;
   }

   /**
    * Generation of DODS code means generation of ant-build, sql, java files, html, pdf,
    * xmi and ptl. Method produces <code>Procces</code> which will generate code.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param genAction type of generation.
    * @param templateSet template set which will be used for generating java code.
    * @param database database vendor for generating sql files.
    * @param genHtml if <code>true</code> generates DODS html documentation from .doml file.
    * @param genPdf if <code>true</code> generates DODS pdf documentation from .doml file.
    * @param genXmi if <code>true</code> generates DODS xmi documentation from .doml file.
    * @param genPtl if <code>true</code> generates DODS ptl documentation from .doml file.
    *
    * @return error code of executing process.
    *
    * @exception  DODSGenerateException if any error occurs.
    */
   public static int generateAll(String  outputDir, String  domlFile, String  genAction, String  templateSet,
                                 String  forceBuild, String  database, boolean genHtml, boolean genPdf,
                                 boolean genXmi, boolean genPtl)
      throws DODSGenerateException {
      int exit = 0;

      try {
         ErrorReader errorReader;
         Process  process;
         BufferedReader  buffer;
         BufferedReader  error;
         String  s;
         File  f = new File (domlFile);

         outputDir = (new File (outputDir)).getAbsolutePath();
         File  dir = new File (outputDir);

         dir.mkdirs();
         domlFile = f.getAbsolutePath();
         f = new File (domlFile);
         if (!f.exists()) {
            throw new DODSGenerateException(NONEXISTING_DOML_FILE + " "
                                               + domlFile + "\n");
         }
         if (genHtml) {
            process = DODSGenerator.generateHTML(outputDir, domlFile, null,
                                                 "true", true);
            buffer = new BufferedReader (new InputStreamReader (process.getInputStream()));
            error = new BufferedReader (new InputStreamReader (process.getErrorStream()));
            (new ErrorReader(error, true)).start();
            while ((s = buffer.readLine()) != null) {
               System.out.println(s);
            }
            if (exit != 0) {
               return exit;
            }
         }
         if (genPdf) {
            process = DODSGenerator.generatePDF(outputDir, domlFile, null,
                                                "true", true);
            buffer = new BufferedReader (new InputStreamReader (process.getInputStream()));
            error = new BufferedReader (new InputStreamReader (process.getErrorStream()));
            (new ErrorReader(error, true)).start();
            while ((s = buffer.readLine()) != null) {
               System.out.println(s);
            }
            if (exit != 0) {
               return exit;
            }
         }
         if (genXmi) {
            process = DODSGenerator.generateXMI(outputDir, domlFile, null,
                                                "true", true);
            buffer = new BufferedReader (new InputStreamReader (process.getInputStream()));
            error = new BufferedReader (new InputStreamReader (process.getErrorStream()));
            (new ErrorReader(error, true)).start();
            while ((s = buffer.readLine()) != null) {
               System.out.println(s);
            }
            if (exit != 0) {
               return exit;
            }
         }
         if (genPtl) {
            process = DODSGenerator.generatePTL(outputDir, domlFile, null,
                                                "true", true);
            buffer = new BufferedReader (new InputStreamReader (process.getInputStream()));
            error = new BufferedReader (new InputStreamReader (process.getErrorStream()));
            (new ErrorReader(error, true)).start();
            while ((s = buffer.readLine()) != null) {
               System.out.println(s);
            }
            if (exit != 0) {
               return exit;
            }
         }
         if(!genAction.equalsIgnoreCase("dods:generatorOff")){
            process = DODSGenerator.generateCode(outputDir, domlFile, genAction,
                                                 templateSet, forceBuild, database, true);
            buffer = new BufferedReader (new InputStreamReader (process.getInputStream()));
            error = new BufferedReader (new InputStreamReader (process.getErrorStream()));
            (new ErrorReader(error, true)).start();
            while ((s = buffer.readLine()) != null) {
               System.out.println(s);
            }
            exit = process.waitFor();
         }
      } catch (IOException  e) {
         throw new DODSGenerateException(e);
      } catch (InterruptedException  e) {
         throw new DODSGenerateException(e);
      } catch (DODSGenerateException e) {
         Throwable  pe = e.getCause();
         if (pe instanceof java.lang.StringIndexOutOfBoundsException ) {
            System.out.println(INVALID_DOML_FILE + domlFile + "\n");
         } else {
            throw new DODSGenerateException(e);
         }
      }
      return exit;
   }

   /**
    * Generation of DODS code means generation of ant-build, sql, java files, html, pdf,
    * xmi and ptl. Method produces <code>Procces</code> which will generate code.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param genAction type of generation.
    * @param templateDir template directory.
    * @param templateSet template set which will be used for generating java code.
    * @param database database vendor for generating sql files.
    * @param genHtml if <code>true</code> generates DODS html documentation from .doml file.
    * @param genPdf if <code>true</code> generates DODS pdf documentation from .doml file.
    * @param genXmi if <code>true</code> generates DODS xmi documentation from .doml file.
    * @param genPtl if <code>true</code> generates DODS ptl documentation from .doml file.
    *
    * @return error code of executing process.
    *
    * @exception  DODSGenerateException if any error occurs.
    */

   public static int generateAll(String  outputDir, String  domlFile, String  genAction, String  configDir, String  templateDir,
                                 String  templateSet, String  forceBuild, String  database, boolean genHtml, boolean genPdf,
                                 boolean genXmi, boolean genPtl)
      throws DODSGenerateException {
      Common.setCustomTemplateDir(templateDir);
      Common.setConfigDir(configDir);

      return generateAll(outputDir, domlFile, genAction, templateSet,
                         forceBuild, database, genHtml, genPdf, genXmi, genPtl);
   }

   /**
    * Generation of DODS code means generation of ant-build, sql, java files, html, pdf,
    * xmi and ptl. Method produces <code>Procces</code> which will generate code.
    *
    * @return error code of executing process.
    *
    * @exception  DODSGenerateException if any error occurs.
    */
   public int generateAll()
      throws DODSGenerateException {
      return generateAll(outputDir, doml, action, templateSet, force, database,
                         html, pdf, xmi, ptl);
   }

   /**
    * Generation of DODS code means generation of ant-build, sql, java files, html, pdf,
    * xmi and ptl. Method produces <code>Procces</code> which will generate code.
    * This method call (dods.bat/dods) with all parameters.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param genAction type of generation.
    * @param configDir path to configuration directory
    * @param templateSet template set which will be used for generating java code.
    * @param forceBuild "true" if code will be always generated, otherwise only
    *        changes will be regenerated.
    * @param database database vendor for generating sql files.
    * @param genHtml if <code>true</code> generates DODS html documentation from .doml file.
    * @param genPdf if <code>true</code> generates DODS pdf documentation from .doml file.
    * @param genXmi if <code>true</code> generates DODS xmi documentation from .doml file.
    * @param genPtl if <code>true</code> generates DODS ptl documentation from .doml file.
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException if any error occurs.
    */
   public static Process  generateAll(String  outputDir, String  domlFile, String  genAction, String  configDir,
                                     String  templateSet, String  forceBuild, String  database,
                                     boolean genHtml, boolean genPdf, boolean genXmi,
                                     boolean genPtl, boolean ownErrorReader)
      throws DODSGenerateException {
      int exit = 0;
      Process  process = null;

      try {
         ErrorReader errorReader;
         BufferedReader  buffer;
         BufferedReader  error;
         String  s;
         File  f = new File (domlFile);

         outputDir = (new File (outputDir)).getAbsolutePath();
         domlFile = f.getAbsolutePath();
         f = new File (domlFile);
         if (!f.exists()) {
            throw new DODSGenerateException(NONEXISTING_DOML_FILE + " "
                                               + domlFile + "\n");
         }
         String  tmp[] = new String [16];
         int param = 1;

         if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
            tmp[0] = System.getProperty("DODS_HOME") + File.separator
               + "bin" + File.separator + "dods.bat";
         } else {
            tmp[0] = System.getProperty("DODS_HOME") + File.separator
               + "bin" + File.separator + "dods";
         }
         if (genAction != null) {
            tmp[param] = "-a";
            param++;
            tmp[param] = genAction;
            param++;
         }
         if (configDir != null) {
            tmp[param] = "-c";
            param++;
            tmp[param] = configDir;
            param++;
         }
         if (templateSet != null) {
            tmp[param] = "-t";
            param++;
            tmp[param] = templateSet;
            param++;
         }
         if (forceBuild != null) {
            if (forceBuild.equals("true")) {
               tmp[param] = "-f";
               param++;
            }
         }
         if (database != null) {
            tmp[param] = "-b";
            param++;
            tmp[param] = database;
            param++;
         }
         if (genHtml) {
            tmp[param] = "-h";
            param++;
         }
         if (genPdf) {
            tmp[param] = "-p";
            param++;
         }
         if (genXmi) {
            tmp[param] = "-x";
            param++;
         }
         if (genPtl) {
            tmp[param] = "-r";
            param++;
         }
         tmp[param] = domlFile;
         param++;
         tmp[param] = outputDir;
         param++;
         String  as[] = new String [param];

         for (int i = 0; i < param; i++) {
            as[i] = tmp[i];
         }
         process = Runtime.getRuntime().exec(as);
         if (!ownErrorReader) {
            BufferedReader  errorBufferedReader = new BufferedReader (new InputStreamReader (process.getErrorStream()));

            (new ErrorReader(errorBufferedReader, true)).start();
         }
      } catch (IOException  e) {
         e.printStackTrace();
         throw new DODSGenerateException(e);
      }
      return process;
   }

   /**
    * Generation of DODS code means generation of ant-build, sql, java files, html, pdf,
    * xmi and ptl. Method produces <code>Procces</code> which will generate code.
    * This method call (dods.bat/dods) with all parameters.
    *
    * @param outputDir full path to output directory that will be used.
    * @param domlFile full path to .doml file for generating code.
    * @param genAction type of generation.
    * @param templateSet template set which will be used for generating java code.
    * @param forceBuild "true" if code will be always generated, otherwise only
    *        changes will be regenerated.
    * @param database database vendor for generating sql files.
    * @param genHtml if <code>true</code> generates DODS html documentation from .doml file.
    * @param genPdf if <code>true</code> generates DODS pdf documentation from .doml file.
    * @param genXmi if <code>true</code> generates DODS xmi documentation from .doml file.
    * @param genPtl if <code>true</code> generates DODS ptl documentation from .doml file.
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException if any error occurs.
    */
   public static Process  generateAll(String  outputDir, String  domlFile, String  genAction,
                                     String  templateSet, String  forceBuild, String  database,
                                     boolean genHtml, boolean genPdf, boolean genXmi,
                                     boolean genPtl, boolean ownErrorReader)
      throws DODSGenerateException {
      return  generateAll(outputDir, domlFile, genAction, null, templateSet,
                          forceBuild, database, genHtml, genPdf, genXmi, genPtl,
                          ownErrorReader);
   }

   /**
    * Generation of DODS code means generation of ant-build, sql, java files, html, pdf,
    * xmi and ptl. Method produces <code>Procces</code> which will generate code.
    * This method call (dods.bat/dods) with all parameters.
    *
    * @param ownErrorReader <code>true</code> if user needs to make its own ErrorReader,
    *          otherwise (<code>false</code>) method will make <code>ErrorReader</code>.
    *
    * @return procces which will generate code.
    *
    * @exception  DODSGenerateException if any error occurs.
    */
   public Process  generateAll(boolean ownErrorReader)
      throws DODSGenerateException {
      return generateAll(outputDir, doml, action, configDir, templateSet,
                         force, database, html, pdf, xmi, ptl, ownErrorReader);
   }

   /**
    * Generates Command line help.
    */
   public static void help() {
      String  dods = "dods";
      String  space = "    ";
      String  example = "/home/tanja/test/discRack.doml /home/tanja/test/discRack";

      if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
         dods = "dods.bat";
         space = "        ";
         example = "c:\\test\\discRack.doml c:\\test\\discRack";
      }
      System.out.println("\nCommand line help:\n");
      System.out.println(dods
                            + " [-?/help] [-a action] [-t templateset] [-b/-database] [-c confPath] ");
      System.out.println(space
                            + "[-f/force] [-h/html] [-p/pdf] [-x/xmi] [-r/ptl] domlfile outputdir\n");
      System.out.println(" where:\n");
      System.out.println("  outputdir   full path to output directory that will be used.\n");
      System.out.println("  domlfile      full path to .doml file for generating code.\n");
      System.out.println(" options:\n");
      System.out.println("  [-? -help]    shows help.\n");
      System.out.println("  [-a action]   ant task parameter for code generation:");
      System.out.println("        dods:build_all       - to create all sql files and java classes (default).");
      System.out.println("        dods:sql             - to create only sql files.");
      System.out.println("        dods:java            - to create only java files and to compile them.");
      System.out.println("        dods:javaNoCompile   - to create only java files and not to compile them.");
      System.out.println("        dods:noCompile       - to create SQL files and java files and not to");
      System.out.println("                               compile them.");
      System.out.println("        dods:build_all_split - to create all sql files and java classes and to");
      System.out.println("                               compile it. SQL files will be divided into separate");
      System.out.println("                               files using SQLSplitter .");
      System.out.println("        dods:sqlsplit        - to create only sql files and separate in different");
      System.out.println("                               files using SQLSplitter.");
      System.out.println("        dods:noCompileSplit  - to create SQL files and separate sql commands using");
      System.out.println("                               SQLSplitter and java files and not to compile them.\n");
      System.out.println("  [-t templateset]  template set for generating java and sql code:");
      System.out.println("                 standard         - generate standard java code (default).");
      System.out.println("                 <user_defined>   - any user defined template set.\n");
      System.out.println("  [-b/-database] sets database vendor for generating sql.\n");
      System.out.println("  [-c confPath]  sets folder with dodsConf.xml file \n");
      System.out.println("  [-f/-force]  with this switch, code will be always generated, without it, only changes");
      System.out.println("               will be regenerated.\n");
      System.out.println("  [-h/-html]    generates DODS html documentation from .doml file.\n");
      System.out.println("  [-p/-pdf]     generates DODS pdf documentation from .doml file.\n");
      System.out.println("  [-x/-xmi]     generates DODS xmi documentation from .doml file.\n");
      System.out.println("  [-r/-ptl]     generates DODS ptl (Rational Rose) documentation from .doml file.\n");
      System.out.println("Example:\n");
      System.out.println("  " + dods + " -a dods:java -t standard -f -pdf -x "
                            + example + "\n");
   }

   /**
    * Parses input parameters.
    *
    * @param args array of input command line parameters.
    * @return error message, null if parsing was OK.
    */
   public String  parse(String [] args) {
      try {
         int curr = 0;
         int param = -1;

         while (curr < args.length) {
            Integer  integer = (Integer ) parameters.get(args[curr]);

            if (integer == null) {
               break;
            }
            param = integer.intValue();
            curr++;
            switch (param) {
               case HELP_PARAMETER: {
                     help = true;
                     return HELP_MESSAGE;
                  }

               case ACTION_PARAMETER: {
                     if (actions.contains(args[curr])) {
                        action = args[curr];
                     } else {
                        return INVALID_ACTION_PARAMETER_MESSAGE + args[curr];
                     }
                     curr++;
                     break;
                  }

               case CONFIGURATION_DIR_PARAMETER: {
                     configDir = args[curr];
                     curr++;
                     break;
                  }

               case TEMPLATE_SET_PARAMETER: {
                     templateSet = args[curr];
                     curr++;
                     break;
                  }

               case DATABASE_PARAMETER: {
                     database = args[curr];
                     curr++;
                     break;
                  }

               case FORCE_PARAMETER: {
                     force = "true";
                     break;
                  }

               case HTML_PARAMETER: {
                     html = true;
                     break;
                  }

               case PDF_PARAMETER: {
                     pdf = true;
                     break;
                  }

               case XMI_PARAMETER: {
                     xmi = true;
                     break;
                  }

               case PTL_PARAMETER: {
                     ptl = true;
                     break;
                  }
            } // switch
         } // while

         try {
            Common.setConfigDir(configDir);
         } catch (Error  e) {
            return INVALID_CONF_DIR_MESSAGE + configDir;
         }
         templateSets = Common.getAllTemplateSets();

         if (!templateSets.contains(templateSet)) {
            return INVALID_TEMPLATE_SET_PARAMETER_MESSAGE + templateSet;
         }
         if (args.length - curr < 2) {
            return INVALID_NUMBER_OF_PARAMETER_MESSAGE;
         }
         if (args[curr].startsWith("-") || args[curr + 1].startsWith("-")) {
            return INVALID_PARAMETER_MESSAGE + args[curr];
         }
         doml = args[curr];
         outputDir = args[curr + 1];
      } catch (NullPointerException  e) {
         return INVALID_PARAMETER_MESSAGE;
      } catch (ArrayIndexOutOfBoundsException  e) {
         return INVALID_NUMBER_OF_PARAMETER_MESSAGE;
      }
      return null;
   }

   /**
    * Generates DODS.
    * Method parses command line parameters and, depending on that, calls wizard
    * or generates DODS.
    *
    * <blockquote><pre>
    * Command line:
    * dods [-?/help] [-a action] [-t templateset] [-b/-database]  [-c confPath]
    *      [-f/force] [-h/html] [-p/pdf] [-x/xmi] [-r/ptl] domlfile outputdir
    *
    *  where:
    *
    *   outputdir   full path to output directory that will be used.
    *
    *   domlfile      full path to .doml file for generating code.
    *
    *  options:
    *
    *   [-? -help]    shows help.
    *
    *   [-a action]   ant task parameter for code generation:
    *         dods:build_all       - to create all sql files and java classes (default).
    *         dods:sql             - to create only sql files.
    *         dods:java            - to create only java files and to compile them.
    *         dods:javaNoCompile   - to create only java files and not to compile them.
    *         dods:noCompile       - to create SQL files and java files and not to
    *                                compile them.
    *         dods:build_all_split - to create all sql files and java classes and to
    *                                compile it. SQL files will be divided into separate
    *                                files using SQLSplitter .
    *         dods:sqlsplit        - to create only sql files and separate in different
    *                                files using SQLSplitter.
    *         dods:noCompileSplit  - to create SQL files and separate sql commands using
    *                                SQLSplitter and java files and not to compile them.
    *
    *   [-t templateset]  template set for generating java and sql code:
    *              standard         - generate standard java code (default).
    *         <user_defined>   - any user defined template set.
    *
    *   [-b/-database] sets database vendor for generating sql.
    *
    *   [-c confPath]  sets folder with dodsConf.xml file.
    *
    *   [-f/-force]  with this switch, code will be always generated, without it, only changes
    *                will be regenerated.
    *
    *   [-h/-html]    generates DODS html documentation from .doml file.
    *
    *   [-p/-pdf]     generates DODS pdf documentation from .doml file.
    *
    *   [-x/-xmi]     generates DODS xmi documentation from .doml file.
    *
    *   [-r/-ptl]     generates DODS ptl (Rational Rose) documentation from .doml file.
    *
    * <p>
    * Example:
    * DODSGenerator -a dods:java -t standard -f -pdf -x discrack.doml /test/discRack
    * </pre></blockquote>
    */
   public static void main(String [] args) {
      DODSGenerator generator = new DODSGenerator();

      if (args.length == 0) {
         generator.runWizard();
         return;
      }
      try {
         String   message = generator.parse(args);

         if (message != null) {
            System.out.println(message);
            help();
            return;
         }
         int exit = generator.generateAll();

         System.exit(exit);
      } catch (DODSGenerateException e) {
         String  message = e.getLocalizedMessage();

         if (message != null) {
            System.out.println(message);
         } else {
            e.printStackTrace();
         }
      } catch (Exception  e) {
         e.printStackTrace();
      }
   }

   /**
    * @param conDir Sets configuration directory
    */
   public void setConfigDir(String  conDir) {
      configDir = conDir;
   }

   /**
    * @return actual config dir
    */
   public String  getConfigDir() {
      return configDir;
   }
}
