
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: Table.java,v 1.1 2004/09/03 13:43:13 sinisa Exp $
 */

/*
 *
 * @author    Nenad Vico
 * @since     LBS1.8
 * @version   $Revision: 1.1 $
 *
 */
package org.enhydra.dods.trans;

import java.util.*;

/**
 * This class represents database table.
 */
public class Table {
    
    /**
     * table package
     */
    protected String  pckg = null; 

    /**
     * author
     */
    protected String  author = DefaultTagValues.TABLE_AUTHOR; 

    /**
     * project name
     */
    protected String  project_name = DefaultTagValues.TABLE_PROJECT_NAME; 

    /**
     * table name
     */
    protected String  table_name = null; 

    /**
     * class name
     */
    protected String  class_name = null; 

    /**
     * database vendor
     */
    protected String  db_vendor = DefaultTagValues.TABLE_DB_VENDOR; 
    private static final String  PARVALUE_COMPATIBLE = "Compatible";
    private static final String  PARVALUE_DEPRECATE = "Deprecate";
    private static final String  PARVALUE_OMIT = "Omit";

    /**
     * How to generate dirty methods.
     */
    protected String  dirtyDOs = PARVALUE_COMPATIBLE;

    /**
     * string match (string which represents LIKE in Standard database)
     */
    public static final String [] caching_value = {"none", "partial", "full",
        "lru"};
    public static final int NONE = 0;
    public static final int PARTIAL = 1;    
    public static final int FULL = 2;
    public static final int LRU = 3;

    /**
     * This variable contains information whether the table is 
     * based on OID primary key.
     */
    protected boolean do_is_oid_based = DefaultTagValues.DO_IS_OID_BASED; 

    /**
     * This variable contains information whether the table need security methods.
     */
    protected boolean  do_secure = DefaultTagValues.DO_SECURITY; 

    /**
     * This variable contains information whether the table need unsecurity methods.
     */
    protected boolean  do_insecure = DefaultTagValues.DO_NON_SECURITY; 

    /**
     * This variable contains information whether the table can use multidb methods.
     */
    protected boolean  do_multidb = DefaultTagValues.DO_MULTIDB; 

    /**
     *  This variable contains information whether the table is abstarct.
     */
    protected boolean is_abstract = false; 

    /**
     * This variable contains information whether the table uses delete cascade capability.
     */
    protected boolean delete_cascade = false; 
    
    /**
     * This variable contains information whether the table uses class for 'mass-updates'.
     */
    protected boolean mass_updates = DefaultTagValues.MASS_UPDATES; 

    /**
     * This variable contains information whether the table uses class for 'mass-deletes'.
     */
    protected boolean mass_deletes = DefaultTagValues.MASS_DELETES; 

    /**
     * list of table columns
     */
    protected ArrayList columns = new ArrayList();

    /**
     * list of table referrers
     */
    protected HashMap referrers = new HashMap();

    /**
     * This variable contains information whether is any of table columns has secure methods.
     */
    protected boolean isAnyColumnSecure = DefaultTagValues.IS_ANY_COLUMN_SECURE;        

    /**
     * list of table indexes
     */
    protected ArrayList indexes = new ArrayList();
    
    public Table(boolean genSecure, boolean genInsecure, String  dirty) {
        do_secure = genSecure;
        do_insecure = genInsecure;
        if (dirtyDOs != null) {
            dirtyDOs = dirty;
        }
    }
     
    public Table() {}     
      
    // -------------------------    get methodes    -----------------------------
 /**
     * Get table package.
     *
     * @return Table package.
     */
    public String  pckg() {
        return pckg;    
    }

    /**
     * Get author.
     *
     * @return Author.
     */
    public String  author() {
        return author;  
    }
   
    /**
     * Get project name.
     *
     * @return Project name.
     */
    public String  projectName() {
        return project_name;    
    }

    /**
     * Get table name.
     *
     * @return Table name.
     */
    public String  tableName() {
        return table_name;  
    }

    /**
     * Get class name.
     *
     * @return Class name.
     */
    public String  className() {
        return class_name;  
    }

    /**
     * Get database vendor.
     *
     * @return Database vendor.
     */
    public String  dbVendor() {
        return db_vendor;   
    }

    /**
     * Get the information whether DO is OID based.
     *
     * @return True if DO is OID based, otherwise false.
     */
    public boolean doIsOidBased() {
        return do_is_oid_based; 
    }

    /**
     * Get the information whether DO has security methods.
     *
     * @return True if DO is security based, otherwise false.
     */
    public boolean doSecure() {
        return do_secure; 
    }

    /**
     * Get the information whether DO has security methods.
     *
     * @return True if DO is unsecurity based, otherwise false.
     */
    public boolean doInSecure() {
        return do_insecure; 
    }

    /**
     * Get the information whether DO has multidb methods.
     *
     * @return True if DO is multidb based, otherwise false.
     */
    public boolean doMultidb() {
        return do_multidb; 
    }

    /**
     * Get the information whether table has mass-updates class.
     *
     * @return True if Update class will be generated, otherwise false.
     */
    public boolean doMassUpdates() {
        return mass_updates; 
    }

    /**
     * Get the information whether table has mass-deletes class.
     *
     * @return True if Delete class will be generated, otherwise false.
     */
    public boolean doMassDeletes() {
        return mass_deletes; 
    }


    /**
     * Get the information whether table is abstarct.
     *
     * @return True if table is abstarct, otherwise false.
     */
    public boolean isAbstract() {
        return is_abstract; 
    }

    /**
     * Get the information is any table column secure based.
     *
     * @return True if table is abstarct, otherwise false.
     */
    public boolean isAnyColumnSecure() {
        return isAnyColumnSecure; 
    }

    /**
     * Get the information whether table uses delete cascade.
     *
     * @return True if table uses delete cascade, otherwise false.
     */
    public boolean deleteCascade() {
        return delete_cascade;  
    }

    /**
     * Get list of columns.
     *
     * @return List of columns.
     */
    public ArrayList columns() {
        return columns;
    }

    /**
     * Get number of columns.
     *
     * @return Number of columns.
     */
    public int columnsSize() {
        return columns.size();
    }

    /**
     * Get column with given index.
     *
     * @param index Index of desired column.
     *
     * @return Column with given index.
     */
    public Column column(int index) {
        return (Column) columns.get(index);
    }

    /**
     * Get referrers.
     *
     * @return Referrers.
     */
    public HashMap referrers() {
        return referrers;
    }

    /**
     * Get number of referrers.
     *
     * @return Number of referrers.
     */
    public int referrersSize() {
        return referrers.size();
    }

    /**
     * Get entitySet iterator of referrers.
     *
     * @return entitySet Iterator of referrers.
     */
    public Iterator referrersIterator() {
        return referrers.entrySet().iterator();
    }

    /**
     * Get key iterator of referrers.
     *
     * @return key Iterator of referrers.
     */
    public Iterator referrersKeyIterator() {
        return referrers.keySet().iterator();
    }

    /**
     * Get value of iterator of referrers.
     *
     * @return value Value of iterator of referrers.
     */
    public Iterator referrersValueIterator() {
        return referrers.values().iterator();
    }

    /**
     * Get value of collection of referrers.
     *
     * @return value Value of iterator of referrers.
     */
    public Collection referrersValues() {
        return referrers.values();
    }

    /**
     * Get list of indexes.
     *
     * @return List of indexes.
     */
    public ArrayList indexes() {
        return indexes;
    }

    /**
     * Get number of indexes.
     *
     * @return Number of indexes.
     */
    public int indexesSize() {
        return indexes.size();
    }

    /**
     * Get index with given index.
     *
     * @return Index with given index.
     */
    public Index index(int index) {
        return (Index) indexes.get(index);
    }

    // -------------------------    set methodes    -----------------------------
 /**
     * Set table package.
     *
     * @param pckg Table package.
     */
    public void pckg(String  pckg) {
        this.pckg = pckg;   
    }

    /**
     * Set author.
     *
     * @param author Author.
     */
    public void author(String  author) {
        this.author = author;   
    }

    /**
     * Set project name.
     *
     * @param project_name Project name.
     */
    public void projectName(String  project_name) {
        this.project_name = project_name;   
    }

    /**
     * Set table name.
     *
     * @param table_name Table name.
     */
    public void tableName(String  table_name) {
        this.table_name = table_name;   
    }

    /**
     * Set class name.
     *
     * @param class_name Class name.
     */
    public void className(String  class_name) {
        this.class_name = class_name;   
    }

    /**
     * Set database vendor.
     *
     * @param db_vendor Database vendor.
     */
    public void dbVendor(String  db_vendor) {
        this.db_vendor = db_vendor; 
    }

    /**
     * Set DO to/not to be OID based.
     *
     * @param do_is_oid_based True if DO is OID based, otherwise false.
     */
    public void doIsOidBased(boolean do_is_oid_based) {
        this.do_is_oid_based = do_is_oid_based; 
    }

    /**
     * Set DO to/not to have security methods.
     *
     * @param do_secure True if DO need security methods, otherwise false.
     */
    public void doSecure(boolean do_secure) {
        this.do_secure = do_secure; 
    }

    /**
     * Set DO to/not to have security methods.
     *
     * @param do_Insecure True if DO need unsecurity methods, otherwise false.
     */
    public void doInSecure(boolean do_Insecure) {
        this.do_insecure = do_Insecure; 
    }

    /**
     * Set table to/not to have Update class.
     *
     * @param do_mass True if table need mass-updates methods, otherwise false.
     */
    public void doMassUpdates(boolean do_mass) {
        this.mass_updates = do_mass; 
    }

    /**
     * Set table to/not to have Delete class.
     *
     * @param do_mass True if table need mass-deletes methods, otherwise false.
     */
    public void doMassDeletes(boolean do_mass) {
        this.mass_deletes = do_mass; 
    }
    /**
     * Set DO to/not to have multidb methods.
     *
     * @param do_Multidb True if DO has multidb methods, otherwise false.
     */
    public void doMultidb(boolean do_Multidb) {
        this.do_multidb = do_Multidb; 
    }

    /**
     * Set table to/not to be abstarct.
     *
     * @param is_abstract True if table is abstarct, otherwise false.
     */
    public void isAbstract(boolean is_abstract) {
        this.is_abstract = is_abstract; 
    }

    /**
     * Set value if any table column secure based.
     *
     * @param value True if table is abstarct, otherwise false.
     */
    public void anyColumnSecure(boolean value) {
        this.isAnyColumnSecure = value; 
    }

    /**
     * Set table to/not use delete cascade.
     *
     * @param delete_cascade True if table uses delete cascade, otherwise false.
     */
    public void refConstarint(boolean delete_cascade) {
        this.delete_cascade = delete_cascade;   
    }

    /**
     * Set list of columns.
     *
     * @param columns List of columns.
     */
    public void columns(ArrayList columns) {
        this.columns = columns;
    }

    /**
     * Add column into table.
     *
     * @param column Column which will be added.
     */
    public void addColumn(Column column) {
        columns.add(column);
    }

    /**
     * Set referrers.
     *
     * @param referrers New referrers of table.
     */
    public void referrers(HashMap referrers) {
        this.referrers = referrers;
    }

    /**
     * Add index into table.
     *
     * @param index Index which will be added to table.
     */
    public void addIndex(Index index) {
        indexes.add(index);
    }

    /**
     * Set indexes.
     *
     * @param indexes New indexes of table.
     */
    public void indexes(ArrayList indexes) {
        this.indexes = indexes;
    }

    /**
     * Add referrer into table.
     *
     * @param referrer Referrer which will be added.
     *
     * @return True if referrer is added, otherwise false.
     */
    public boolean addReferrer(Referrer referrer) {
        String  id = referrer.pckg() + "." + referrer.name();
        boolean inserted = !referrers.containsKey(id);

        if (inserted) {
            referrers.put(id, referrer);
        }
        return inserted;
    }

    /**
     * Sets String specifying what to do with dirty DO methods
     *
     * @param howToGenerate - can be "Compatble", "Deprecate", "Omit"
     */
    public void setDirtyDOs(String  howToGenerate)throws InvalidDomlException {
        if ((0 != howToGenerate.compareTo(PARVALUE_COMPATIBLE))
                && (0 != howToGenerate.compareTo(PARVALUE_DEPRECATE))
                && (0 != howToGenerate.compareTo(PARVALUE_OMIT))) {
            throw new InvalidDomlException ("DirtyDOs out of range!");
        }
        dirtyDOs = howToGenerate;
    }

    /**
     * @return String specifying what to do with dirty DO methods
     */
    public String  getDirtyDOs() {
        return dirtyDOs;
    }

    /**
     * Get string representation.
     *
     * @return String representation.
     */
    public String  toString() {
        StringBuffer  ret = new StringBuffer ("package=").append(pckg).append(" author=").append(author);

        ret.append(" project_name=").append(project_name).append(" table_name=").append(table_name);
        ret.append(" class_name=").append(class_name).append(" db_vendor=").append(db_vendor);
        ret.append(" do_is_oid_based=").append(do_is_oid_based).append(" is_abstract=").append(is_abstract);
        ret.append(" delete_cascade=").append(delete_cascade);
        ret.append(" do_secure=").append(do_secure);
        ret.append(" do_insecure=").append(do_insecure);
        ret.append("\ncolumns:\n");
        for (int i = 0; i < columns.size(); i++) {
            ret.append("  column=").append(columns.get(i)).append("\n");
        }
        ret.append(":referrers:\n");
        Iterator iter = referrers.values().iterator();

        while (iter.hasNext()) {
            ret.append("  referrer=").append(iter.next()).append("\n");
        }
        ret.append("\nindexess:\n");
        for (int i = 0; i < indexes.size(); i++) {
            ret.append("  index=").append(indexes.get(i)).append("\n");
        }
        return ret.toString();
    }

    public static void main(String [] args) {
        Table table = new Table();

        table.pckg("packag0.package1");
        table.tableName("Person");
        Column column1 = new Column();
        Column column2 = new Column();

        column1.name("name");
        column1.name("surname");
        table.addColumn(column1);
        table.addColumn(column2);
        Referrer referrer1 = new Referrer("package0.package1", "Person");

        referrer1.addAttribute("Manager", "package0.package1.Manager", false);
        referrer1.addAttribute("Employee", "package0.package1.Employee", false);
        Referrer referrer2 = new Referrer("package0.package1", "Manager");

        referrer2.addAttribute("Employee", "package0.package1.Employee", false);
        table.addReferrer(referrer1);
        table.addReferrer(referrer2);
        System.out.println("Table: \n" + table); 
    }
}
