
/*
 * Title:        DODS Ant task
 * Description:  DODS
 */
package  org.enhydra.ant.taskdefs;

import java.io.File ;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import org.enhydra.dods.generator.DODSGenerator;

/**
 *
 * @author Sinisa Milosevic
 * Invokes DODS to generate a set of java classes from a doml file.<BR>
 * The files will only be regenerated/compiled if the date on the doml file is
 * newer than at least one of the generated files.<BR>
 * This taskdef extends Ant's <javac> task; refer to that documentation for
 * parameters that affect compilation.<BR>
 * Typically made visible to an Ant build file with the following declaration:<PRE>
 *  &lt;taskdef name="dods" classname="org.enhydra.ant.taskdefs.Dods"/&gt;</PRE>
 *
 * <B>Parameters</B><PRE>
 *
 * domlfile    - The doml input file describing data object mapping. Required = Yes.
 * outputDir   - Target for generated classes, expressed as a directory path. Required = Yes.
 * force       - Forces DODS always to regenerate source files.
 *               Possible values: ("true", "false"(default)). Required = No.
 * action      - Name of Ant task from generate.xml.
 *               If only documentation need to be generated then set action="dods:generatorOff"
 *               and set indicators for generating documentation (pdf,xmi,ptl,html).
 *               Required = No.
 * templateDir - Name of folder for template set for generating java code, expressed as a directory path. Required = No.
 * templateSet - Template set for generating java code. Required = No.
 * confDir     - Path to custom configuration folder = No.
 * database    - Sets database vendor for generating sql. Required = No.
 * html        - Indicates DODS to generate html documentation from .doml file
 *               Possible values: ("true", "false"(default)). Required = No.
 * pdf         - Indicates DODS to generate pdf documentation from .doml file
 *               Possible values: ("true", "false"(default)). Required = No.
 * xmi         - Indicates DODS to generate xmi documentation from .doml file
 *               Possible values: ("true", "false"(default)). Required = No.
 * ptl         - Indicates DODS to generate ptl (Rational Rose) documentation from .doml file.
 *               Possible values: ("true", "false"(default)). Required = No.
 * without parameters - to create all sql files and java classes and to compile it.
 *
 * action parameters:
 *      dods:build_all - to create all sql files and java classes.
 *      dods:sql - to create only sql files.
 *      dods:java -to create only java files and to compile them.
 *      dods:javaNoCompile -To create only java files and not to compile them.
 *      dods:noCompile -To create SQL files and java files and not to compile them.
 *      dods:build_all_split - to create all sql files and java classes and to compile it. SQL files will be divided into separate files using SQLSplitter
 *      dods:sqlsplit - to create only sql files and separate in different files using SQLSplitter.
 *      dods:noCompileSplit - To create SQL files and separate sql commands using SQLSplitter and java files and not to compile them.
 *
 * templateset parameters:
 *      standard         - generate standard java code.
 *      <user_defined>   - any user defined template set.
 *</PRE>
 *<BR>
 * Examples<PRE>
 * &lt;dods domlfile="${basedir}/discRack.doml"
 *        outputDir="${basedir}/src"
 *        templateset="standard"/&gt;</PRE>
 *
 */
public class Dods extends Task {
    protected String  domlfile = null;
    protected String  outputDir = null;
    protected String  action = "dods:build_all"; // action
 protected String  templateDir = null;
    protected String  confDir = null;
    protected String  templateSet = "standard";
    protected String  force = "false";
    protected String  database = DODSGenerator.DATABASE_NOT_SET;
    protected boolean html = false;
    protected boolean pdf = false;
    protected boolean xmi = false;
    protected boolean ptl = false;
    public Dods() {}

    /**
     * Simply invokes DODS Generator, and then compiles the generated files.
     *
     * @exception org.apache.tools.ant.BuildException
     */
    public void execute() throws org.apache.tools.ant.BuildException {
        if (domlfile == null) {
            throw new BuildException("doml attribute must be set!");
        }
        if (outputDir == null) {
            throw new BuildException("output directory attribute must be set!");
        }
        if (!(new File (outputDir)).isDirectory()) {
            throw new BuildException("output directory must be a valid directory!");
        }
        if (confDir != null) {
            if (!(new File (confDir)).isDirectory()) {
                throw new BuildException("configuration directory must be a valid directory!");
            }
        }
        try {
            if (System.getProperty("DODS_HOME") == null) {
                if (System.getProperty("ENHYDRA_DIR") != null) {
                    System.setProperty("DODS_HOME",
                            System.getProperty("ENHYDRA_DIR") + File.separator
                            + "dods");
                } else {
                    throw new BuildException("DODS_HOME not set");
                }
            }
            System.setProperty("JAVA_HOME", System.getProperty("JAVA_HOME"));
            System.setProperty("TEMPLATESET", templateSet);
            int exit = DODSGenerator.generateAll(outputDir, domlfile, action,
                    confDir, templateDir, templateSet, force, database, html,
                    pdf, xmi, ptl);
        } catch (Exception  e) {
            throw new BuildException(e);
        }
    }

    /**
     * Sets the doml file.
     *
     * @param domlfile the doml input file describing data object mapping.
     */
    public void setDomlfile(String  domlfile) {
        this.domlfile = domlfile;
    }

    /**
     * Sets the templateDir.
     *
     * @param tempDir Path to folder with templates.
     */
    public void setTemplateDir(String  tempDir) {
        this.templateDir = tempDir;
        if (tempDir != null) { 
            try {
                File  tempFile = new File (tempDir); 

                if (!tempFile.isDirectory()) {
                    this.templateDir = null; 
                }
            } catch (NullPointerException  e) {
                this.templateDir = null;    
            }
        }  
         
    }

    /**
     * The destination directory in which the generated sources are written to.
     *
     * @param outdir target for generated classes, expressed as a directory path.
     */
    public void setOutputDir(String  outdir) {
        outputDir = outdir;
    }

    /**
     * Sets action parameter.
     *
     * @param action value of action parameter.
     */
    public void setAction(String  action) {
        this.action = action;
    }

    /**
     * Sets template set parameter.
     *
     * @param template value of template set.
     */
    public void setTemplateSet(String  template) {
        this.templateSet = template;
    }

    /**
     * Set the database vendor for creating sql
     *
     * @param database database vendor for creating sql
     */
    public void setDatabase(String  database) {
        this.database = database;
    }

    /**
     * Forces DODS always to regenerate source files.
     *
     * @param force "true" if source files shoult be regenerated, otherwise "false".
     */
    public void setForce(String  force) {
        this.force = "false";
        if (force.equals("true")) {
            this.force = "true";
        }
    }

    /**
     * Indicates DODS to generate html documentation from .doml file.
     * @param html "true" if html documentation shoult be generated, otherwise "false".
     */
    public void setHtml(String  html) {
        this.html = false;
        if (html.equals("true")) {
            this.html = true;
        }
    }

    /**
     * Indicates DODS to generate pdf documentation from .doml file.
     * @param pdf "true" if pdf documentation shoult be generated, otherwise "false".
     */
    public void setPdf(String  pdf) {
        this.pdf = false;
        if (pdf.equals("true")) {
            this.pdf = true;
        }
    }

    /**
     * Indicates DODS to generate xmi documentation from .doml file.
     * @param xmi "true" if xmi documentation shoult be generated, otherwise "false".
     */
    public void setXmi(String  xmi) {
        this.xmi = false;
        if (xmi.equals("true")) {
            this.xmi = true;
        }
    }

    /**
     * Indicates DODS to generate ptl (Rational Rose) documentation from .doml file.
     * @param ptl "true" if ptl documentation shoult be generated, otherwise "false".
     */
    public void setPtl(String  ptl) {
        this.ptl = false;
        if (ptl.equals("true")) {
            this.ptl = true;
        }
    }

    /**
     * Indicates DODS to search for configuration files in confDir.
     * @param string configuration dir
     */
    public void setConfDir(String  string) {
        confDir = string;
    }
}
