
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: DODSTag.java,v 1.3 2005/05/27 09:50:11 predrag Exp $
 */

/*
 *   DODSTag.java
 *
 * @author    Nenad Vico 
 *
 */
package org.enhydra.dods.xslt;

import java.io.*;
import java.util.*;
import org.enhydra.dods.Common;

/**
 *
 *   Utility class for DODS template xsl transformation.
 *
 */
public class DODSTag {
    protected DODSTag() {}

    /**
     *  Generate SUPER_LOADDATA tag.
     *
     * @param className Class member name (column table name).
     *
     * @return SUPER_LOADDATA tag
     *
     * @exception Exception Null pointer exception.
     */
    static public String  superLoadData(String  className) throws Exception  {
        if (-1 != className.indexOf("GenericDO")) {
            return new String ("");
        } else {
            return new String ("super.loadData();");
        }
    }
    static final String  defaultStringLength = "32";

    /**
     *  Generate DECL_MAX_LENGTH tag.
     *  
     * @param name Column name.
     * @param size Column size.
     * @param dbtype Column database type.
     * @param type Column java type.
     *
     * @return SECL_MAX_LENGTH tag.
     *
     * @exception Exception Null pointer exception.
     */
    static public String  declMaxLength(String  name, String  size, String  dbtype, String  type) throws Exception  {
        String  valueMaxLength = "";
        String  declMaxLength = "";

        if (type.equals("String")) {
            if (dbtype.equals("LONGVARCHAR")) {
                valueMaxLength = size = "0";
            } else {
                if (null == size || 0 == size.length()) {
                    size = defaultStringLength;
                }
                valueMaxLength = name + "_MaxLength";
                declMaxLength = "static public final int " + valueMaxLength
                        + " = " + size + ";";
            }
        }
        return declMaxLength;
    }

    /**
     *  Generate MARK_NEW_VALUE_EXTRAS tag.
     *
     * @param name Column name.
     * @param dbtype Column database type.
     * @param type Column java type.
     * @param canBeeNull "true" if column can be null, otherwise "false".
     *
     * @return MARK_NEW_VALUE_EXTRAS tag.
     *
     * @exception Exception Null pointer exception.
     */
    static public String  markNewValueExtras(String  name, String  dbtype, String  type, String  canBeeNull) throws Exception  {
        String  valueMaxLength = "";
        String  markNewValue_extraParms = "";

        if (type.equals("String")) {
            if (dbtype.equals("LONGVARCHAR")) {
                valueMaxLength = "0";
            } else {
                valueMaxLength = name + "_MaxLength";
            }
            markNewValue_extraParms = ", 0, " + valueMaxLength + ", "
                    + canBeeNull;
        }
        return markNewValue_extraParms;
    }

    /**
     * Generate BIG_DECIMAL_SCALE tag.
     *
     * @param type Column java type.
     * @param size Column size.
     * @param objRef "true" if column is objectRefereence, otherwise "false".
     * @param foreignKey "true" if column is foreignKey, otherwise "false".
     *
     * @return BIG_DECIMAL_SCALE tag.
     *
     * @exception Exception Null pointer exception.
     */
    static public String  bigDecimalScale(String  type, String  size, String  objRef, String  foreignKey) throws Exception  {     
        String  bigDecimalScale = "";

        if (type.equals("java.math.BigDecimal")) {
            if (null == size || 0 == size.length()) {
                bigDecimalScale = ", 0";
            } else {
                bigDecimalScale = ", " + size;
            }
        }
        
        if (objRef.equals("true")) {
            if (foreignKey.equals("false")) {
                bigDecimalScale = ", 0";
            }
        }
        return bigDecimalScale;
    }

    /**
     * Generate ARRAY_QUERY_NAME tag.
     *
     * @param objName Object/table name.
     * @param pack Package name.
     * @param name Column name.
     * @param isArray "true" if this is array, otherwise "false".
     *
     * @return ARRAY_QUERY_NAME tag.
     *
     * @exception Exception Null pointer exception.
     */
    static public String  arrayQueryName(String  objName, String  pack, String  name, String  isArray) throws Exception  {
        String  arrayQueryName = "";

        if (isArray.equals("true")) {
            arrayQueryName = pack + "." + objName + "_" + name + "Query";
        }
        return arrayQueryName;
    }

    /**
     * Generate ARRAY_DO_NAME tag.
     *
     * @param objName Object/table name.
     * @param name Attribute name.
     * @param isArray "true" if this is array, otherwise "false".
     *
     * @return ARRAY_DO_NAME tag.
     *
     * @exception Exception Null pointer exception.
     */
    static public String  arrayDOName(String  objName, String  name, String  isArray) throws Exception  {
        String  arrayDOName = "";

        if (isArray.equals("true")) {
            arrayDOName = objName + "_" + name + "DO";
        }
        return arrayDOName;
    }
    public static Vector columnNamesVect;
    public static Vector primaryKeysVect;
    public static Vector refsN21Vect;
    public static Vector refAttrs;
 
    /**
     * Initialize column names <CODE>Vector</CODE>.
     *
     * @exception Exception Null pointer exception.
     */
    static public void initColumn() throws Exception  {
        columnNamesVect = new Vector();
        primaryKeysVect = new Vector();
        refsN21Vect = new Vector();
        fKGroupedByRefDO = new Hashtable();
    }

    /**
     *  Add column name in columnNames <CODE>Vector</CODE>.
     *
     * @param name Column name.
     * @param isPrimaryKey "true" if the column is primary key, otherwise "false".  
     * @param isObjRef "true" if the column references an table, otherwise "false". 
     * @param isUsedForQuery "true" if the column is used for query, otherwise "false". 
     * @param isForeignKey  "true" if the column is foreign key, otherwise "false".
     * @param objRefDOisAbstract "true" if the object the column references is abstract, otherwise "false".
     *
     * @exception Exception Null pointer exception. 
     */
    static public void addColumnName(String  name, String  isPrimaryKey, String  isObjRef, String  isUsedForQuery, String  isForeignKey, String  objRefDOisAbstract) throws Exception  {
        columnNamesVect.add(name);
        if (isPrimaryKey.equals("true")) {
            primaryKeysVect.add(name);
        }
    }

    /**
     * Initialize refererr <CODE>Vector</CODE>.
     *
     * @exception Exception Null pointer exception. 
     */
    static public void initRefererr() throws Exception  {
        refsN21Vect = new Vector();
    }

    /**
     * Add Refererr atribute in <CODE>Vector</CODE>.
     *
     * @param name Attribute name.
     * @param do_name Name of refererr DO.
     * @param classPackage Name of classes package.
     * @param className Name of object's class.
     *
     * @exception Exception Null pointer exception. 
     */
    static public void addRefAttr(String  name, String  do_name, String  classPackage, String  className) throws Exception  {
        if (do_name.equals((classPackage + "." + className))) {
            String [] pair = new String [2];

            pair[0] = name;
            pair[1] = do_name;
            refsN21Vect.add(pair);
        }
    }

    /**
     * Generate INSERT_COLUMN_NAMES tag.
     *
     * @return INSERT_COLUMN_NAMES tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  insertColumnNames() throws Exception  {
        String  sql_insert_column_names = "";
        String  comma = "";
        int i;

        for (i = 0; i < columnNamesVect.size(); i++) {
            sql_insert_column_names += comma + columnNamesVect.elementAt(i);
            comma = ", ";
        }
        if (0 == primaryKeysVect.size()) {
            sql_insert_column_names += ", \" + get_OIdColumnName() + \", \" + get_versionColumnName() + \"";
        }
        return sql_insert_column_names;
    }

    /**
     * Generate INSERT_QUESTION_MARKS tag.
     *
     * @return INSERT_QUESTION_MARKS tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  insertQuestionMarks() throws Exception  {
        String  sql_insert_question_marks = "";
        String  comma = "";

        if (0 == primaryKeysVect.size()) {
            sql_insert_question_marks = "?, ?, ";
        }  // oid and version
 int i;

        for (i = 0; i < columnNamesVect.size(); i++) {
            sql_insert_question_marks += comma + "?";
            comma = ", ";
        }
        return sql_insert_question_marks;
    }

    /**
     * Generate UPDATE_WHERE_CLAUSE tag.
     *
     * @return UPDATE_WHERE_CLAUSE tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  updateWhereClause() throws Exception  {
        String  updateWhereClauseString = "\" + get_OIdColumnName() + \" = ? and \" + get_versionColumnName() + \" = ?";

        /*
         String updateWhereClauseString   = "";
         String commaPK = "";
         int i;
         for(i=0; i< primaryKeysVect.size(); i++) {
         updateWhereClauseString += commaPK + " " + primaryKeysVect.elementAt(i) + " = ?";
         commaPK = ", ";
         }
         */
        return updateWhereClauseString;
    }

    /**
     * Generate UPDATE_QUAL_QUESTION_MARKS tag.
     *
     * @return UPDATE_QUAL_QUESTION_MARKS tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  updateQualQuestionMarks() throws Exception  {
        String  sql_update_question_marks = "";
        String  comma = "";
        int i;

        for (i = 0; i < columnNamesVect.size(); i++) {
            sql_update_question_marks += comma + columnNamesVect.elementAt(i)
                    + " = ?";
            comma = ", ";
        }
        return sql_update_question_marks;
    }

    /**
     * Generate MEMBER_REMARKS tag.
     *
     * @param isForeignKey "true" if column is foreign key, otherwise "false".
     * @param isPrimaryKey "true" if column is primary key, otherwise "false".
     *
     * @return MEMBER_REMARKS tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  memberRemarks(String  isForeignKey, String  isPrimaryKey) throws Exception  {
        String  memberRemarks = "";

        if (isForeignKey.equals("true")) {
            memberRemarks += "(FK)";
        }
        if (isPrimaryKey.equals("true")) {
            memberRemarks += "(PK)";
        }
        return memberRemarks;
    }

    /**
     * Generate DUMP_METHOD tag.
     *
     * @param name Column name.
     * @param isObjRef "true" if column is Object Reference, otherwise "false".
     * @param isForeignKey "true" if column is foreign key, otherwise "false".
     *
     * @return DUMP_METHOD tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  dumpMethod(String  name, String  isObjRef, String  isForeignKey) throws Exception  {
        String  dumpMethod = "";

        if (isObjRef.equals("true") && isForeignKey.equals("false")) {
            dumpMethod = "( null == data." + name + " ? null " + " : data."
                    + name + ".toString( indentCount + 1 ) )";
        } else {
            dumpMethod = "data." + name;
        }
        return dumpMethod;
    }

    /**
     * Generate DUMP_METHOD1 tag.
     * 
     * changed a bit for "double_trouble" template ;)
     *
     * @param name Column name.
     * @param isObjRef "true" if column is Object Reference, otherwise "false".
     * @param isForeignKey "true" if column is foreign key, otherwise "false".
     *
     * @return DUMP_METHOD tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  dumpMethod1(String  name, String  isObjRef, String  isForeignKey) throws Exception  {
        String  dumpMethod = "";

        name = name.toUpperCase(java.util.Locale.ENGLISH).substring(0, 1).concat(name.substring(1));
        if (isObjRef.equals("true") && isForeignKey.equals("false")) {
            dumpMethod = " ( null == get_DataStruct().get" + name + "() ? null "
                    + " : get_DataStruct().get" + name + "().get_OId())";
        } else {
            dumpMethod = "get_DataStruct().get" + name + "()";
        }
        return dumpMethod;
    }

    /**
     * Generate GET_ARRAY_DO_METHOD_NAME tag.
     *
     * @param DOName Refererr DO name.
     * @param attrName Refererr attribute name.
     *
     * @return GET_ARRAY_DO_METHOD_NAME tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  getArrayDO(String  DOName, String  attrName) throws Exception  {
        String  get_array_DO = "get" + DOName + "DOArray";

        if (1 < refsN21Vect.size()) {
            get_array_DO += "_" + attrName;
        }
        return get_array_DO;
    }

    /**
     * Get DO name from full DO name (name with package). 
     *
     * @param fullname Name of DO with package.
     *
     * @return DO name from full DO name (DO name without package).
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  getDoName(String  fullname) throws Exception  {
        String  doName = fullname.substring(fullname.lastIndexOf('.') + 1);

        return doName;
    }

    /**
     * Generate GET_ARRAY_DO_METHOD_NAME tag for N2N.
     *
     * @param name2 Name of refererr DO. 
     * @param name Column name.
     *
     * @return GET_ARRAY_DO_METHOD_NAME tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  getArrayDON2N(String  name2, String  name) throws Exception  {
        String  get_array_DO = "get" + name2 + "DOArray_via_" + name;

        return get_array_DO;
    }

    /**
     * Generate GET_SINGLE_DO_METHOD_NAME tag.
     *
     * @param DOName Name of refererr DO.
     * @param attrName Name of Refererr attribute.
     *
     * @return GET_ARRAY_DO_METHOD_NAME tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  getSingleDO(String  DOName, String  attrName) throws Exception  {
        String  get_single_DO = "get" + DOName + "DO";

        if (1 < refsN21Vect.size()) {
            get_single_DO += "_" + attrName;
        }
        return get_single_DO;
    }

    /**
     * Generate ADD_SINGLE_DO_METHOD_NAME tag.
     *
     * @param DOName Name of refererr DO.
     * @param attrName Name of Refererr attribute.
     *
     * @return ADD_ARRAY_DO_METHOD_NAME tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  addSingleDO(String  DOName, String  attrName) throws Exception  {
        String  add_single_DO = "add" + DOName + "DO";

        if (1 < refsN21Vect.size()) {
            add_single_DO += "_" + attrName;
        }
        return add_single_DO;
    }

    /**
     * Generate REMOVE_DO_METHOD_NAME tag.
     *
     * @param DOName Name of refererr DO.
     * @param attrName Name of Refererr attribute.
     *
     * @return REMOVE_DO_METHOD_NAME tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  removeSingleDO(String  DOName, String  attrName) throws Exception  {
        String  remove_single_DO = "remove" + DOName + "DO";

        if (1 < refsN21Vect.size()) {
            remove_single_DO += "_" + attrName;
        }
        return remove_single_DO;
    }

    /**
     * Compare referrer DO name.
     *
     * @param classPackage Class package of referrer table.
     * @param className Class name of referrer table.
     * @param DOName Name of refererr DO.
     *
     * @return "true" if it is equal, otherwise "false".
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  compareReferrerDoName(String  classPackage, String  className, String  DOName) throws Exception  {
        if ((classPackage + "." + className).equals(DOName)) {
            return "true";
        } else { 
            return "false";
        }
    }
    static public Hashtable fKGroupedByRefDO;

    /**
     * Generate Hashtable of foreign keys, grouped by Referenced DO.
     *
     * @param name Column name.
     * @param isRefConstraint "true" if column has referece constraint, otherwise "false".
     * @param refName Name of Referenced DO. 
     * @param pack Referenced DO package.
     * @param fKeyColName Foreign key column name.
     * @param foreignKeyGroup Foreign key group.
     *
     * @exception Exception Null pointer exception. 
     */
    static public void addFK(String  name, String  isRefConstraint, String  refName, String  pack, String  fKeyColName, String  foreignKeyGroup) throws Exception  {
        if (isRefConstraint.equals("true")) {
            String  nameOfPKinReferencedTable = fKeyColName;

            if (null != nameOfPKinReferencedTable
                    && 0 != nameOfPKinReferencedTable.length()) {
                // a is a foreign key Attribute
                Hashtable groups = (Hashtable) fKGroupedByRefDO.get(refName);

                if (null == groups) {
                    groups = new Hashtable();
                    fKGroupedByRefDO.put(refName, groups);
                }
                Vector pairs = (Vector) groups.get(foreignKeyGroup);

                if (null == pairs) {
                    pairs = new Vector();
                    groups.put(foreignKeyGroup, pairs);
                }
                String [] pk_fk_pair = new String [ 2 ];

                pk_fk_pair[0] = nameOfPKinReferencedTable;
                pk_fk_pair[1] = name; // name of FK column
                pairs.addElement(pk_fk_pair);
            }
        }
    }
    static public Enumeration enumFKGroupedByRefDOKeys;
    static public Enumeration enumFKGroupedByRefDOElements;
    static public FKElements elements;
    static class FKElements {
        public String  fkRefName;
        public String  getMethodName;
        public String  setMethodName;
        public StringWriter fKsetQueryCalls;
        public StringWriter fKsetCalls;
        public FKElements() {
            fkRefName = "";
            getMethodName = "";
            setMethodName = "";
            fKsetQueryCalls = new StringWriter();
            fKsetCalls = new StringWriter();
        }
    }

    /**
     * Initialize enumeration of foreign keys grouped by Referenced DO.
     *
     * @exception Exception Null pointer exception. 
     */
    static public void initEnumFKGroupedByRefDO() throws Exception  {
        enumFKGroupedByRefDOKeys = fKGroupedByRefDO.keys();
        enumFKGroupedByRefDOElements = fKGroupedByRefDO.elements();
        elements = new FKElements();
    }

    /**
     * Test if enumeration of foreign keys has more elements.
     *
     * @return String "true" if enumeration of foreign keys 
     * has more elements, otherwise "false".
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  hasMoreElements() throws Exception  {
        String  ret = "false";

        if (enumFKGroupedByRefDOKeys.hasMoreElements()) {
            ret = "true";
            enumFKGroupedByRefDOElements.hasMoreElements();
            elements.fkRefName = (String ) enumFKGroupedByRefDOKeys.nextElement();
            Hashtable fkGroups = (Hashtable) enumFKGroupedByRefDOElements.nextElement();

            for (Enumeration e2 = fkGroups.keys(); e2.hasMoreElements();) {
                String  fkGroup = (String ) e2.nextElement();

                elements.getMethodName = "_" + fkGroup;
                elements.setMethodName = "_" + fkGroup;
                Vector v = (Vector) fkGroups.get(fkGroup);

                for (int i = 0; i < v.size(); i++) {
                    String [] pk_fk_pair = (String []) v.elementAt(i);
                    String  pkName = Common.capitalizeName(pk_fk_pair[0]);
                    String  fkName = Common.capitalizeName(pk_fk_pair[1]);

                    /*
                     // prepare data for setting START_MEMBER_NOT_IN_COMBO_FK tag
                     numColumnsInFK.put( fkName, new Integer( v.size() ) );
                     fkNameByFkRef.put( fkName, fkRef );
                     */
                    // set tags for generating method contents
                    elements.fKsetQueryCalls.write("        q.setQuery" + fkName
                            + "( get" + pkName + "() );\n");
                    elements.fKsetCalls.write("     _set" + fkName + "( x.get"
                            + pkName + "() );\n");
                } // for
            } // for
        } // if
 return ret;
    }

    /**
     * Get FK_REF tag.
     *
     * @return FK_REF tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  getFKRef() throws Exception  {
        return elements.fkRefName;
    }

    /**
     * Get FK_REF_GET_METHOD tag.
     *
     * @return FK_REF_GET_METHOD tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  getMethodName() throws Exception  {
        return elements.getMethodName;
    }

    /**
     * Get FK_REF_SET_METHOD tag.
     *
     * @return FK_REF_SET_METHOD tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  setMethodName() throws Exception  {
        return elements.setMethodName;
    }

    /**
     * Get FK_SETQUERY_CALLS tag.
     *
     * @return FK_SETQUERY_CALLS tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  getFKsetQueryCalls() throws Exception  {
        return elements.fKsetQueryCalls.toString();
    }

    /**
     * Get FK_SET_CALLS tag.
     *
     * @return FK_SET_CALLS tag.
     *
     * @exception Exception Null pointer exception. 
     */
    static public String  getFKsetCalls() throws Exception  {
        return elements.fKsetCalls.toString();
    }
    static public int counter = 0;

    /**
     * Return counter and increment them.
     *
     * @return current counter.
     */
    static public void resetCounter() {
        counter = 0;
    }

    /**
     * Increment counter.
     */
    static public void incCounter() {
        counter++;
    }

    /**
     * Return counter.
     *
     * @return current counter.
     */
    static public String  getCounter() {
        return "" + counter;
    }
    
        /**
     * Return counter.
     *
     * @return current counter.
     */
    static public String  convertJavaTypes(String  typ ) {

      String  retValue;
      
      if(typ.equals("boolean")) 
            retValue = "new Boolean(x)";
      else if(typ.equals("int")) 
            retValue = "new Integer(x)";
      else if(typ.equals("long")) 
            retValue = "new Long(x)";
      else if(typ.equals("float")) 
            retValue = "new Float(x)";
      else if(typ.equals("double")) 
            retValue = "new Double(x)";
      else if(typ.equals("byte")) 
            retValue = "new Byte(x)";
      else if(typ.equals("short")) 
            retValue = "new Short(x)";
      else if(typ.equals("char")) 
            retValue = "new Char(x)";
      else
            retValue = "x";    

        return retValue;
    }
}
