
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: XSLTUtil.java,v 1.32.6.1.2.1 2002/12/11 
 */

/*
 *
 * @author    Nenad Vico
 * @version   $Revision: 1.1 $
 *
 */
package org.enhydra.dods.xml;

import java.io.File ;
import java.io.FileOutputStream ;
import java.io.OutputStream ;
import java.util.Properties ;
import javax.xml.parsers.DocumentBuilder ;
import javax.xml.parsers.DocumentBuilderFactory ;
import javax.xml.transform.Transformer ;
import javax.xml.transform.TransformerFactory ;
import javax.xml.transform.dom.DOMSource ;
import javax.xml.transform.stream.StreamResult ;
import org.apache.xalan.processor.TransformerFactoryImpl;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.NodeList ;

public class XmlUtil {
    
    protected String  xmlFile;
    protected Document  document;    
    public XmlUtil(String  xmlFile) throws XmlUtilException {
        this.xmlFile = xmlFile;
        try { // read DOM parser
 DocumentBuilderFactory  dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder  db = dbf.newDocumentBuilder();

            document = db.parse(xmlFile);
        } catch (Exception  e) {
            e.printStackTrace();
        }
    }

    /**
     * Store DOM source in xml file.
     */
    public void store() throws XmlUtilException {
        store(null);
    }

    /**
     * Store DOM source in xml file.
     *
     * @param properties Output properties for xml file. 
     */
    public void store(Properties  properties) throws XmlUtilException {
        store(null, properties);
    }

    /**
     * Store DOM source in xml file.
     *
     * @param properties Output properties for xml file. 
     * @param fileName Output xml file name. 
     */
    public void store(String  fileName, Properties  properties) throws XmlUtilException {
        TransformerFactory  tf = null;
        DOMSource  src = null;

        if (fileName == null) {
            fileName = xmlFile;
        }
        tf = TransformerFactoryImpl.newInstance();
        src = new DOMSource (document.getDocumentElement());
        if (tf == null) {
            return;
        }
        if (src == null) {
            return;
        }
        OutputStream  outputs = null;

        try {
            File  f = new File (fileName);
            File  pf = f.getParentFile();

            if (pf != null) {
                pf.mkdirs();
            }
            outputs = new FileOutputStream (f.getPath());
            Transformer  serializer = tf.newTransformer();

            if (properties != null) {
                serializer.setOutputProperties(properties);
            }
            serializer.transform(src, new StreamResult (outputs));
        } catch (java.io.FileNotFoundException  e) {
            throw new XmlUtilException(e);
        } catch (javax.xml.transform.TransformerConfigurationException  e) {
            throw new XmlUtilException(e);
        } catch (javax.xml.transform.TransformerException  e) {
            throw new XmlUtilException(e);
        }
    }

    /**
     * Return value of single text node defined by key parameter.
     *
     * @param key Full name of tag which is about to be read. The tags in the key are separated by "/" 
     * (e.g. /doml/database).
     * 
     * @return Value of single text node defined by key parameter or null if tag doesn't exist.
     */
    public String  getSingleTextNode(String  key) throws XmlUtilException {
        return getSingleTextNode(key, null);
    }

    /**
     * Return value of single text node defined by key parameter.
     *
     * @param key Full name of tag which is about to be read. The tags in the key are separated by "/" 
     * (e.g. /doml/database).
     * @param levelNodes Array of ordinal numbers. There is one number for each tag level
     * (starting from level 0). These numbers tell to which appearance of specific 
     * tag key tag belongs to. It is not necessary to provide all tag levels.
     * 
     * @return Value of single text node defined by key parameter or null if tag doesn't exist.
     */
    public String  getSingleTextNode(String  key, int levelNodes[]) throws XmlUtilException {
        String  nodeValue = "";
        NodeList  tagElems;
        Element  tagElem;
        String  tag;
        String  subKey;
        int cuurentLevel = 0;
        int pos = key.indexOf("/");

        if (pos < 0) {
            return null;
        }
        if (pos == 0) { // cut leading separator "/"
            subKey = key.substring(pos + 1);
            pos = subKey.indexOf("/");
        } else {
            subKey = key;
        }
        try {
            if (pos != -1) {
                tag = subKey.substring(0, pos);
                subKey = subKey.substring(pos + 1);
                tagElems = document.getElementsByTagName(tag);
                tagElem = (Element ) tagElems.item(0);
                cuurentLevel++;
                pos = subKey.indexOf("/");
                if (pos != -1) {
                    while (pos != -1) {
                        tag = subKey.substring(0, pos);
                        subKey = subKey.substring(pos + 1);
                        tagElems = tagElem.getElementsByTagName(tag);
                        if (levelNodes != null
                                && cuurentLevel < levelNodes.length) {
                            tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                        } else {
                            tagElem = (Element ) tagElems.item(0);
                        }
                        if (tagElem == null) {
                            return null;
                        }
                        cuurentLevel++;
                        pos = subKey.indexOf("/");
                    }
                }
                tagElems = tagElem.getElementsByTagName(subKey);
                if (levelNodes != null && cuurentLevel < levelNodes.length) {
                    tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                } else {
                    tagElem = (Element ) tagElems.item(0);
                }
                tagElems = tagElem.getChildNodes();
                String  text = "";

                for (int i = 0; i < tagElems.getLength(); i++) {
                    if (tagElems.item(i).getNodeType() == 3) { // read only text values 
                        text += tagElems.item(i).getNodeValue();
                    } 
                }
                return text;
            } else {
                tagElems = document.getElementsByTagName(subKey);
                if (levelNodes != null && cuurentLevel < levelNodes.length) {
                    tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                } else {
                    tagElem = (Element ) tagElems.item(0);
                }
                tagElems = tagElem.getChildNodes();
                String  text = "";

                for (int i = 0; i < tagElems.getLength(); i++) {
                    if (tagElems.item(i).getNodeType() == 3) { // read only text values 
                        text += tagElems.item(i).getNodeValue();
                    } 
                }
                return text;
            }
        } catch (NullPointerException  ex) {
            throw new XmlUtilException(ex);
        }
    }

    /**
     * Return value of single text node defined by key parameter.
     *
     * @param key Full name of tag attribute which is about to be read. The tags in the key 
     * are separated by "/". Name of the attribute is separated by "@" from tag name
     * (e.g. /doml/database@templateset).
     * 
     * @return Value of node attribute defined by key parameter or null if attribute doesn't exist.
     */
    public String  getSingleAttribute(String  key) throws XmlUtilException {
        return getSingleAttribute(key, null);
    }

    /**
     * Return value of attribute node defined by key parameter.
     *
     * @param key Full name of tag attribute which is about to be read. The tags in the key 
     * are separated by "/". Name of the attribute is separated by "@" from tag name
     * (e.g. /doml/database@templateset).
     * @param levelNodes Array of ordinal numbers. There is one number for each tag level
     * (starting from level 0). These numbers tell to which appearance of specific 
     * tag key tag belongs to. It is not necessary to provide all tag levels.
     * 
     * @return Value of node attribute defined by key parameter or null if attribute doesn't exist.
     */
    public String  getSingleAttribute(String  key, int levelNodes[]) throws XmlUtilException {
        String  nodeValue = "";
        NodeList  tagElems = null;
        Element  tagElem = null;
        String  tag;
        String  subKey;
        String  attrValue = null;
        int cuurentLevel = 0;
    
        int pos = key.indexOf("/");

        if (pos < 0) {  
            return null;
        } 
        try {
            if (pos == 0) { // cut leading separator "/"
                subKey = key.substring(pos + 1);
                pos = subKey.indexOf("/");
            } else {
                subKey = key;
            }
            if (pos != -1) {
                tag = subKey.substring(0, pos);
                subKey = subKey.substring(pos + 1);
                tagElems = document.getElementsByTagName(tag);
                tagElem = (Element ) tagElems.item(0);
                cuurentLevel++;
                pos = subKey.indexOf("/");
                if (pos != -1) {
                    while (pos != -1) {
                        tag = subKey.substring(0, pos);
                        subKey = subKey.substring(pos + 1);
                        tagElems = tagElem.getElementsByTagName(tag);
                        if (levelNodes != null
                                && cuurentLevel < levelNodes.length) {
                            tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                        } else {
                            tagElem = (Element ) tagElems.item(0);
                        }
                        if (tagElem == null) {
                            return null;
                        }
                        cuurentLevel++;
                        pos = subKey.indexOf("/");
                    }
                }
                pos = subKey.indexOf("@");
                tag = subKey.substring(0, pos);
                if (pos != -1) {
                    tagElems = tagElem.getElementsByTagName(tag);
                    if (levelNodes != null && cuurentLevel < levelNodes.length) {
                        tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                    } else {
                        tagElem = (Element ) tagElems.item(0);
                    }
                    String  attrName = subKey.substring(pos + 1);

                    attrValue = tagElem.getAttribute(attrName);
                }
            } else {
                pos = subKey.indexOf("@");
                tag = subKey.substring(0, pos);
                if (pos != -1) {
                    tagElems = document.getElementsByTagName(tag);
                    if (levelNodes != null && cuurentLevel < levelNodes.length) {
                        tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                    } else {
                        tagElem = (Element ) tagElems.item(0);
                    }
                    String  attrName = subKey.substring(pos + 1);

                    attrValue = tagElem.getAttribute(attrName);
                }
            }
        } catch (NullPointerException  ex) {
            throw new XmlUtilException(ex);
        }
        if (attrValue.equals("")) {
            attrValue = null;
        }
        return attrValue;
    }

    /**
     * Set value of attribute node defined by key parameter.
     *
     * @param key Full name of tag attribute which is about to be read. The tags in the key 
     * are separated by "/". Name of the attribute is separated by "@" from tag name
     * (e.g. /doml/database@templateset).
     * @param levelNodes Array of ordinal numbers. There is one number for each tag level
     * (starting from level 0). These numbers tell to which appearance of specific 
     * tag key tag belongs to. It is not necessary to provide all tag levels.
     * @param value New value of attribute node.
     * 
     */
    public void setSingleAttribute(String  key, int levelNodes[], String  value) throws XmlUtilException {
        String  nodeValue = "";
        NodeList  tagElems = null;
        Element  tagElem = null;
        String  tag;
        String  subKey;
        int cuurentLevel = 0;
    
        int pos = key.indexOf("/");

        if (pos < 0) {  
            return;
        } 
        try {
            if (pos == 0) { // cut leading separator "/"
                subKey = key.substring(pos + 1);
                pos = subKey.indexOf("/");
            } else {
                subKey = key;
            }
            if (pos != -1) {
                tag = subKey.substring(0, pos);
                subKey = subKey.substring(pos + 1);
                tagElems = document.getElementsByTagName(tag);
                tagElem = (Element ) tagElems.item(0);
                cuurentLevel++;
                pos = subKey.indexOf("/");
                if (pos != -1) {
                    while (pos != -1) {
                        tag = subKey.substring(0, pos);
                        subKey = subKey.substring(pos + 1);
                        tagElems = tagElem.getElementsByTagName(tag);
                        if (levelNodes != null
                                && cuurentLevel < levelNodes.length) {
                            tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                        } else {
                            tagElem = (Element ) tagElems.item(0);
                        }
                        if (tagElem == null) {
                            return;
                        }
                        cuurentLevel++;
                        pos = subKey.indexOf("/");
                    }
                }
                pos = subKey.indexOf("@");
                tag = subKey.substring(0, pos);
                if (pos != -1) {
                    tagElems = tagElem.getElementsByTagName(tag);
                    if (levelNodes != null && cuurentLevel < levelNodes.length) {
                        tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                    } else {
                        tagElem = (Element ) tagElems.item(0);
                    }
                    String  attrName = subKey.substring(pos + 1);

                    tagElem.setAttribute(attrName, value);
                }
            } else {
                pos = subKey.indexOf("@");
                tag = subKey.substring(0, pos);
                if (pos != -1) {
                    tagElems = document.getElementsByTagName(tag);
                    if (levelNodes != null && cuurentLevel < levelNodes.length) {
                        tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                    } else {
                        tagElem = (Element ) tagElems.item(0);
                    }
                    String  attrName = subKey.substring(pos + 1);

                    tagElem.setAttribute(attrName, value);
                }
            }
        } catch (NullPointerException  ex) {
            throw new XmlUtilException(ex);
        }
    }

    /**
     * Return number of nodes defined by key parameter.
     *
     * @param key Full name of tag which is about to be read. The tags in the key are separated by "/" 
     * (e.g. /doml/database).
     * 
     * @return Number of nodes defined by key parameter.
     *
     */
    public int getNumberOfNodes(String  key) throws XmlUtilException {
        return getNumberOfNodes(key, null);
    }

    /**
     * Return number of nodes defined by key parameter.
     *
     * @param key Full name of tag which is about to be read. The tags in the key are separated by "/" 
     * (e.g. /doml/database).
     * @param levelNodes Array of ordinal numbers. There is one number for each tag level
     * (starting from level 0). These numbers tell to which appearance of specific 
     * tag key tag belongs to. It is not necessary to provide all tag levels.
     * 
     * @return Number of nodes defined by key parameter.
     *
     */
    public int getNumberOfNodes(String  key, int levelNodes[]) throws XmlUtilException {
        String  nodeValue = "";
        NodeList  tagElems;
        Element  tagElem;
        String  tag;
        String  subKey;
        int cuurentLevel = 0;
        int pos = key.indexOf("/");

        if (pos < 0) {
            return 0;
        }
        if (pos == 0) { // cut leading separator "/"
            subKey = key.substring(pos + 1);
            pos = subKey.indexOf("/");
        } else {
            subKey = key;
        }
        try {
            if (pos != -1) {
                tag = subKey.substring(0, pos);
                subKey = subKey.substring(pos + 1);
                tagElems = document.getElementsByTagName(tag);
                tagElem = (Element ) tagElems.item(0);
                cuurentLevel++;
                pos = subKey.indexOf("/");
                if (pos != -1) {
                    while (pos != -1) {
                        tag = subKey.substring(0, pos);
                        subKey = subKey.substring(pos + 1);
                        tagElems = tagElem.getElementsByTagName(tag);
                        if (levelNodes != null
                                && cuurentLevel < levelNodes.length) {
                            tagElem = (Element ) tagElems.item(levelNodes[cuurentLevel]);
                        } else {
                            tagElem = (Element ) tagElems.item(0);
                        }
                        if (tagElem == null) {
                            return 0;
                        }
                        cuurentLevel++;
                        pos = subKey.indexOf("/");
                    }
                }
                tagElems = tagElem.getElementsByTagName(subKey);
                return tagElems.getLength();
            } else {
                tagElems = document.getElementsByTagName(subKey);
                return tagElems.getLength();
            }
        } catch (NullPointerException  ex) {
            throw new XmlUtilException(ex);
        }
    }

    /**
     * Return value of single text node defined by key parameter.
     *
     * @param xmlfile full path name of xml file which is about to be read.
     * @param key Full name of tag which is about to be read. The tags in the key are separated by "/" 
     * (e.g. /doml/database).
     * @param levelNodes Array of ordinal numbers. There is one number for each tag level
     * (starting from level 0). These numbers tell to which appearance of specific 
     * tag key tag belongs to. It is not necessary to provide all tag levels.
     * 
     * @return Value of single text node defined by key parameter or null if attribute doesn't exist.
     *
     * Example:  
     *
     *  In faculty.xml xmlfile:
     *  <faculty>
     *    <department name="electronics>
     *      <student>
     *        <name>Marko</name>
     *      </student>
     *      <student>
     *        <name>Ivana</name>
     *      </student>
     *    </department>
     *    <department name="computers">
     *      <student>
     *        <name>Zeljko</name>
     *      </student>
     *      <student>
     *        <name>Jelena</name>
     *      </student>
     *      <student>
     *        <name>Boris</name>
     *      </student>
     *    </department>
     *  </faculty>
     *
     *  int levelNodes[] = {0,1,2};
     *  getSingleTextNode("faculty.xml","/faculty/department/student/name",levelNodes)  will produce:  Boris
     * 
     *  The same result will produce: int levelNodes[] = {0,1,2,0};
     */
    static public String  getSingleTextNode(String  xmlfile, String  key, int levelNodes[]) throws XmlUtilException {
        XmlUtil xml = new XmlUtil(xmlfile);

        return xml.getSingleTextNode(key, levelNodes);
    }

    /**
     * Return value of node attribute defined by key parameter.
     *
     * @param xmlfile full path name of xml file which is about to be read.
     * @param key Full name of tag which is about to be read. The tags in the key are separated by "/" 
     * (e.g. /doml/database).
     * @param levelNodes Array of ordinal numbers. There is one number for each tag level
     * (starting from level 0). These numbers tell to which appearance of specific 
     * tag key tag belongs to. It is not necessary to provide all tag levels.
     * 
     * @return Value of node attribute defined by key parameter or null if attribute doesn't exist.
     *
     * Example:  
     *
     *  In faculty.xml xmlfile:
     *  <faculty>
     *    <department name="electronics>
     *      <student>
     *        <name>Marko</name>
     *      </student>
     *      <student>
     *        <name>Ivana</name>
     *      </student>
     *    </department>
     *    <department name="computers">
     *      <student index="6403">
     *        <name>Zeljko</name>
     *      </student>
     *      <student>
     *        <name>Jelena</name>
     *      </student>
     *      <student>
     *        <name>Boris</name>
     *      </student>
     *    </department>
     *  </faculty>
     *
     *  int levelNodes[] = {0,1,0};
     *  getSingleTextNode("faculty.xml","/faculty/department/student@index",levelNodes)  will produce:  6403
     */
    static public String  getSingleAttribute(String  xmlfile, String  key, int levelNodes[]) throws XmlUtilException {
        XmlUtil xml = new XmlUtil(xmlfile);

        return xml.getSingleAttribute(key, levelNodes);
    }

    /**
     * Set and save value of attribute node defined by key parameter.
     *
     * @param xmlfile full path name of xml file which is about to be read.
     * @param key Full name of tag attribute which is about to be read. The tags in the key 
     * are separated by "/". Name of the attribute is separated by "@" from tag name
     * (e.g. /doml/database@templateset).
     * @param levelNodes Array of ordinal numbers. There is one number for each tag level
     * (starting from level 0). These numbers tell to which appearance of specific 
     * tag key tag belongs to. It is not necessary to provide all tag levels.
     * @param value New value of attribute node.
     * @param prop Output properties for xml file.
     */
    static public void setSingleAttribute(String  xmlfile, String  key, int levelNodes[], String  value, Properties  prop)
        throws XmlUtilException {
        XmlUtil xml = new XmlUtil(xmlfile);

        xml.setSingleAttribute(key, levelNodes, value);
        xml.store(prop);        
    }

    /**
     * Return number of nodes defined by key parameter.
     *
     * @param xmlfile full path name of xml file which is about to be read.
     * @param key Full name of tag which is about to be read. The tags in the key are separated by "/" 
     * (e.g. /doml/database).
     * @param levelNodes Array of ordinal numbers. There is one number for each tag level
     * (starting from level 0). These numbers tell to which appearance of specific 
     * tag key tag belongs to. It is not necessary to provide all tag levels.
     * 
     * @return Number of nodes defined by key parameter.
     *
     * Example:  
     *
     *  In faculty.xml xmlfile:
     *  <faculty>
     *    <department name="electronics>
     *      <student>
     *        <name>Marko</name>
     *      </student>
     *      <student>
     *        <name>Ivana</name>
     *      </student>
     *    </department>
     *    <department name="computers">
     *      <student>
     *        <name>Zeljko</name>
     *      </student>
     *      <student>
     *        <name>Jelena</name>
     *      </student>
     *      <student>
     *        <name>Boris</name>
     *      </student>
     *    </department>
     *  </faculty>
     *
     *  int levelNodes[] = {0,1};
     *  getSingleTextNode("faculty.xml","/faculty/department/student",levelNodes)  will produce:  3
     * 
     */
    static public int getNumberOfNodes(String  xmlfile, String  key, int levelNodes[])
        throws XmlUtilException {
        XmlUtil xml = new XmlUtil(xmlfile);

        return xml.getNumberOfNodes(key, levelNodes);
    }

    /*
     * Test.
     */
    public static void main(String [] args) {
        int levelNodes[] = {0, 0, 1};
        String  key = "/databaseManager/database/objectId@name";
        String  value = "webdocwf";

        try {
            XmlUtil xml = new XmlUtil("discRackDods.xml");

            xml.setSingleAttribute(key, levelNodes, value);
            System.out.println(key + " = "
                    + xml.getSingleAttribute(key, levelNodes));
            xml.store();
        } catch (XmlUtilException e) {
            e.printStackTrace();
        }
    }
}
