
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: Referrer.java,v 1.1 2004/09/03 13:43:13 sinisa Exp $
 */

/*
 *
 * @author    Nenad Vico
 * @since     LBS1.8
 * @version   $Revision: 1.1 $
 *
 */
package org.enhydra.dods.trans;

import java.util.ArrayList ;

/**
 * This class represents referrer object. When one table references another table, information about referrer table 
 * is saved in this class.
 */
public class Referrer {
    
    /**
     * List of referrer attributes.
     */
    protected ArrayList  refAttr = new ArrayList (); 

    /**
     * referrer name
     */
    protected String  name = null;

    /**
     * referrer package
     */
    protected String  pckg = null;

    /**
     * referrer security
     */
    protected boolean do_secure = DefaultTagValues.DO_SECURITY; 

    /**
     * constructor (String, String)
     *
     * @param pckg Referrer package. 
     * @param name Referrer package. 
     */
    public Referrer(String  pckg, String  name) {
        this.name = name;
        this.pckg = pckg;
    }

    /**
     * Add referrer attribute.
     *
     * @param name Name of attribute. 
     * @param do_name Name of DO class which references current table. 
     */
    public void addAttribute(String  name, String  do_name, boolean isSecure) {
        refAttr.add(new RefAttr(name, do_name, isSecure));
    }

    /**
     * Add referrer attribute.
     *
     * @param attr Referrer attribute. 
     */
    public void addAttribute(RefAttr attr) {
        refAttr.add(attr);
    }

    /**
     * Get number of referrer attributes.
     *
     * @return Number of referrer attributes. 
     */
    public int size() {
        return refAttr.size();  
    }

    /**
     * Get referrer name.
     *
     * @return Referrer name.
     */
    public String  name() {
        return name;
    }

    /**
     * Set referrer security.
     *
     * @param is_secure true/false
     */
    public void secure(boolean is_secure) {
        do_secure = is_secure;
    }
    
    /**
     * Get referrer secyrity.
     *
     * @return if Referrer is security based.
     */
    public boolean isSecure() {
        return do_secure;
    }
    
    /**
     * Get referrer package.
     *
     * @return Referrer package.
     */
    public String  pckg() {
        return pckg;
    }

    /**
     * Get referrer attributes.
     *
     * @return Referrer attributes.
     */
    public ArrayList  referrerAttributes() {
        return refAttr;
    }

    /**
     * Get referrer attribute with appropriate index.
     *
     * @param index Index of referrer attribute.
     * 
     * @return Attribute with appropriate index.
     */
    public RefAttr attribute(int index) {
        return (RefAttr) refAttr.get(index);
    }

    /**
     * Get referrer attribute name with appropriate index.
     *
     * @param index Index of referrer attribute.
     * 
     * @return Attribute name with appropriate index.
     */
    public String  attributeName(int index) {
        return ((RefAttr) refAttr.get(index)).name();
    }

    /**
     * Get referrer attribute DO name with appropriate index.
     * 
     * @param index Index of referrer attribute.
     * 
     * @return Attribute DO name with appropriate index.
     */
    public String  attributeDoName(int index) {
        return ((RefAttr) refAttr.get(index)).doName();
    }

    /**
     * Get referrer generateSecure parameter with appropriate index.
     *
     * @param index Index of referrer attribute.
     * 
     * @return Is column secure based.
     */
    public boolean attributeSecurity(int index) {
        return ((RefAttr) refAttr.get(index)).isSecure();
    }

    /**
     * Get String representation.
     *
     * @return String representation.
     */
    public String  toString() {
        StringBuffer  ret = new StringBuffer ("name=").append(name).append(" package=").append(pckg).append(" attributes: ");

        for (int i = 0; i < refAttr.size(); i++) {
            ret.append(refAttr.get(i)).append(" ");
        }
        return ret.toString();
    }

    public static void main(String [] args) {
        Referrer referrer = new Referrer("package0.package1", "Person");

        referrer.addAttribute("Manager", "package0.package1.Manager", false);
        referrer.addAttribute("Employee", "package0.package1.Employee", false);
        System.out.println("Refferer: \n" + referrer); 
    }

    /**
     * This class represent referrer attribute object.
     */
    public class RefAttr {

        /**
         * referrer attribute name
         */
        protected String  name = null;
       
        /**
         * referrer attribute DO class name (class which reference current table)
         */
        protected String  do_name = null;
        
        /**
         * referrer security
         */
        protected boolean do_secure = DefaultTagValues.DO_SECURITY; 

        /**
         * Constructor (String, String)
         *
         * @param name Referrer attribute name.
         * @param do_name Attribute do_name.
         * @param isSecure Attribute isSecure.
         */
        public RefAttr(String  name, String  do_name, boolean isSecure) {
            this.name = name;
            this.do_name = do_name;
            this.do_secure = isSecure;
        }

        /**
         * Get referrer attribute name.
         *
         * @return Referrer name.
         */
        public String  name() {
            return name;
        }
    
        /**
         * Get referrer attribute DO class name.
         *
         * @return Referrer DO class name.
         */
        public String  doName() {
            return do_name;
        }
    
        /**
         * Get referrer secyrity.
         *
         * @return if Referrer is security based.
         */
        public boolean isSecure() {
            return do_secure;
        }
    
        /**
         * Get String representation.
         *
         * @return String representation.
         */
        public String  toString() {
            return new StringBuffer ("name=").append(name).append(" do_name=").append(do_name).toString();
        }
    }
}
