
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: Index.java,v 1.1 2004/09/03 13:43:13 sinisa Exp $
 */

/*
 *
 * @author    Nenad Vico
 * @since     LBS1.8
 * @version   $Revision: 1.1 $
 *
 */
package org.enhydra.dods.trans;

import java.util.ArrayList ;

/**
 * This class represents index object. 
 */
public class Index {
    
    /**
     * list of indexed columns
     */
    protected ArrayList  indColumn = new ArrayList (); 

    /**
     * id of index
     */
    protected String  id = null;

    /**
     * This variable contains information weather index is unique.
     * True - is unique.
     * False - is not unique.
     */
    protected boolean unique = false;

    /**
     * This variable contains information weather index is clustered.
     * True - is clustered.
     * False - is not clustered.
     */
    protected boolean clustered = false;

    /**
     * default constructor
     */
    public Index() {
        this.id = null;
        this.unique = false;
    }

    /**
     * Constructor (String, boolean)
     *
     * @param id Index id.
     * @param unique True if index is unique, otherwise false.
     */
    public Index(String  id, boolean unique) {
        this.id = id;
        this.unique = unique;
    }

    /**
     * Set index id.
     *
     * @param id Index id.
     */
    public void id(String  id) {
        this.id = id;
    }

    /**
     * Set index to/not to be unique.
     *
     * @param unique True if index is unique, otherwise false.
     */
    public void isUnique(boolean unique) {
        this.unique = unique;
    }

    /**
     * Set index to/not to be clustered.
     *
     * @param clustered True if index is clustered, otherwise false.
     */
    public void isClustered(boolean clustered) {
        this.clustered = clustered;
    }

    /**
     * Add indexed column.
     *
     * @param id Indexed column's id.
     */
    public void addIndexColumn(String  id) {
        indColumn.add(new String (id));
    }

    /**
     * Get number of indexed columns.
     *
     * @return Number of indexed columns.
     */
    public int size() {
        return indColumn.size();    
    }

    /**
     * Get index id
     * 
     * @return Index id.
     */
    public String  id() {
        return id;
    }

    /**
     * Get information whether column is unique.
     *
     * @return True if column is unique, otherwise false.
     */
    public boolean isUnique() {
        return unique;
    }

    /**
     * Get information whether column is clustered.
     *
     * @return True if column is clustered, otherwise false.
     */
    public boolean isClustered() {
        return clustered;
    }

    /**
     * Get indexed columns.
     *
     * @return Indexed columns.
     */
    public ArrayList  indexColumns() {
        return indColumn;
    }

    /**
     * Get indexed column with appropriate index.
     *
     * @param index Index of indexed column.
     *
     * @return Id of indexed column.
     */
    public String  indexColumn(int index) {
        return (String ) indColumn.get(index);
    }

    /**
     * Get String representation.
     *
     * @return String representation.
     */
    public String  toString() {
        StringBuffer  ret = new StringBuffer ("id=").append(id).append(" unique=").append(unique).append(" index columns: ");

        for (int i = 0; i < indColumn.size(); i++) {
            ret.append(indColumn.get(i)).append(" ");
        }
        return ret.toString();
    }

    public static void main(String [] args) {
        Index ind = new Index("MeinIndexName1", true);

        ind.addIndexColumn("KEYVALUE");
        ind.addIndexColumn("ATTRTYPE");
        System.out.println("Index: \n" + ind); 
    }
}
