
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: TransientXMLBuilderFactory.java,v 1.2 2004/12/16 12:15:04 predrag Exp $
 */

/*
 *
 * @author    Dragan Radeka & Nenad Vico
 * @since     LBS1.8
 * @version   $Revision: 1.2 $
 *
 */
package org.enhydra.dods.trans;

import java.io.File ;
import java.io.FileOutputStream ;
import java.util.HashMap ;
import java.util.Iterator ;
import java.util.Map ;
import javax.xml.parsers.DocumentBuilder ;
import javax.xml.parsers.DocumentBuilderFactory ;
import org.enhydra.dods.wizard.TraceDialog;
import org.enhydra.dods.xslt.XSLTUtil;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.NamedNodeMap ;
import org.w3c.dom.Node ;
import org.w3c.dom.NodeList ;
import org.enhydra.dods.Common;
import org.enhydra.dods.generator.DODSGenerator;

/**
 * This class is factory class for generating transient XML files into given project root by reading input doml file.
 */
public class TransientXMLBuilderFactory {
    // errors messages
 public static final String  ERROR_NO_PARSED_DOCUMENT = "ERROR_NO_PARSED_DOCUMENT";
    public static final String  ERROR_NO_DATABASE_TAG = "ERROR_NO_DATABASE_TAG";
    public static final String  ERROR_NO_DATABASE_ATTRIBUTES = "ERROR_NO_DATABASE_ATTRIBUTES";
    public static final String  ERROR_NO_DATABASE_ATTRIBUTE = "ERROR_NO_DATABASE_ATTRIBUTE";
    public static final String  ERROR_NO_TABLE_TAG = "ERROR_NO_TABLE_TAG";
    public static final String  ERROR_NO_COLUMN_TAG = "ERROR_NO_COLUMN_TAG";
    public static final String  TABLE_ATTR_NAMES[] = {
        "id", "dbTableName", "isView", "generateSecure", "generateInsecure",
        "multidb", "dirtyDOs","massUpdates","massDeletes"
    };                       
    public static final String  DIRTY_DO_DEFAULT = "Compatible";
    static final int TABLE_ID = 0;
    static final int TABLE_DB_TABLE_NAME = 1;
    static final int TABLE_IS_VIEW = 2;    
    static final int TABLE_SECURITY = 3;
    static final int TABLE_NON_SECURITY = 4;    
    static final int TABLE_MULTIDB = 5;        
    static final int TABLE_DIRTY_DOS = 6;
    static final int TABLE_MASS_UPDATES = 7;
    static final int TABLE_MASS_DELETES = 8;
    public static final String  COLUMN_ATT_NAMES[] = {
        "id", "usedForQuery", "isConstant", "generateSecure", "generateInsecure",
    };
    static final int COLUMN_ID = 0;
    static final int COLUMN_USED_FOR_QUERY = 1;
    static final int COLUMN_IS_CONSTANT = 2;
    static final int COLUMN_SECURITY = 3;    
    static final int COLUMN_NON_SECURITY = 4;        
    public static final String  REF_OBJECT_ATTR_NAMES[] = {
        "constraint", "reference"
    };
    static final int COLUMN_CONSTRAINT = 0;
    static final int COLUMN_REFERENCE = 1;
    // attributes of type tag in column tag
 public static final String  TYPE_ATTR_NAMES[] = {
        "size", "canBeNull", "dbType", "javaType"
    };
    static final int COLUMN_SIZE = 0;
    static final int COLUMN_CAN_BE_NULL = 1;
    static final int COLUMN_DB_TYPE = 2;
    static final int COLUMN_JAVA_TYPE = 3;
     
    // attributes of index tag in table tag
 public static final String  INDEX_ATTR_NAMES[] = {
        "id", "unique", "clustered"
    };
    static final int INDEX_ID = 0;
    static final int INDEX_UNIQUE = 1;
    static final int INDEX_CLUSTERED = 2;

    /**
     * Store all tables read from doml file.
     */
    protected HashMap  tables;

    /**
     * used project
     */
    protected String  project = DefaultTagValues.TABLE_PROJECT_NAME;

    /**
     * used author
     */
    protected String  author = DefaultTagValues.TABLE_AUTHOR;

    /**
     * used database
     */
    protected String  database = null;

    /**
     * used database
     */
    protected String  templateSet = DefaultTagValues.TABLE_TEMPLATE_SET;    

    /**
     * trace dialog used for showing output
     */
    protected TraceDialog td = null;

    /**
     * Constructor.
     */
    public TransientXMLBuilderFactory() {
        tables = new HashMap ();
    }

    /**
     * Constructor with trace dialog.
     *
     * @param td Trace dialog used for showing output.
     */
    public TransientXMLBuilderFactory(TraceDialog td) {
        this.td = td;
        tables = new HashMap ();
    }

    /**
     * Load doml file in memory structure.
     */
    public String  readDoml() throws InvalidDomlException {
        String  projectRoot = Common.getProjectRoot();
        String  domlFile = projectRoot + File.separator
                + Common.getDomlFileName(); // read doml file project root
 Document  doc = null;
        String  nodeValue = "";

        try { // read DOM parser
 DocumentBuilderFactory  dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder  db = dbf.newDocumentBuilder();

            doc = db.parse(domlFile);
        } catch (RuntimeException  e) {
            e.printStackTrace();
        } catch (Exception  e1) {
            e1.printStackTrace();
        }
    
        if (doc == null) { 
            return ERROR_NO_PARSED_DOCUMENT;
        }
        // read author tag
 try {
            NodeList  nodeListTagAuthor = doc.getElementsByTagName("author");

            if (nodeListTagAuthor != null) {
                NodeList  nodeText = nodeListTagAuthor.item(0).getChildNodes();

                if (nodeText.item(0) != null) {
                    author = nodeText.item(0).getNodeValue();
                }
            }
        } catch (Exception  e) {}
        // read project tag
 try {
            NodeList  nodeListTagProject = doc.getElementsByTagName("projectname");

            if (nodeListTagProject != null) {
                NodeList  nodeText = nodeListTagProject.item(0).getChildNodes();

                if (nodeText.item(0) != null) {
                    project = nodeText.item(0).getNodeValue();
                }
            }
        } catch (Exception  e) {}
           
        // read database tag
 NodeList  nodeListTagDatabase = doc.getElementsByTagName("database");

        if (nodeListTagDatabase == null) { 
            return ERROR_NO_DATABASE_TAG;
        }
        NamedNodeMap  databaseAttrs = nodeListTagDatabase.item(0).getAttributes();

        if (database == null || database.equals(DODSGenerator.DATABASE_NOT_SET)) {
            if (databaseAttrs == null) { 
                return ERROR_NO_DATABASE_ATTRIBUTES;
            }
            Node  nodeDatabase = databaseAttrs.getNamedItem("database");

            if (nodeDatabase == null) { 
                return ERROR_NO_DATABASE_ATTRIBUTE;
            }
            database = nodeDatabase.getNodeValue();
            if(System.getProperty("DATABASE_VENDOR")==null || System.getProperty("DATABASE_VENDOR").equals(DODSGenerator.DATABASE_NOT_SET))
               System.setProperty("DATABASE_VENDOR",database);
        }
        DefaultTagValues.loadDatabaseValues(database);
        Node  nodeTemplateSet = databaseAttrs.getNamedItem("templateset");

        if (nodeTemplateSet != null) {
            templateSet = nodeTemplateSet.getNodeValue();
        }
                
        Node  dirtyTag = databaseAttrs.getNamedItem("dirtyDOs");
        String  dirty = null;

        if (dirtyTag != null) {
            dirty = dirtyTag.getNodeValue();
        } else {
            dirty = DIRTY_DO_DEFAULT;
        }
        boolean dbMassUpdate = DefaultTagValues.DATABASE_MASS_UPDATE;
        Node  dbMassUpdateTag = databaseAttrs.getNamedItem("massUpdates");

        if (dbMassUpdateTag != null) {
            dbMassUpdate = dbMassUpdateTag.getNodeValue().equalsIgnoreCase("true");
        }

        boolean dbMassDelete = DefaultTagValues.DATABASE_MASS_DELETE;
        Node  dbMassDeleteTag = databaseAttrs.getNamedItem("massDeletes");

        if (dbMassDeleteTag != null) {
            dbMassDelete = dbMassDeleteTag.getNodeValue().equalsIgnoreCase("true");
        }

        boolean dbGenerateSecure = DefaultTagValues.DATABASE_SECURITY;
        Node  dbGenerateSecureTag = databaseAttrs.getNamedItem("generateSecure");

        if (dbGenerateSecureTag != null) {
            dbGenerateSecure = dbGenerateSecureTag.getNodeValue().equalsIgnoreCase("true");
        }
                        
        boolean dbGenerateInsecure = DefaultTagValues.DATABASE_NON_SECURITY;
        Node  dbGenerateInsecureTag = databaseAttrs.getNamedItem("generateInsecure");

        if (dbGenerateInsecureTag != null) {
            dbGenerateInsecure = dbGenerateInsecureTag.getNodeValue().equalsIgnoreCase("true");
        }
            
        // read all table tags
 NodeList  nodeListTagTable = doc.getElementsByTagName("table");

        if (nodeListTagTable == null) { 
            return ERROR_NO_TABLE_TAG;
        }
  
        // loop trough all tables 
 for (int i = 0; i < nodeListTagTable.getLength(); i++) {
            NamedNodeMap  tableAttrs = nodeListTagTable.item(i).getAttributes();

            if (tableAttrs == null) { 
                continue;
            } // attributes don't exist
 Table table = null;
            String  tableID = null;
            Referrer referrer = null;
            // read all table attributes
 String  dirtyTable = null;
            boolean md = dbMassDelete;
            boolean mu = dbMassUpdate;

            for (int j = 0; j < TABLE_ATTR_NAMES.length; j++) {
                Node  nodeTable = tableAttrs.getNamedItem(TABLE_ATTR_NAMES[j]);

                if (nodeTable != null) {
                    nodeValue = nodeTable.getNodeValue();
                    switch (j) { // depend on table attributes
 case TABLE_ID: {
                            tableID = nodeValue;
                            table = (Table) tables.get(tableID);
                            if (table == null) {
                                table = new Table(dbGenerateSecure,
                                        dbGenerateInsecure, dirty);
                                tables.put(tableID, table);
                            }
                            String  className = Common.capitalizeName(tableID.substring(tableID.lastIndexOf('.')
                                    + 1));

                            table.pckg(tableID.substring(0,
                                    tableID.lastIndexOf('.')));
                            table.projectName(tableID.substring(0,
                                    tableID.indexOf('.')));
                            table.tableName(className);
                            table.className(className);
                        }
                        break;

                    case TABLE_DB_TABLE_NAME: {
                            table.tableName(nodeValue);
                        }
                        break;

                    case TABLE_IS_VIEW: {}
                        break;

                    case TABLE_SECURITY: {
                            table.doSecure(nodeValue.equalsIgnoreCase("true"));
                        }
                        break;

                    case TABLE_NON_SECURITY: {
                            table.doInSecure(nodeValue.equalsIgnoreCase("true"));
                        }
                        break;

                    case TABLE_MULTIDB: {
                            table.doMultidb(nodeValue.equalsIgnoreCase("true"));
                        }
                        break;

                    case TABLE_DIRTY_DOS: {
                            table.setDirtyDOs(nodeValue);
                        }
                        break;
                    case TABLE_MASS_UPDATES: {
                            mu = nodeValue.equalsIgnoreCase("true");
                            if(!dbMassUpdate)
                              mu = mu || dbMassUpdate;
                        }
                        break;
                    case TABLE_MASS_DELETES: {
                            md = nodeValue.equalsIgnoreCase("true");
                            if(!dbMassDelete)
                              md = md || dbMassDelete;
//                            table.doMassDeletes(md || dbMassDelete);
                        }
                        break;        
                            
                    } // switch
                } // if (nodeTable != null)
 
            } // for TABLE_ATTR_NAMES.length 
 
            Element  tagTable = (Element ) nodeListTagTable.item(i);
            // read all column tags 
 NodeList  nodeListTagColumn = tagTable.getElementsByTagName("column");

            if (nodeListTagColumn == null) { 
                return ERROR_NO_COLUMN_TAG;
            }
            // loop through all column tags
 for (int j = 0; j < nodeListTagColumn.getLength(); j++) {
                Column column = new Column(table.doSecure(), table.doInSecure());
                NamedNodeMap  columnAttrs = nodeListTagColumn.item(j).getAttributes();

                if (columnAttrs == null) { 
                    continue;
                }
                // read and process all column attributes
 for (int k = 0; k < COLUMN_ATT_NAMES.length; k++) {
                    Node  nodeColumn = columnAttrs.getNamedItem(COLUMN_ATT_NAMES[k]);

                    if (nodeColumn != null) {
                        nodeValue = nodeColumn.getNodeValue();
                        switch (k) { // depend on column attributes
 case COLUMN_ID: {
                                column.name(nodeValue.substring(nodeValue.lastIndexOf('.')
                                        + 1));
                            }
                            break;

                        case COLUMN_USED_FOR_QUERY: {
                                column.usedForQuery(nodeValue.equalsIgnoreCase("true"));
                            }
                            break;

                        case COLUMN_IS_CONSTANT: {
                                column.isConstant(nodeValue.equalsIgnoreCase("true"));
                            }
                            break;

                        case COLUMN_SECURITY: {
                                column.isSecure(nodeValue.equalsIgnoreCase("true"));
                                if (nodeValue.equalsIgnoreCase("true")) {
                                    table.anyColumnSecure(true);
                                }
                            }
                            break;

                        case COLUMN_NON_SECURITY: {
                                column.isInSecure(nodeValue.equalsIgnoreCase("true"));
                            }
                            break;
                        } // switch
                    } // if (nodeColumn != null)
                }
                Element  columnTag = (Element ) nodeListTagColumn.item(j);
                // read error tag
 NodeList  nodeListTagError = columnTag.getElementsByTagName("error");

                if (nodeListTagError != null) { 
                    for (int k = 0; k < nodeListTagError.getLength(); k++) {
                        NodeList  nodeText = nodeListTagError.item(k).getChildNodes();

                        if (nodeText.item(0) != null) {// column.error(nodeText.item(0).getNodeValue());
                        }
                    }
                }           
                // read javadoctag tag
 NodeList  nodeListTagJavadoc = columnTag.getElementsByTagName("javadoc");

                if (nodeListTagJavadoc != null) {
                    for (int k = 0; k < nodeListTagJavadoc.getLength(); k++) {
                        NodeList  nodeText = nodeListTagJavadoc.item(k).getChildNodes();

                        if (nodeText.item(0) != null) {
                            column.javadoc(nodeText.item(0).getNodeValue());
                        }
                    }
                }
                // read initialValue tag
 NodeList  nodeListTagInitialValue = columnTag.getElementsByTagName("initialValue");

                if (nodeListTagInitialValue != null) {
                    for (int k = 0; k < nodeListTagInitialValue.getLength(); k++) {
                        NodeList  nodeText = nodeListTagInitialValue.item(k).getChildNodes();

                        if (nodeText.item(0) != null) {
                            column.javaDefaultValue(nodeText.item(0).getNodeValue());
                        }
                    }
                }
                // read referenceObject tag 
 NodeList  nodeListTagReferenceObject = columnTag.getElementsByTagName("referenceObject");

                if (nodeListTagReferenceObject != null) {
                    for (int k = 0; k < nodeListTagReferenceObject.getLength(); k++) {
                        NamedNodeMap  referenceObjectAttrs = nodeListTagReferenceObject.item(k).getAttributes();

                        if (referenceObjectAttrs == null) { 
                            break;
                        }
                        // read all referenceObject attributes 
 for (int l = 0; l < REF_OBJECT_ATTR_NAMES.length; l++) {
                            Node  nodeReferenceObject = referenceObjectAttrs.getNamedItem(REF_OBJECT_ATTR_NAMES[l]);//

                            if (nodeReferenceObject != null) {
                                nodeValue = nodeReferenceObject.getNodeValue();
                                switch (l) { // depend on column reference attributes
 case COLUMN_REFERENCE: {
                                        column.refPackage(nodeValue.substring(0,
                                                nodeValue.lastIndexOf('.')));
                                        column.refName(Common.capitalizeName(nodeValue.substring(nodeValue.lastIndexOf('.')
                                                + 1)));
                                    }
                                    break;

                                case COLUMN_CONSTRAINT: {
                                        column.refConstarint(nodeValue.equalsIgnoreCase("true"));
                                    }
                                    break;
                                } // switch
                            } // if (nodeReferenceObject != null)
                        } // for  l < REF_OBJECT_ATTR_NAMES.length;
                    } // for  k < nodeListTagReferenceObject.getLength
                } // if (nodeListTagReferenceObject != null) 
 // read type tag    
 NodeList  nodeListTagType = columnTag.getElementsByTagName("type");

                if (nodeListTagType != null) {
                    for (int k = 0; k < nodeListTagType.getLength(); k++) {
                        if (nodeListTagType.item(k) != null) {
                            NamedNodeMap  typeAttrs = nodeListTagType.item(k).getAttributes();

                            if (typeAttrs == null) { 
                                break;
                            }
                                
                            // read all type attributes 
 for (int l = 0; l < TYPE_ATTR_NAMES.length; l++) {
                                Node  nodeType = typeAttrs.getNamedItem(TYPE_ATTR_NAMES[l]);//

                                if (nodeType != null) {
                                    nodeValue = nodeType.getNodeValue();
                                    switch (l) { // depend on column reference attributes
 case COLUMN_SIZE: {
                                            column.size(nodeValue);
                                        }
                                        break;

                                    case COLUMN_CAN_BE_NULL: {
                                            column.canBeNull(nodeValue.equalsIgnoreCase("true"));
                                        }
                                        break;

                                    case COLUMN_DB_TYPE: {
                                            column.dbType(nodeValue);
                                        }
                                        break;

                                    case COLUMN_JAVA_TYPE: {
                                            column.javaType(nodeValue);
                                        }
                                        break;
                                    } // switch
                                } // if (nodeType != null)
                            } // for l < TYPE_ATTR_NAMES.length
                        } // if(nodeListTagType.item(k) != null
                    } // for  k < nodeListTagType.getLength();
                } // if (nodeListTagType != null)
                table.addColumn(column); 
                // if column has reference, adds referrer attribute in referrer of appropriate table
 if (column.isReference()) {
                    if (column.refPackage() != null && column.refName() != null) {
                        String  refTableID = column.refPackage() + "."
                                + column.refName();

                        // extra conditions for referrer atribute
 if (!refTableID.equals(".")) {
                            if (column.usedForQuery() && !column.refIsAbstarct()) {     
                                if (!column.refIsForeignKey()) {
                                    if (referrer == null) {
                                        referrer = new Referrer(table.pckg(),
                                                table.className());
                                    }
                                    referrer.secure(table.doSecure());
                                    referrer.addAttribute(column.name(),
                                            refTableID, column.isSecure());
                                }
                            }
                        } 
                    }
                }
            } // for j < nodeListTagColumn.getLength(); -- end of one table
 
            // if table has referrer, adds it to approperate table
 if (referrer != null) {
                for (int j = 0; j < referrer.size(); j++) {
                    String  refTableID = referrer.attributeDoName(j);    
                    Table refTable = (Table) tables.get(refTableID);

                    if (refTable == null) {
                        refTable = new Table(dbGenerateSecure,
                                dbGenerateInsecure, dirty);
                        tables.put(refTableID, refTable);
                    }
                    refTable.addReferrer(referrer);
                }
            }
            // read index tag   
 NodeList  nodeListTagIndex = tagTable.getElementsByTagName("index");

            if (nodeListTagIndex != null) {
                for (int j = 0; j < nodeListTagIndex.getLength(); j++) {
                    if (nodeListTagIndex.item(j) != null) {
                        NamedNodeMap  indexAttrs = nodeListTagIndex.item(j).getAttributes();

                        if (indexAttrs == null) { 
                            break;
                        }
                        Index index = new Index();

                        // read all type attributes 
 for (int k = 0; k < INDEX_ATTR_NAMES.length; k++) {
                            Node  nodeType = indexAttrs.getNamedItem(INDEX_ATTR_NAMES[k]);//

                            if (nodeType != null) {
                                nodeValue = nodeType.getNodeValue();
                                switch (k) { // depend on column reference attributes
 case INDEX_ID: {
                                        index.id(nodeValue);
                                    }
                                    break;

                                case INDEX_UNIQUE: {
                                        index.isUnique(nodeValue.equalsIgnoreCase("true"));
                                    }
                                    break;

                                case INDEX_CLUSTERED: {
                                        index.isClustered(nodeValue.equalsIgnoreCase("true"));
                                    }
                                    break;
                                } // switch
                            } // if (nodeType != null)
                        } // for k < TYPE_ATTR_NAMES.length
 // read indexColumns tag    
 Element  indexTag = (Element ) nodeListTagIndex.item(j);
                        NodeList  nodeListTagIndexColumn = indexTag.getElementsByTagName("indexColumn");

                        if (nodeListTagIndexColumn != null) {
                            for (int k = 0; k
                                    < nodeListTagIndexColumn.getLength(); k++) {
                                NamedNodeMap  indexColumnAttrs = nodeListTagIndexColumn.item(k).getAttributes();

                                if (indexColumnAttrs == null) { 
                                    break;
                                }
                                // read all referenceObject attributes 
 Node  nodeIndexColumnID = indexColumnAttrs.getNamedItem("id");//

                                if (nodeIndexColumnID != null) {
                                    index.addIndexColumn(new String (nodeIndexColumnID.getNodeValue()));
                                }
                            } // for  k < nodeListTagIndexColumn.getLength
                        } // if (nodeListTagIndexColumn != null)
                        table.addIndex(index);
                    } // if(nodeListTagIndex.item(j) != null
                } // for j < nodeListTagIndex.getLength()
            } // if (nodeListTagIndex != null)
            table.doMassUpdates(mu);
            table.doMassDeletes(md);
        } // for  i < nodeListTagTable.getLength();
 
        return "";
    }

    /**
     * Generate transient XML file from memory structure.
     */
    public void generateTransientXML() throws InvalidDomlException {
        if (td == null) {
            System.out.println("Creating TransientXML");
        } else {
            td.appendLine("Creating TransientXML\n");
        }           
        Iterator  iter = tables.values().iterator();
        // loop through all tables
 // String templateSet = Common.getExtensions();
 StringBuffer  tablesBuff = new StringBuffer ("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n\n");

        tablesBuff.append("<DATABASE>\n");
        while (iter.hasNext()) {
            Table table = (Table) iter.next();
            
            // errors in table tags of doml file
 if (table.pckg() == null) {
                throw new InvalidDomlException("Invalid table package");
            }
            if (table.tableName() == null) {
                throw new InvalidDomlException("Invalid table name");
            }
            if (table.className() == null) {
                throw new InvalidDomlException("Invalid class name");
            }
            if (table.pckg() == null) {
                throw new InvalidDomlException("Invalid table package");
            }
            tablesBuff.append(" <TABLE name=\"" + table.tableName() + "\" ");
            tablesBuff.append("id=\"" + table.pckg() + "." + table.className()
                    + "\" ");
            tablesBuff.append("path=\"" + table.pckg().replace('.', '/') + "\" ");
            tablesBuff.append("class=\"" + table.className() + "\"/>\n");
            
            // write table
 //StringBuffer xmlBuff = new StringBuffer();
 StringBuffer  xmlBuff = new StringBuffer ("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n\n");

            xmlBuff.append("\n<TABLE>\n");
            xmlBuff.append("    <PACKAGE>"
                    + XSLTUtil.getAdjustedPackageName(table.pckg())
                    + "</PACKAGE>\n");
            xmlBuff.append("    <AUTHOR>" + author + "</AUTHOR>\n");
            xmlBuff.append("    <PROJECT_NAME>" + project + "</PROJECT_NAME>\n");
            xmlBuff.append("    <TABLE_NAME>" + table.tableName()
                    + "</TABLE_NAME>\n");
            xmlBuff.append("    <CLASS_NAME>" + table.className()
                    + "</CLASS_NAME>\n");
            xmlBuff.append("    <DB_VENDOR>" + table.dbVendor()
                    + "</DB_VENDOR>\n");
            xmlBuff.append("    <TEMPLATE_SET>" + templateSet
                    + "</TEMPLATE_SET>\n");
            xmlBuff.append("    <GENERATE_SECURE>" + table.doSecure()
                    + "</GENERATE_SECURE>\n");
            xmlBuff.append("    <GENERATE_INSECURE>" + table.doInSecure()
                    + "</GENERATE_INSECURE>\n");
            xmlBuff.append("    <MASS_UPDATES>" + table.doMassUpdates()
                    + "</MASS_UPDATES>\n");
            xmlBuff.append("    <MASS_DELETES>" + table.doMassDeletes()
                    + "</MASS_DELETES>\n");
            xmlBuff.append("    <DO_IS_OID_BASED>" + table.doIsOidBased()
                    + "</DO_IS_OID_BASED>\n");
            xmlBuff.append("    <IS_ABSTRACT>" + table.isAbstract()
                    + "</IS_ABSTRACT>\n");
            xmlBuff.append("    <DELETE_CASCADES>" + table.deleteCascade()
                    + "</DELETE_CASCADES>\n");
            xmlBuff.append("    <DO_IS_MULTIDB_BASED>" + table.doMultidb()
                    + "</DO_IS_MULTIDB_BASED>\n");
            xmlBuff.append("    <IS_ANY_COLUMN_SECURE>"
                    + table.isAnyColumnSecure() + "</IS_ANY_COLUMN_SECURE>\n");
            xmlBuff.append("    <GENERATE_DIRTY>" + table.getDirtyDOs()
                    + "</GENERATE_DIRTY>\n\n");            
            
            // loop through all columns of table
 for (int i = 0; i < table.columnsSize(); i++) {
                Column column = table.column(i);

                // errors in column tags of doml file
 if (column.name() == null) {
                    throw new InvalidDomlException("Invalid column name");
                }
                if (column.dbType() == null) {
                    throw new InvalidDomlException("Invalid column database type");
                }
                if (column.javaType() == null) {
                    throw new InvalidDomlException("Invalid column java type");
                }
                if (column.size() == null) {
                    column.size(DefaultTagValues.getDefaultSize(column.dbType(),
                            column.javaType()));
                }
                if (column.isSecure() == false && column.isInSecure() == false) {
                    column.isInSecure(true);
                }
                // write column
                xmlBuff.append("    <COLUMN name=\"" + column.name() + "\">\n");
                if (column.isReference()) {
                    String  refTableName = column.refName();
                    Table refTable = (Table) tables.get(column.refPackage()
                            + "." + column.refName());

                    if (refTable != null) {
                        refTableName = refTable.tableName();
                    }
                    
                    if (column.refName() == null) {
                        throw new InvalidDomlException("Invalid column reference name");
                    }
                    if (column.refPackage() == null) {
                        throw new InvalidDomlException("Invalid column reference package");
                    }
                    xmlBuff.append("      <REFERENCE_OBJECT name=\""
                            + column.refName() + "\">\n");
                    xmlBuff.append("          <CONSTRAINT>"
                            + column.refConstarint() + "</CONSTRAINT>\n");
                    xmlBuff.append("          <IS_ABSTRACT>"
                            + column.refIsAbstarct() + "</IS_ABSTRACT>\n");
                    xmlBuff.append("          <IS_FOREIGN_KEY>"
                            + column.refIsForeignKey() + "</IS_FOREIGN_KEY>\n");
                    xmlBuff.append("          <PACKAGE>"
                            + XSLTUtil.getAdjustedPackageName(column.refPackage())
                            + "</PACKAGE>\n");
                    xmlBuff.append("          <TABLE_NAME>" + refTableName
                            + "</TABLE_NAME>\n");
                    if (column.refForeignKeyColumnName() != null) {
                        xmlBuff.append("            <FOREIGN_KEY_COLUMN>"
                                + column.refForeignKeyColumnName()
                                + "</FOREIGN_KEY_COLUMN>\n");
                    }
                    if (column.refForeignKeyGroup() != null) {
                        xmlBuff.append("            <FOREIGN_KEY_GROUP>"
                                + column.refForeignKeyGroup()
                                + "</FOREIGN_KEY_GROUP>\n");
                    }
                    xmlBuff.append("      </REFERENCE_OBJECT>\n");
                }
                xmlBuff.append("        <IS_CONSTANT>" + column.isConstant()
                        + "</IS_CONSTANT>\n");
                if (column.javadoc() != null) {
                    xmlBuff.append("      <JAVADOC>" + column.javadoc()
                            + "</JAVADOC>\n");
                }
                xmlBuff.append("        <DB_TYPE>" + column.dbType()
                        + "</DB_TYPE>\n");
                xmlBuff.append("        <JAVA_TYPE>"
                        + XSLTUtil.adjustJavaType(column.javaType())
                        + "</JAVA_TYPE>\n");
                if (column.javaDefaultValue() != null) {
                    xmlBuff.append("        <JAVA_DEFAULT_VALUE>"
                            + column.javaDefaultValue()
                            + "</JAVA_DEFAULT_VALUE>\n");
                }
                xmlBuff.append("        <USED_FOR_QUERY>"
                        + column.usedForQuery() + "</USED_FOR_QUERY>\n");
                xmlBuff.append("        <CAN_BE_NULL>" + column.canBeNull()
                        + "</CAN_BE_NULL>\n");
                xmlBuff.append("        <IS_PRIMARY_KEY>"
                        + column.isPrimaryKey() + "</IS_PRIMARY_KEY>\n");
                if (column.size() != null) {
                    xmlBuff.append("        <SIZE>" + column.size()
                            + "</SIZE>\n");
                }
                xmlBuff.append("        <IS_ARRAY>" + column.isArray()
                        + "</IS_ARRAY>\n");
                xmlBuff.append("        <GENERATE_SECURE>" + column.isSecure()
                        + "</GENERATE_SECURE>\n");
                xmlBuff.append("        <GENERATE_INSECURE>"
                        + column.isInSecure() + "</GENERATE_INSECURE>\n");
                xmlBuff.append("    </COLUMN>\n\n\n");
            }
            // loop through all idexes of table
 for (int i = 0; i < table.indexesSize(); i++) {
                Index index = table.index(i);

                // errors in column tags of doml file
 if (index.id() == null) {
                    throw new InvalidDomlException("Invalid index id");
                }
                // write column
                xmlBuff.append("    <INDEX id=\""
                        + XSLTUtil.getAdjustedPackageName(index.id())
                        + "\" unique=\"" + index.isUnique() + "\" clustered=\""
                        + index.isClustered() + "\">\n");
                for (int j = 0; j < index.size(); j++) {
                    xmlBuff.append("        <INDEX_COLUMN id=\""
                            + index.indexColumn(j) + "\"/>\n");
                }
                xmlBuff.append("    </INDEX>\n\n");
            }
            // loop through all referrers of table
 for (Iterator  refIter = table.referrersValueIterator(); refIter.hasNext();) {
                Referrer referrer = (Referrer) refIter.next();

                xmlBuff.append("    <REFERRER name=\"" + referrer.name()
                        + "\" package=\""
                        + XSLTUtil.getAdjustedPackageName(referrer.pckg())
                        + "\" generateSecure=\"" + referrer.isSecure() + "\">\n");
                for (int i = 0; i < referrer.size(); i++) {
                    boolean n2n = false;
                    String  another = "";

                    if (referrer.size() == 2) {
                        if (!(table.pckg() + "." + table.className()).equals(referrer.attributeDoName(i))) {
                            n2n = true;
                            Table doTable = (Table) tables.get(referrer.attributeDoName(i));

                            if (doTable.isAbstract()) {
                                n2n = false;
                            }
                            if (doTable.referrers().containsKey(table.pckg()
                                    + "." + table.className())) {
                                n2n = false;
                            }
                            if (table.referrers().containsKey(referrer.attributeDoName(i))) {
                                n2n = false;
                            }
                            if (n2n) {
                                int j = (0 == i ? 1 : 0);  // look at other attr

                                another = "\" another=\""
                                        + referrer.attributeName(j);
                            }
                        }
                    }
                    xmlBuff.append("        <REFATTR name=\""
                            + referrer.attributeName(i) + "\" do_name=\""
                            + XSLTUtil.getAdjustedPackageName(referrer.attributeDoName(i))
                            + "\" n2n=\"" + n2n + another
                            + "\" generateSecure=\""
                            + referrer.attributeSecurity(i) + "\"/>\n");
                }
                xmlBuff.append("    </REFERRER>\n");
            }
            xmlBuff.append("\n</TABLE>\n");
            // Creating transient XML file and target directory
 String  filename = Common.getProjectRoot() + File.separator
                    + (XSLTUtil.getAdjustedPackageName(table.pckg()) + "." + table.className()).replace('.',
                            File.separatorChar) + ".xml";
            String  dirname = Common.getProjectRoot() + File.separator
                    + XSLTUtil.getAdjustedPackageName(table.pckg()).replace('.',
                            File.separatorChar);

            if (td == null) {
                System.out.println("Creating " + filename);
            } else {
                td.appendLine("Creating " + filename + "\n");
            }           
            File  file = new File (filename);
            File  dir = new File (dirname);

            dir.mkdirs();
            try {
                FileOutputStream  outStream = new FileOutputStream (file);

//                outStream.write(xmlBuff.toString().getBytes());
                outStream.write(xmlBuff.toString().getBytes("UTF-8"));
                outStream.close();
            } catch (Exception  e) {
                e.printStackTrace();
            }
        } // while (iter.hasNext())
        tablesBuff.append("</DATABASE>\n");
        String  tablesname = Common.getProjectRoot() + File.separator
                + "tables.xml";
        File  file = new File (tablesname);

        try {
            FileOutputStream  outStream = new FileOutputStream (file);

            //outStream.write(tablesBuff.toString().getBytes());
            outStream.write(tablesBuff.toString().getBytes("UTF-8"));
            outStream.close();
        } catch (Exception  e) {
            e.printStackTrace();
        }
    }

    /**
     * Generate tables.xml file from memory structure.
     */
    public void generateTablesXML() throws InvalidDomlException {
        Iterator  iter = tables.values().iterator();
        // loop through all tables
 StringBuffer  tablesBuff = new StringBuffer ("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n\n");

        tablesBuff.append("<DATABASE>\n");
        while (iter.hasNext()) {
            Table table = (Table) iter.next();
            
            // errors in table tags of doml file
 if (table.pckg() == null) {
                throw new InvalidDomlException("Invalid table package");
            }
            if (table.tableName() == null) {
                throw new InvalidDomlException("Invalid table name");
            }
            if (table.className() == null) {
                throw new InvalidDomlException("Invalid class name");
            }
            if (table.pckg() == null) {
                throw new InvalidDomlException("Invalid table package");
            }
            tablesBuff.append(" <TABLE name=\"" + table.tableName() + "\" ");
            tablesBuff.append("id=\""
                    + XSLTUtil.getAdjustedPackageName(table.pckg()) + "."
                    + table.className() + "\" ");
            tablesBuff.append("path=\""
                    + XSLTUtil.getAdjustedPackageName(table.pckg()).replace('.',
                            '/') + "\" ");
            tablesBuff.append("massUpdates=\"" + table.doMassUpdates() + "\" ");
            tablesBuff.append("massDeletes=\"" + table.doMassDeletes() + "\" ");
            tablesBuff.append("class=\"" + table.className() + "\"/>\n");
        }
        tablesBuff.append("</DATABASE>\n");
        String  tablesname = Common.getProjectRoot() + File.separator
                + "tables.xml";
        File  file = new File (tablesname);

        try {
            FileOutputStream  outStream = new FileOutputStream (file);

            //outStream.write(tablesBuff.toString().getBytes());
            outStream.write(tablesBuff.toString().getBytes("UTF-8"));
            outStream.close();
        } catch (Exception  e) {
            e.printStackTrace();
        }
    }   

    /**
     * Generate tables.xml file from memory structure.
     */
    public void generateClassList() throws InvalidDomlException {
        Iterator  iter = tables.values().iterator();
        // loop through all tables
 StringBuffer  tablesBuff = new StringBuffer ("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n\n");
        tablesBuff.append("<CLASSES>\n");
        
        
        while (iter.hasNext()) {
            Table table = (Table) iter.next();
            
            // errors in table tags of doml file
 if (table.pckg() == null) {
                throw new InvalidDomlException("Invalid table package");
            }
            if (table.tableName() == null) {
                throw new InvalidDomlException("Invalid table name");
            }
            if (table.className() == null) {
                throw new InvalidDomlException("Invalid class name");
            }
            if (table.pckg() == null) {
                throw new InvalidDomlException("Invalid table package");
            }
            tablesBuff.append("    <CLASS name=\"" + XSLTUtil.getAdjustedPackageName(table.pckg()) + "."+ table.className() + "DO"+ "\"/>\n" );
        }
        tablesBuff.append("</CLASSES>\n");
        String  dirname = Common.getProjectRoot() +File.separator+ "org"+File.separator+"enhydra"+File.separator+"dods";
        
        String  filename = Common.getProjectRoot() +File.separator+ "org"+File.separator+"enhydra"+File.separator+"dods"+ File.separator+"DODSClassList.xml";

        File  file = new File (filename);
        File  dir = new File (dirname);

        dir.mkdirs();
        
        try {
            FileOutputStream  outStream = new FileOutputStream (file);

            //outStream.write(tablesBuff.toString().getBytes());
            outStream.write(tablesBuff.toString().getBytes("UTF-8"));
            outStream.close();
        } catch (Exception  e) {
            e.printStackTrace();
        }
    }   


    /*
     * Created for debugging.
     */
    protected void showAllTables() {
        Iterator  iter = tables.entrySet().iterator();

        while (iter.hasNext()) {
            Map.Entry  map = (Map.Entry ) iter.next();
            String  tableID = (String ) map.getKey();
            Table table = (Table) map.getValue();

            System.out.println("Table: " + tableID + "\n" + table);
            
        }
    }
    
    /*
     * Argument: parameter isTablesOnly 
     * If it is true, class generates only tables.xml file;
     * otherwise class generates both (tables.xml and transient XML) files for all tables.
     */
    public static void main(String [] args) throws Exception {
        try {
            TransientXMLBuilderFactory transientXMLBuilder = new TransientXMLBuilderFactory();

            if (args.length > 1) {
                transientXMLBuilder.database = args[1];
            }
            transientXMLBuilder.readDoml();

            if ((args[0]).equals("ClassList")) 
                transientXMLBuilder.generateClassList();
            else if ((new Boolean (args[0])).booleanValue()) 
                transientXMLBuilder.generateTablesXML();
             else 
                transientXMLBuilder.generateTransientXML();
          } catch (InvalidDomlException ide) {
                ide.printStackTrace();     
        } catch (Exception  e) {
            e.printStackTrace();
        }
    }
}
