
/**
 * Title:        DODS Access User Interface<p>
 * Description:  Determines if the user has access to details of DODS object or the whole class<p>
 * Copyright:    Copyright (c) Together Teamlösungen EDV-Dienstleistungen GmbH. under LGPL<p>
 * Company:      Together Teamlösungen EDV-Dienstleistungen GmbH.<p>
 * @author Alfred Madl
 * @version 1.0
 */
package org.webdocwf.dods.access;

import com.lutris.appserver.server.sql.Query;
import com.lutris.dods.builder.generator.dataobject.GenericDO;

public interface User {

    /**
     * Get the name of the user for output.
     * Used for Exception printing.
     * 
     * @return The printable name of the user.
     */
    public String  getName();

    /**
     * Restrict the Query for DODS Query access using SQL
     * 
     * @param query The DODS Query object which will be executed
     */
    public void restrictQuery(Query query) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * copy
     * the DO
     * 
     * @param obj The object which will be copied
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOCopyAccess(GenericDO obj) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read the existance of
     * the DO
     * 
     * @param obj The object which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAccess(GenericDO obj) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read the version of
     * the DO
     * 
     * @param obj The object which version will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetVersionAccess(GenericDO obj) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read the readOnly attribute of
     * the DO
     * 
     * @param obj The object which readonly attribute will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOIsReadOnlyAccess(GenericDO obj) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set the readOnly attribute = true of
     * the DO
     * 
     * @param obj The object which readonly attribute will be set to true
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOMakeReadOnlyAccess(GenericDO obj) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set the readOnly attribute = false of
     * the DO
     * 
     * @param obj The object which readonly attribute will be set to false
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOMakeReadWriteAccess(GenericDO obj) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * delete
     * the DO
     * 
     * @param obj The object which will be deleted
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDODeleteAccess(GenericDO obj) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value/type of
     * the DO, used for enums
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @param type The current type of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, String  type, String  value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, Object  value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, boolean value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, byte value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, short value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, int value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, long value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, float value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, double value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * read an attribute with a certain value of
     * the DO
     * 
     * @param obj The object which will be read
     * @param attrName The name of the attribute which will be read ("packages.class:attrname")
     * @param value The current value of the attribute which will be read
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOGetAttrAccess(GenericDO obj, String  attrName, byte[] value) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * create 
     * the DO of a certain class
     * 
     * @param className The name of the class which will be instanciated
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOCreateAccess(String  className) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value/type and a certain new value/type of
     * the DO, used for enums
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param oldType The current type of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @param newType The new type of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, String  oldType, String  oldValue, String  newType, String  newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, Object  oldValue, Object  newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, boolean oldValue, boolean newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, byte oldValue, byte newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, short oldValue, short newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, int oldValue, int newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, long oldValue, long newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, float oldValue, float newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, double oldValue, double newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * set an attribute with a certain old value and a certain new value of
     * the DO
     * 
     * @param obj The object which will be modified
     * @param attrName The name of the attribute which will be set ("packages.class:attrname")
     * @param oldValue The current value of the attribute which will be overwritten
     * @param newValue The new value of the attribute which will be set
     * @return Whether the User has the right to perform this task
     */
    public boolean hasDOSetAttrAccess(GenericDO obj, String  attrName, byte[] oldValue, byte[] newValue) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query
     * the DO's of a certain class
     * 
     * @param className The name of the class which will be queried for DO's
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAccess(String  className) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * find
     * the DO during a query
     * 
     * @param obj The DO which would be found with a query
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryFindAccess(GenericDO obj) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, Object  queryValue, String  cmp_op) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, boolean queryValue, String  cmp_op) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, byte queryValue, String  cmp_op) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, short queryValue, String  cmp_op) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, int queryValue, String  cmp_op) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, long queryValue, String  cmp_op) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, float queryValue, String  cmp_op) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, double queryValue, String  cmp_op) throws AccessEvalException;

    /**
     * Decide whether the User is allowed to 
     * query DO's for an attribute with a certain query-value and comparison operator
     * 
     * @param className The name of the class which will be queried for DO's
     * @param attrName The name of the attribute which will be queried ("packages.class:attrname")
     * @param queryValue The query value for the attribute
     * @param cmp_op Compare operator
     * @return Whether the User has the right to perform this task
     */
    public boolean hasQueryAttrAccess(String  className, String  attrName, byte[] queryValue, String  cmp_op) throws AccessEvalException;
}
