
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache;

import org.enhydra.dods.statistics.Statistics;
import org.enhydra.dods.exceptions.CacheObjectException;
import com.lutris.util.Config;

/**
 * This interface contains part of caches' (data object (or DataStruct object)
 *  and query) mechanisms needed for cache administration.
 *
 * @author    Tanja Jovanovic
 * @author    Nenad Vico
 * @author    Zorica Suvajdzin
 * @version   2.0  15.06.2003.
 */
public abstract class ConfigurationAdministration {

    /**
     * Returns CacheAdministration for data object (or DataStruct object) cache,
     * simple, or complex query cache. Object CacheAdministration handles
     * configuration settings about these caches.
     * Parameter cacheType can have one of these values:
     *  0 - for CacheAdministration of data object (or DataStruct object) cache
     *  1 - for CacheAdministration of simple query cache
     *  2 - for CacheAdministration of complex query cache
     *
     * @param cacheType  0 - for data object (or DataStruct object),
     * 1 for simple query and 2 for complex query cache.
     */
    public abstract CacheAdministration getCacheAdministration(int cacheType);

    /**
     * Returns initialQueryCache. This attribute contains "where" clause
     * which is used during data object (or DataStruct object) cache
     * initialization.
     *
     * @return Attribute initialQueryCache.
     */
    public abstract String  getInitialQueryCache();

    /**
     * Sets initialQueryCache attribute. This attribute contains "where" clause
     * which is used during data object (or DataStruct object) cache
     * initialization.
     *
     * @param initQ New value of initialQueryCache attribute.
     */
    protected abstract void setInitialQueryCache(String  initQ);

    /**
     * Returns statistics of used table (and of the caches, if exist).
     *
     * @return statistics of used table (and of the caches, if exist).
     */
    public abstract Statistics getStatistics();

    /**
     * Refreshes statistics.
     */
    public abstract void refreshStatistics();

    /**
     * Returns information if data object (or DataStruct object) cache if "full".
     * "Full" cache contains all data objects (or DataStruct objects) from
     * the table. The cache is "full", if data (or DataStruct) object cache is
     * unbounded, if the cached table is read-only, and if initialQueryCache
     * attribute is set to "*".
     *
     * @return true if data object (or DataStruct object) cache if "full",
     * otherwise false.
     */
    public abstract void checkFull();

    /**
     * Returns information if data object (or DataStruct object) cache if "full".
     * "Full" cache contains all data objects (or DataStruct objects) from
     * the table. The cache is "full", if data (or DataStruct) object cache is
     * unbounded, and if initialQueryCache attribute is set to "*".
     *
     * @return true if data object (or DataStruct object) cache if "full",
     * otherwise false.
     */
    public abstract boolean isFull();

    /**
     * Returns data object (or DataStruct object) cache type.
     * Possible values are:
     *   "none" - no caching available
     *   "lru"  - cache with LRU (least-recently used) algorithm
     *   "full" - special case of LRU cache - the whole table is cached
     *
     * @return data object (or DataStruct object) cache type.
     */
    public abstract String  getCacheType();

    /**
     * Returns caching level.
     * Possible values:
     *  org.enhydra.dods.cache.CacheConstants.DATA_CACHING (value 1) for
     *      data object (or DataStruct object) caching without query caching, and
     *  org.enhydra.dods.cache.CacheConstants.QUERY_CACHING (value 2) for
     *      data object (or DataStruct object) caching with query caching.
     *
     * @return Caching level (1 or 2).
     */
    public abstract int getLevelOfCaching();

    /**
     * Returns object TableConfiguration.
     * TableConfiguration contains parameters (and their set and get methods)
     * for table configuration.
     *
     * @return TableConfiguration.
     */
    public abstract TableConfiguration getTableConfiguration();

    /**
     * Returns reserveFactor.
     * reserveFactor attribute is used in query caching. It is percent of how
     * many more object are taken for evaluation. If <CODE>num</CODE> is number
     * of needed results, then  it is used
     * <CODE>num</CODE> + DEFAULT_RESERVE_FACTOR * <CODE>num</CODE> of
     * objects for estimating what is quicker: go to database for all object that
     * are not in the cache, or run again query on database.
     * This value is given in percents, as number between 0 and 1
     * (0.25 means 25%).
     * For example, if DEFAULT_RESERVE_FACTOR is 0.5, and wanted number of
     * results is 50, the estimation will be done on 75 (50 + 0.5 * 50) objects.
     *
     * @return reserveFactor.
     */
    public abstract double getReserveFactor();

    /**
     * Sets reserveFactor.
     * reserveFactor attribute is used in query caching. It is percent of how
     * many more object are taken for evaluation. If <CODE>num</CODE> is number
     * of needed results, then  it is used
     * <CODE>num</CODE> + DEFAULT_RESERVE_FACTOR * <CODE>num</CODE> of
     * objects for estimating what is quicker: go to database for all object that
     * are not in the cache, or run again query on database.
     * This value is given in percents, as number between 0 and 1
     * (0.25 means 25%).
     * For example, if DEFAULT_RESERVE_FACTOR is 0.5, and wanted number of
     * results is 50, the estimation will be done on 75 (50 + 0.5 * 50) objects.
     *
     * @param res New reserveFactor.
     */
    protected abstract void setReserveFactor(double res);

    /**
     * Returns information whether caching is disabled.
     *
     * @return true is caching is disabled, otherwise false.
     */
    public abstract boolean isDisabled();

    /**
     * Read cache configuration from application's configuration file.
     */
    public abstract double getCachePercentage();
    
    protected abstract void setCachePercentage(double cp);
  
    
    /**
     * @return
     */
    public abstract int getInitialCacheFetchSize() ;

    /**
     * @return
     */
    public abstract int getInitialDSCacheSize() ;

    /**
     * @param i
     */
    public abstract void setInitialCacheFetchSize(int i);


    /**
     * @param i
     */
    public abstract void setInitialDSCacheSize(int i) ;



    
    public abstract void readConfiguration(Config tableConfig, Config cacheConfig, String  dbName) throws CacheObjectException;
}
