
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: Common.java,v 1.2 2005/03/17 05:21:43 predrag Exp $
 */

/*
 *
 * @author    Dragan Radeka & Nenad Vico
 * @since     LBS1.8
 * @version   $Revision: 1.2 $
 *
 */
package org.enhydra.dods;

import java.io.File ;
import java.io.InputStream ;
import java.net.URL ;
import java.net.URLClassLoader ;
import java.io.FileInputStream ;
import java.io.FilenameFilter ;
import java.io.IOException ;
import java.util.HashMap ;
import java.util.HashSet ;
import java.util.Iterator ;
import java.util.Map ;
import java.util.Properties ;
import java.util.Set ;
import java.util.Vector ;

import org.enhydra.dods.wizard.DirectoryNameFilter;
import org.enhydra.xml.XMLConfig;
import org.enhydra.xml.XMLDocumentFactory;
import com.lutris.logging.Logger;
import com.lutris.util.ConfigException;
import javax.xml.parsers.DocumentBuilderFactory ;
import javax.xml.parsers.DocumentBuilder ;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.NodeList ;
import org.w3c.dom.Node ;

public class Common {

    /**
     * Database Vendor driver specific configuration parameter name. 
     * values: true/false
     */
    
    static public final String  VENDOR_ORDERED_RESULT_SET = "OrderedResultSet";
    

    /**
     * dods configuration switch property
     */
    static protected boolean configured = false;

    /**
     * dods configuration switch property
     */
    static protected String  configDir = null;


    /**
     * dods configuration path (in dods-runtime.jar file )
     */
    static public final String  DODS_CONF_JAR_PATH = "org/enhydra/dods/conf";
    
    /**
     * default app configuration file.  
     */
    
    static public final String  DATABASE_MANAGER_CONF_FILE="dods/conf/databaseManager.conf";



    /**
     * dods configuration property
     */
    static protected Properties  dods_conf;

    /**
     * last used database read from doml file
     */
    static protected String  dbase;

    /**
     * dods configuration
     */
    static protected XMLConfig dodsConf;

    /**
     * dods vendor configuration
     */
    static protected HashMap  dodsVendorConfs;
    

    /**
     * dods driver-vendor map 
     *  
     */
    static protected HashMap  dodsDriversMap;
    
    /**
     * dods (driver_name/vendor) map
     *  
     */
    static protected Vector  dodsDriversNamesVec;
    static protected Vector  dodsVendorsNamesVec;
    static protected Vector  dodsDriverClassVec;
    



    /**
     * last used project root
     */
    static protected String  projRoot;

    /**
     * last used doml file
     */
    static protected String  domlfile;

    /**
     * last used template dir
     */
    static protected String  customTemplateDir = null;
    
    
    /**
     * HashMap with changeAutocommit parameter value (from app conf file)
     * for all logical databases (declared in same file) - default true
     */
     static protected HashMap  changeAutocommit = new HashMap ();
     
    /**
     * last used template set
     */
    static protected String  templateSet;
 
 
    /**
     * Sets value of changeAutocommit 
     * @param dbName database name
     * @param value  value to set 
     */
    public static void setChangeAutocommit(String  dbName, boolean value){
        Boolean  newValue= new Boolean (value);
        changeAutocommit.put(dbName, newValue);
    }
    
    
    /**
     * Return changeAutocommit value for given datbase name.
     * @param dbName databse name
     * @return changeAutocommit value for given datbase name.
     */
    public static boolean isChangeAutocommitEnabled(String  dbName){
        Boolean  res = (Boolean )changeAutocommit.get(dbName);
        if(res==null){
            return true;
        }else{
            return res.booleanValue();
        } 
    }
  
  
    static private void init() {
        if (!configured) {      
            if (configDir != null) {
                init(configDir);
            } else {
                init(getDefaultConfigDir());
            }
        }
    
    }

    static private void init(String  confDir) {
        setConfigDir(confDir);
        String  dodsFile = confDir + File.separator + "dodsConf.xml";
        dodsFile.replace('\\', '/');
        
        Document  doc = null; 
        if (confDir != null) { 
            doc = XMLDocumentFactory.parse(dodsFile);
        }else{ 
            try {
                Class  comClass = Class.forName("org.enhydra.dods.Common");
                InputStream  inStream =comClass.getClassLoader().getResourceAsStream(DODS_CONF_JAR_PATH+"/"+"dodsConf.xml");
                DocumentBuilderFactory  factory = DocumentBuilderFactory.newInstance();
                DocumentBuilder  builder = factory.newDocumentBuilder();
                doc = builder.parse(inStream);      
            } catch (Exception  e) {
                e.printStackTrace();
            }
        }
        
        dodsConf = XMLConfig.newXMLConfigInstance(doc);
        dodsVendorConfs = new HashMap ();
        dodsDriversMap = new HashMap ();
        dodsDriversNamesVec = new Vector ();
        dodsVendorsNamesVec = new Vector ();
        dodsDriverClassVec  = new Vector ();
        XMLConfig vendor = null;
        String  vendorName = null;
        String  vendorFile = null;
        
        
        NodeList  vendors = dodsConf.getSubElementsByTagName("Database/Vendor");

        for (int i = 0; i < vendors.getLength(); i++) {
            vendor = (XMLConfig) vendors.item(i);
            vendorName = vendor.getAttribute("name");
            vendorFile = confDir + File.separator + vendor.getText();
            
            if (confDir != null) { 
                doc = XMLDocumentFactory.parse(vendorFile);
            }else{          
                try {
                    Class  comClass = Class.forName("org.enhydra.dods.Common");
                    InputStream  inStream =comClass.getClassLoader().getResourceAsStream(DODS_CONF_JAR_PATH+"/"+vendor.getText());
                    DocumentBuilderFactory  factory = DocumentBuilderFactory.newInstance();
                    DocumentBuilder  builder = factory.newDocumentBuilder();
                    doc = builder.parse(inStream);      
                } catch (Exception  e) {
                    e.printStackTrace();
                }
            }
            
            XMLConfig dodsVendorConf = XMLConfig.newXMLConfigInstance(doc);
            dodsVendorConfs.put(vendorName, dodsVendorConf);
        }


        XMLConfig driverN; 
        XMLConfig driver;
        NodeList  driverClassesNl;
        String  driverVendorName;    
        String  driverClassName;       
        String  driverMetadataName;   
        NodeList  driverVendorsNl = dodsConf.getSubElementsByTagName("Drivers/DatabaseVendor");
        for (int i = 0; i < driverVendorsNl.getLength(); i++) {
            driverN = (XMLConfig)driverVendorsNl.item(i);
            driverVendorName = driverN.getAttribute("name"); 
            driverClassesNl = driverN.getChildrenByTagName("Driver"); 
            for (int j = 0; j < driverClassesNl.getLength(); j++) {
                driverClassName = null;
                driverMetadataName = null;
                driver = (XMLConfig)driverClassesNl.item(j);
                driverClassName = driver.getAttribute("class");
                driverMetadataName = driver.getAttribute("name");
                if (driverClassName !=null && driverClassName!=""){
                    dodsDriversMap.put(driverClassName,driverVendorName);
                }
                if (driverMetadataName!=null && driverMetadataName!=""){
                    dodsDriverClassVec.add(driverClassName);
                    dodsDriversNamesVec.add(driverMetadataName); 
                    dodsVendorsNamesVec.add(driverVendorName);
                }
            }       
        }       
        configured = true;
    }

    /**
     * Get database vendor name for given driver class name.
     *
     * @param driverClassName Class name.
     * @return  Database Vendor name.
     */
    static public String  getDatabaseVendor(String  driverClassName) {    
        String  res=null;
        init();
        if(driverClassName!=null && dodsDriversMap.containsKey(driverClassName)){
            res = (String )dodsDriversMap.get(driverClassName);
        }else{
            System.out.println("Unknown JDBC Driver Vendor for : "+driverClassName+" using default (Standard) insted");
            res = "Standard";
        }
        return res;
    }

    /**
     * Get database vendor name for given driver class name.
     *
     * @param driverClassName Class name.
     * @return  Database Vendor name.
     */
    static public String  getDatabaseVendorFromDriverName(String  driverMetadataName)
    throws ConfigException {    
        init();
        int vendorIndex=-1;
        for(int i=0;i<dodsDriversNamesVec.size();i++){
            if (driverMetadataName.indexOf((String )dodsDriversNamesVec.elementAt(i))!=-1){
                String  res =(String )dodsVendorsNamesVec.elementAt(i);
                return (String )dodsVendorsNamesVec.elementAt(i);
            }
        }
        throw new ConfigException("Unknown JDBC Driver Vendor for : "+driverMetadataName); 
    }

    /**
     * Get database driver full class name for given driver name.
     *
     * @param driverClassName Class name.
     * @return  Database Vendor name.
     */
    static public String  getDatabaseDriverClassFromDriverName(String  driverMetadataName)
    throws ConfigException {    
        init();
        int vendorIndex=-1;
        for(int i=0;i<dodsDriversNamesVec.size();i++){
            if (driverMetadataName.indexOf((String )dodsDriversNamesVec.elementAt(i))!=-1){
                String  res =(String )dodsVendorsNamesVec.elementAt(i);
                return (String )dodsDriverClassVec.elementAt(i);
            }
        }
        throw new ConfigException("Unknown JDBC Driver for : "+driverMetadataName); 
    }
    

    /**
     * Get dods configuration.
     *
     * @return Dods configuration.
     */
    static public XMLConfig getDodsConf() {
        init();
        return dodsConf;
    }

    /**
     * Get dods conf property.
     *
     * @param key Key of the property.
     * @param database Used database.
     *
     * @return Dods conf property.
     */
    static public String  getDodsConfProperty(String  key, String  database) {
        init();
        XMLConfig vendor = (XMLConfig) dodsVendorConfs.get(database);

        if (vendor != null) {
            return vendor.getText(key);
        }
        return null;
    }

    /**
     * Show dods.config properties.
     */
    static public void showDodsConf() {
        init();
        StringBuffer  ret = new StringBuffer ();
        Iterator  iter = dods_conf.entrySet().iterator();

        while (iter.hasNext()) {
            Map.Entry  elm = (Map.Entry ) iter.next();

            ret.append(elm.getKey()).append("=").append(elm.getValue()).append("\n");
        }
    }

    /**
     * Get dods conf property.
     *
     * @return Dods conf vendor names.
     */
    static public Set  getDodsConfVendorNames() {
        init();
        return dodsVendorConfs.keySet();
    }

    /**
     * Get path to dods-ejen.properties.
     *
     * @return Path to dods-ejen.properties.
     */
    public static String  getDodsEjenPropertyFilename() {
        final String  SYS_USER_HOME = "user.home";
        final String  DIR_ENHYDRA = ".enhydra";
        final String  PROPERTY_FILENAME = "dods-ejen.properties";
        StringBuffer  buf = new StringBuffer ();

        buf.append(System.getProperties().getProperty(SYS_USER_HOME));
        buf.append(File.separator);
        buf.append(DIR_ENHYDRA);
        buf.append(File.separator);
        buf.append(PROPERTY_FILENAME);
        return buf.toString();
    }

    /**
     * Get used doml file.
     * @return Used doml file.
     */
    public static String  getDomlFile() {
        init();
        String  strDoml = null;

        strDoml = System.getProperty("DOML_FILE");
        String  domlFile = null;

        if (strDoml == null) {
            File  current = new File (".");
            String [] files = current.list();

            out:
            for (int i = 0; i < files.length; i++) {
                if (files[i].toLowerCase().endsWith(".doml")) {
                    domlFile = files[i];
                    break out;
                } else {
                    domlFile = null;
                }
            }
            strDoml = domlFile;
        }
        return strDoml;
    }

    /**
     * Get used doml file name.
     * @return Used doml file name.
     */
    public static String  getDomlFileName() {
        init();
        String  domlFileName = getDomlFile();
        String  fileSep = System.getProperty("file.separator");

        if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
            if (fileSep != "/") {
                domlFileName = domlFileName.replace('\\', '/');
            }
        }
        return domlFileName.substring(domlFileName.lastIndexOf("/") + 1);
    }

    /**
     * Get file name from used doml file name.
     * @return file name from used doml file name.
     */
    public static String  getFileName() {
        init();
        String  fileName = getDomlFileName();
        int index = fileName.lastIndexOf(".");

        if (index != -1) {
            fileName = fileName.substring(0, index);
        }
        return fileName;
    }

    /**
     * Get used databse vendor.
     * @return used databse vendor.
     */
    public static String  getDatabaseVendor() {
        init();
        String  database = System.getProperty("DATABASE_VENDOR");
        return database;
    }

    /**
     * Get used databse vendor.
     * @return used databse vendor.
     */
    public static String  getSplitSQLPrimary(){
        String  ret="true";
        try{    
            ret=Common.getDodsConfProperty("SplitSQLPrimary",getDatabaseVendor());
            
        }catch(Exception  e){}
        if (ret==null) return "true";
        else return ret;
        
    }



    /**
     * Set used doml file.
     * @param doml Used doml file.
     */
    public static void setDomlFile(String  doml) {
        domlfile = doml;
    }

    /**
     * Set used configuration folder.
     * @param confD Used configurationfolder.
     */
    public static void setConfigDir(String  confD) {
        if (confD != null) { 
            File  tempFile = new File (confD + File.separator + "dodsConf.xml");  
            if (!tempFile.isFile()) {           
                    System.out.println("File dodsConf.xml not exists on path: '"
                   + confD + "' using default insted");
                   confD = getDefaultConfigDir();
            }
            configDir = confD; 
        } else {
//               System.out.println("Using default configuration!");
              configDir = getDefaultConfigDir();
        }
        configured = false; 
        
    }
    
    /**
     * Get default configuration folder.
     * 
     */
    public static String  getDefaultConfigDir() {
        String  dodsHome = System.getProperty("DODS_HOME");
        if (dodsHome == null) {
            dodsHome = System.getProperty("enhydra.home");
            if (dodsHome != null) {
                dodsHome = dodsHome + File.separator + "dods";
            } else {
                return null;
            }
        }
        String   resultStr = dodsHome + File.separator + "build" + File.separator
                + "conf";
        return resultStr; 
    }
    
    /**
     * Get used configuration folder.
     * 
     */
    public static String  getConfigDir() {
        init(); 
        String  resultStr = configDir;  

        if (configDir == null) {
            resultStr = getDefaultConfigDir();
        }
        return resultStr; 
    }

    /**
     * Get configuration file from URL as inputStream.
     * @param confURL Aditional search path for configuration file (folder or *.jar).
     * If null then system classPath is used
     * @param confFile configuration file name (from folder or *.jar file). 
     * If null then try to get default configuraton file from org/enhydra/dods/conf/databaseManager.conf
     * @return InputStream with configuration file.
     * If can not found confFile or confFile==null then try to get default.
     */
    public static InputStream  getConfFileFromURL(URL  confURL, String  confFile)
        throws ConfigException {
        URL [] cURL;
        if (confURL!=null){  
            cURL= new URL [1];
            cURL[0]=confURL;
        }else{
            cURL= new URL [0];
        }
    //  URLClassLoader ucl = new URLClassLoader(cURL,Common.class.getClassLoader());
 ClassLoader  ucl = Common.class.getClassLoader();
            
        InputStream  configIS = null;
        if (confFile!=null){
            try{
                configIS = ucl.getResourceAsStream(confFile);
            }catch(Exception  e){}           
        }
        if (configIS==null){
            try{
                configIS = ucl.getResourceAsStream(DATABASE_MANAGER_CONF_FILE);
            }catch(Exception  e){
                throw new ConfigException(e);  
            }
        }
        return configIS;
    }



    /**
     * Set Custom Template configuration folder.
     * @param dirString path to template folder
     */
    public static void setCustomTemplateDir(String  dirString) {
        customTemplateDir = dirString;
    }

    /**
     * Get Custom Template configuration folder.
     * 
     */
    public static String  getCustomTemplateDir() {
        init();     
        return customTemplateDir;
    }

    /**
     * Get used project root.
     * @return Project root.
     */
    public static String  getProjectRoot() {
        String  projectRoot = null;

        projectRoot = System.getProperty("PROJECT_ROOT");
        if (projectRoot == null) {
            projectRoot = ".";
        }
        return projectRoot;
    }

    /**
     * Set used project root.
     * @param projectRoot Project root.
     */
    public static void setProjectRoot(String  projectRoot) {
        projRoot = projectRoot;
    }

    /**
     * Get used template set.
     * @return template set.
     */
    public static String  getTemplateSet() {
        String  tempSet = System.getProperty("TEMPLATESET");

        if (tempSet == null) {
            tempSet = "standard";
        }
        if (tempSet.equals("multidb")) {
            tempSet = "standard";
        }
        if (tempSet.equals("webdocwf")) {
            tempSet = "webdocwf";
        }
        if (tempSet.equals("multidb_webdocwf")) {
            tempSet = "webdocwf";
        }
        return tempSet;
    }

    /**
     * Set used template set.
     * @param set Template set.
     */
    public static void setTemplateSet(String  set) {
        templateSet = set;
    }

    /**
     * Get used extensions (template set).
     * @return extensions (template set).
     */
    public static String  getExtensions() {
        String  tempSet = null;

        tempSet = System.getProperty("TEMPLATESET");
        if (tempSet == null) {
            tempSet = "standard";
        }
        return tempSet;
    }

    /**
     * Get force parameter.
     * @return Force parameter.
     */
    public static String  getForce() {
        String  force = System.getProperty("FORCE");

        return force;
    }

    /**
     * Get install properties file.
     * @return Get dods.properties file.
     */
    public static String  getInstallProperties() {
        String  dodsHome = null;
        String  home     = null;
        try {
            Properties  properties = new Properties ();
            String  strFileSep = System.getProperty("file.separator");
            home = System.getProperty("DODS_HOME");
            dodsHome = home + strFileSep + "build"+ strFileSep + "dods.properties";
            if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                if (dodsHome != "/") {
                    dodsHome = dodsHome.replace('/', '\\');
                }
            }
            if (home == null) {
                dodsHome = System.getProperty("ENHYDRA_DIR") + strFileSep
                        + "dods.properties";
                if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                    if (dodsHome != "/") {
                        dodsHome = dodsHome.replace('/', '\\');
                    }
                }
            }
        } catch (NullPointerException  nullpointerexception) {}
        return dodsHome;
    }

    /**
     * Get enhydra root.
     * @return Enhydra root.
     */
    public static String  getDODSRoot() {
        String  enhHome = null;

        try {
            String  strFileSep = System.getProperty("file.separator");

            enhHome = System.getProperty("DODS_HOME");
            if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                if (enhHome != "/") {
                    enhHome = enhHome.replace('\\', '/');
                }
            }
            if (enhHome == null) {
                String  strInstall = getInstallProperties();

                if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                    if (System.getProperty("file.separator") != "/") {
                        strInstall = strInstall.replace('/', '\\');
                    }
                }
                FileInputStream  fisInstallProp = new FileInputStream (strInstall);
                Properties  properties = new Properties ();

                properties.load(fisInstallProp);
                enhHome = properties.getProperty("enhydra.root");
                if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                    if (enhHome != "/") {
                        enhHome = enhHome.replace('/', '\\');
                    }
                }
            }
        } catch (NullPointerException  nullpointerexception) {} catch (IOException  ioexception) {}
        return enhHome;
    }

    /**
     * Get install properties file.
     * @return Install properties file.
     */
    public static String  getInstallPropertiesParam() {
        String  enhydraRoot = null;
        String  dodsConf = null;
        String  strReturn = null;

        init();
        try {
            Properties  properties = new Properties ();
            String  strFileSep = System.getProperty("file.separator");
            String  strInstall = getInstallProperties();

            if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                if (strFileSep != "/") {
                    strInstall = strInstall.replace('/', '\\');
                }
            }
            FileInputStream  fisInstallProp = new FileInputStream (strInstall);

            properties.load(fisInstallProp);
            enhydraRoot = properties.getProperty("enhydra.root");
            if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                if (enhydraRoot != "/") {
                    enhydraRoot = enhydraRoot.replace('/', '\\');
                }
            }
            dodsConf = properties.getProperty("dods.conf");
            if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                if (dodsConf != "/") {
                    dodsConf = dodsConf.replace('/', '\\');
                }
            }
            fisInstallProp.close();
        } catch (IOException  ioexception) {} catch (NullPointerException  nullpointerexception) {}
        strReturn = "<property name=\"enhydra.root\" value=\"" + enhydraRoot
                + "\"/>\n";
        strReturn = strReturn + "<property name=\"dods.conf\" value=\""
                + dodsConf + "\"/>\n";
        return strReturn;
    }

    /**
     * Get template directory.
     * @return Template directory.
     */
    public static String  getTemplateDir() {   
        String  templateDir = null;

        init();
        try {
            Properties  properties = new Properties ();
            String  strFileSep = System.getProperty("file.separator");
            String  strInstall = getInstallProperties();

            if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                if (strFileSep != "/") {
                    strInstall = strInstall.replace('/', '\\');
                }
            }
            FileInputStream  fisDodsEjen = new FileInputStream (strInstall);

            properties.load(fisDodsEjen);
            fisDodsEjen.close();
            templateDir = properties.getProperty("template.dir");
            if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
                if (templateDir != "/") {
                    templateDir = templateDir.replace('/', '\\');
                }
            }
        } catch (IOException  ioexception) {} catch (NullPointerException  nullpointerexception) {return null;}
        return templateDir;
    }

    /**
     * Get all template set directories.
     * @return all template set directories.
     */
    public static HashSet  getAllTemplateSets() {
        HashSet  dirs = new HashSet ();

        init();
        try {
            File  templateDirs = new File (Common.getTemplateDir());
            String [] temp = templateDirs.list((FilenameFilter ) new DirectoryNameFilter());

            dirs.add("standard");
            for (int i = 0; i < temp.length; i++) {
                dirs.add(temp[i]);
            }
        } catch (NullPointerException  nullpointerexception) {}
        return dirs;
    }

    /**
     *  Transform specify name in capital letters.
     *
     *  @param name Class member name (column table name).
     *
     *  @return Capitalized name.
     */
    static public String  capitalizeName(String  name) { // throws Exception {
 String  otherLetters = name.substring(1);

        return name.toUpperCase(java.util.Locale.ENGLISH).substring(0, 1).concat(otherLetters);
    }

    /**
     *  Transform name in upper case name.
     *
     *  @param name Class member name (column table name).
     *
     *  @return upper case name.
     */
    static public String  upperCaseName(String  name) {
        return name.toUpperCase();
    }

    
    
    /**
     * Replace all occurence of forReplace with replaceWith in input string.
     * @param input represents input string
     * @param forReplace represents substring for replace
     * @param replaceWith represents replaced string value
     * @return new string with replaced values
     */
    public static String  replaceAll(String  input, String  forReplace,
                                    String  replaceWith) {
      if( input == null ) return null;
      StringBuffer  result = new StringBuffer ();
      boolean hasMore = true;
      while (hasMore) {
          int start = input.indexOf(forReplace);
          int end = start + forReplace.length();
          if (start != -1) {
             result.append(input.substring(0, start) + replaceWith);
             input = input.substring(end);
          } else {
              hasMore = false;
              result.append(input);
          }
      }
      if (result.toString().equals("")) return input; //nothing is changed
 else return result.toString();
    }   

    
    public static void main(String [] args) {
        showDodsConf();
        System.out.println("Database.OidJdbcType = "
                + getDodsConfProperty("Database.OidJdbcType", "Standard"));
    }
}
