
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: CoreDO.java,v 1.3 2005/05/26 08:08:10 predrag Exp $
 */
package com.lutris.appserver.server.sql;

import java.io.Serializable ;
import java.sql.PreparedStatement ;
import java.sql.ResultSet ;
import java.sql.SQLException ;

import com.lutris.dods.builder.generator.query.DataObjectException;

/**
 * Base class from which the data objects are derived.
 *
 * @version $Revision: 1.3 $
 * @author Kyle Clark
 */
abstract public class CoreDO implements Transaction, Serializable  {
    private static boolean transactionCheck = false;
    private static boolean deleteCheckVersion = false;
    private static boolean autoSave = false;
    public static boolean versioning = true;

    /**
     * Disable use of the version column.
     *
     * author    Jay Gunter
     */
    public static void disableVersioning() {
        versioning = false;
    }
    // True if this object exists in the database.
 protected boolean persistent = false;
    protected CoreDataStruct originalData = null;

    /**
     * Public constructor.
     */
    public CoreDO() {
        persistent = false;
    }

    /**
     * Public constructor.
     * @param rs a result set.
     * @exception SQLException if an error occurs while instantiating this
     * object from the result set.
     * @exception ObjectIdException if the object id for this object is
     * invalid.
     */
    public CoreDO(ResultSet  rs) throws SQLException , ObjectIdException {
        persistent = true;
        set_OId(new ObjectId(rs.getBigDecimal(get_OIdColumnName())));
        if (versioning) {
            set_Version(rs.getInt(get_versionColumnName()));
        }
    }

    /**
     * Returns the object identifier column name.
     * @return the object identifier column name.
     */
    public static String  get_OIdColumnName() {
        return CoreDataStruct.get_OIdColumnName();
    }

    /**
     * Returns the object identifier column name.
     * @return the object identifier column name.
     * @deprecated use get_OIdColumnName()
     */
    public static String  getOIdColumnName() {
        return get_OIdColumnName();
    }

    /**
     * Sets the object identifier column name.
     * @param _oidColumnName Name of object identifier column.
     */
    public static void set_OIdColumnName(String  _oidColumnName) {
        CoreDataStruct.set_OIdColumnName(_oidColumnName);
    }

    /**
     * Sets the object identifier column name.
     * @param _oidColumnName Name of object identifier column.
     * @deprecated use get_versionColumnName()
     */
    public static void setOIdColumnName(String  _oidColumnName) {
        set_OIdColumnName(_oidColumnName);
    }

    /**
     * Returns the version column name.
     * @return the version column name.
     */
    public static String  get_versionColumnName() {
        return CoreDataStruct.get_versionColumnName();
    }

    /**
     * Returns the version column name.
     * @return the version column name.
     * @deprecated use get_versionColumnName()
     */
    public static String  getVersionColumnName() {
        return get_versionColumnName();
    }

    /**
     * Sets the version column name.
     * @param _versionColumnName Name of version column.
     */
    public static void set_versionColumnName(String  _versionColumnName) {
        CoreDataStruct.set_versionColumnName(_versionColumnName);
    }

    /**
     * Sets the version column name.
     * @param _versionColumnName Name of version column.
     * @deprecated use set_versionColumnName()
     */
    public static void setVersionColumnName(String  _versionColumnName) {
        set_versionColumnName(_versionColumnName);
    }

    /**
     * @deprecated Use get_OId()
     * @return this object's identifier.
     */
    public ObjectId getOId() {
        return get_OId();
    }

    /**
     * Returns this object's identifier.
     * @return this object's identifier.
     */
    public ObjectId get_OId() {
        if (originalData != null) {
            return originalData.get_OId();
        } else {
            return null;
        }
    }

    /**
     * Sets this object's identifier.
     * @deprecated Use set_OId()
     * @param oId this object's identifier.
     */
    protected void setOId(ObjectId oId) {
        set_OId(oId);
    }

    /**
     * Sets this object's identifier.
     * @param oId this object's identifier.
     */
    protected void set_OId(ObjectId oId) {
        if (originalData != null) {
            originalData.set_OId(oId);
        }
    }

    /**
     * @deprecated Use set_Version()
     * @param version the object's version.
     */
    protected void setVersion(int version) {
        set_Version(version);
    }

    /**
     * Sets this object's version.
     * @param version the object's version.
     */
    protected void set_Version(int version) {
        if (originalData != null) {
            originalData.set_Version(version);
        }
    }

    /**
     * @deprecated Use get_Version()
     * @return this object's version.
     */
    protected int getVersion() {
        return get_Version();
    }

    /**
     * Returns this object's version.
     * @return this object's version.
     */
    protected int get_Version() {
        if (originalData != null) {
            return originalData.get_Version();
        } else {
            return 0;
        }
    }

    /**
     * Sets this object's new version number.
     * @param newVersion this object's next version.
     * @deprecated use set_NewVersion instead
     */
    protected void setNewVersion(int newVersion) {
        if (originalData != null) {
            originalData.newVersion = newVersion;
        }
    }

    /**
     * Sets this object's new version number.
     * @param newVersion this object's next version.
     */
    protected void set_NewVersion(int newVersion) {
        if (originalData != null) {
            originalData.newVersion = newVersion;
        }
    }

    /**
     * Returns this object's new version.
     * @return this object's new version.
     * @deprecated use get_NewVersion instead
     */
    protected int getNewVersion() {
        if (originalData != null) {
            return originalData.newVersion;
        } else {
            return 0;
        }
    }

    /**
     * Returns this object's new version.
     * @return this object's new version.
     */
    protected int get_NewVersion() {
        if (originalData != null) {
            return originalData.newVersion;
        } else {
            return 0;
        }
    }

    /**
     *
     * @return true if this data object has been inserted into
     * the database.
     *
     * Returns true if this object is persistent in the database
     * (i.e. it has been inserted and not deleted)
     *
     */
    public boolean isPersistent() {
        return this.persistent;
    }

    /**
     * Sets the persistent state for this object.
     * @param persistent true if this object is present in
     *   persistent store.
     */
    public void setPersistent(boolean persistent) {
        this.persistent = persistent;
    }

    /**
     * Returns the statement that can be used to insert this
     * object into the database.
     *
     * @param conn the database connection.
     * @return the statement used to insert this object into the
     *   database.  null if the object cannot be inserted.
     * @exception java.sql.SQLException If an error occurs.
     */
    public abstract PreparedStatement  getInsertStatement(DBConnection conn)
        throws SQLException ;

    /**
     * Returns the statement that can be used to update this
     * object in the database.
     *
     * @param conn the database connection.
     * @return the statement used to update this object in the
     *   database.  null if the object cannot be updated.
     * @exception java.sql.SQLException If an error occurs.
     */
    public abstract PreparedStatement  getUpdateStatement(DBConnection conn)
        throws SQLException ;

    /**
     * Returns the statement that can be used to delete this
     * object from the database.
     *
     * @param conn the database connection.
     * @return the statement used to delete this object from the
     *   database.  null if the object cannot be deleted.
     * @exception java.sql.SQLException If an error occurs.
     */
    public abstract PreparedStatement  getDeleteStatement(DBConnection conn)
        throws SQLException ;

    /**
     * Inserts this object into the database.
     *
     * @param conn the database connection.
     * @exception java.sql.SQLException if a database access error occurs.
     * @exception DBRowUpdateException If a version error occurs.
     */
    public synchronized void executeInsert(DBConnection conn)
        throws SQLException , DBRowUpdateException {
        PreparedStatement  stmt;

        if (persistent) {
            executeUpdate(conn);
        } else {
            stmt = getInsertStatement(conn);
            if (stmt != null) {
                try {
                    conn.executeUpdate(stmt, "executeInsert");
                } catch (SQLException  e) {
                    throw e;
                } finally{
                    if(((ExtendedDBConnection)conn).getMaxPreparedStmts()==0){
                      stmt.close();
                    }
                    //stmt.close();
                }
            }
            persistent = true;
        }
    }

    /**
     * Updates the persistent state.
     *
     * @param success true if the transaction succeeded
     *   and this object was successfully inserted into the database.
     */
    public void finalizeInsert(boolean success) {
        persistent = success;
        /*
         if (versioning) {
         if (success) {
         set_Version(get_NewVersion());
         } else {
         set_NewVersion(get_Version());
         }
         }
         */
    }

    /**
     * Updates the contents of this object in the database.
     *
     * @param conn the database connection.
     * @exception java.sql.SQLException If a database access error occurs.
     * @exception DBRowUpdateException If a version error occurs.
     */
    public synchronized void executeUpdate(DBConnection conn)
        throws SQLException , DBRowUpdateException {
        boolean mustUpdate = (get_NewVersion() == get_Version());
        /*
         if (mustUpdate && versioning) {
         set_NewVersion(get_Version() + 1);
         }
         */
        PreparedStatement  stmt = getUpdateStatement(conn);

        if (stmt != null) {
            try{
                if ((conn.executeUpdate(stmt, "execute update") == 0) && mustUpdate) {
                    // Zero rows were affected by the update.
 // This usually occurs when the version number does not match
 // the version number for this oId in the database table.
 // We verify that this is the case, and throw an appropriate
 // exception.
                    stmt.close();
                    stmt=null;
                    seeWhatHappened(conn,"Update");
                }
            } catch (SQLException  e) {
                throw e;
            } finally{
                if(((ExtendedDBConnection)conn).getMaxPreparedStmts()==0 && (stmt!= null)){
                    stmt.close();
                  }
                  //stmt.close();
            }
        }
    }

    private void seeWhatHappened(DBConnection conn,               
                                 String  action)
        throws SQLException , DBRowUpdateException {
        ResultSet  rs=null;
        PreparedStatement  stmt=null;
        try {
            String  table = this.getTableName();

            stmt = conn.prepareStatement("select " + get_versionColumnName()
                                             + " from " + table + " where " + get_OIdColumnName() + " = "
                                             + get_OId());
            rs = stmt.executeQuery();

            if (false == rs.next()) {
                throw new DBRowUpdateException(action + " failed: Table "
                                                   + table + " contains no row with id=" + get_OId());
            }
            int v = rs.getInt(1);

            if (get_Version() != v) {
                throw new DBRowUpdateException(action + " failed: Table "
                                                   + table + " id=" + get_OId() + " has version " + v
                                                   + " where object has version " + get_Version()
                                                   + (versioning ? "." : ", and versioning is disabled!"));
            }
            throw new SQLException (action + " failed, but Table " + table
                                       + " id=" + get_OId() + " does exist with version=" + v);
        } catch (DBRowUpdateException re) {
            throw re;
        } catch (SQLException  se) {
            throw se;
        } catch (Exception  e) {
            throw new SQLException ("Error determining cause of update failure.",
                                   e.getMessage());
        }finally{
            if (rs!=null){
                rs.close();
            }
            if(stmt!=null){
                stmt.close();
            }
        }
    }

    /**
     * Return the name of the table whose rows represent these objects.
     * This method should be overridden by derived classes.
     *
     * @return the name of the table.
     * @see #executeUpdate
     */
    protected String  getTableName()throws SQLException  {
        // FIXME: really should be a specific error
 throw new SQLException ("getTableName() not implemented. DO classes need to be regenerated");
    }

    /**
     * The version number of this object is set to the next
     * version for this object if
     * this object was successfully updated in the database.
     *
     * @param success true if the transaction succeeded
     *   and this object was successfully updated in the database.
     * @see #get_NewVersion
     */
    public void finalizeUpdate(boolean success) {
        /*
         if (versioning) {
         if (success) {
         set_Version(get_NewVersion());
         } else {
         set_NewVersion(get_Version());
         }
         }
         */
    }

    /**
     * Deletes this object from the database.
     *
     * @param conn Database connection.
     * @exception java.sql.SQLException If a database access error
     *  occurs.
     */
    public void executeDelete(DBConnection conn)
        throws SQLException  {
        PreparedStatement  stmt = getDeleteStatement(conn);

        if (stmt != null) {
            try{
                if (conn.executeUpdate(stmt, "execute delete") == 0) {
                    try {
                        stmt.close();
                        stmt=null;
                        seeWhatHappened(conn, "Delete");
                    } catch (DBRowUpdateException de) {
                        SQLException  sqle = new SQLException (de.toString());
                        throw sqle;
                    }
                }
            }catch( SQLException  e){
                throw e;
            }finally{
                if(((ExtendedDBConnection)conn).getMaxPreparedStmts()==0 && (stmt!=null)){
                    stmt.close();
                  }
                  //stmt.close();
            }
        }
    }

    /**
     * Currently does nothing.
     *
     * @param success true if the transaction succeeded
     *   and this object was successfully deleted from the
     *   database.
     */
    public void finalizeDelete(boolean success) {
        if (persistent) {
            persistent = !success;
        }
    }

    /**
     * This method is invoked whenever an object original data is needed. This
     * method returns null and should be overwritten by subclasses.
     */
    public Object  originalData_get() {
        return null;
    }

    /**
     * This method is invoked whenever an object is inserted or updated in the
     * cache. This is empty method and should be overwritten by subclasses.
     */
    public void updateCache() {}

    /**
     * This method is invoked whenever an object is deleted from the cache.
     * This is empty method and should be overwritten by subclasses.
     */
    public void deleteFromCache() {}

    /**
     * This method is invoked whenever an object is added to the cache.
     * This is empty method and should be overwritten by subclasses.
     */
    public void addToCache() {}

    /**
     * This method is invoked whenever an object is removed from the cache.
     * This is empty method and should be overwritten by subclasses.
     */
    public void evict() {}

    /**
     * This method is invoked whenever object's data needs to be loaded.
     * This is empty method and should be overwritten by subclasses.
     *
     * @exception DataObjectException If a data access error occurs.
     */
    public void refresh() throws DataObjectException {}

    /**
     *
     */
    public int getOriginalVersion() {
        if (originalData != null) {
            return originalData.get_Version();
        } else {
            return 0;
        }
    }

    /**
     *
     */
    public void makeVisible() {}

    /**
     *
     */
    public void makeInvisible() {}

    /**
     *
     */
    public void executeLockingStatement(DBConnection conn) throws SQLException  {}

    /**
     *
     */
    abstract public void dumpData(boolean incrementVersion);
}

