
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache;

import java.util.ArrayList ;
import java.util.Collection ;
import com.lutris.dods.builder.generator.dataobject.GenericDO;
import com.lutris.appserver.server.sql.CoreDataStruct;

/**
 *
 * This interface stores one query and its necessary data, for query array.
 *
 * @author    Tanja Jovanovic
 * @author    Nenad Vico
 * @version   2.0  15.06.2003.
 *
 */
public interface QueryCacheItem {

    /**
     * Returns query id (String it the form:
     * "query_database_name.String_presentation_of_query").
     *
     * @return Query id.
     */
    public String  getQueryId();

    /**
     * Sets query id (String it the form:
     * "query_database_name.String_presentation_of_query").
     *
     * @param queryId <code>String</code> that unique represents query.
     */
    public void setQueryId(String  queryId);

    /**
     * Returns OIds (Collection of object IDs which are results of the query).
     *
     * @return Collection of object IDs which are results of the query.
     */
    public Collection  getOIds();

    /**
     * Returns number of cached query results.
     *
     * @return Number of cached query results.
     */
    public int getResultNum();

    /**
     * Returns true if all query results are cached, otherwise false.
     *
     * @return true if all query results are cached, otherwise false.
     */
    public boolean isCompleteResult();

    /**
     * Sets new boolean value about the cached query results (true if all query
     * results are cached, otherwise false).
     *
     * @param newCompleteRes true if all query results are cached, otherwise
     * false.
     */
    public void setCompleteResult(boolean newCompleteRes);

    /**
     * Returns true if there have been performed inserts, updates or deletes  
     * concerning results of this query, otherwise false.
     *
     * @return true if there have been performed inserts, updates or deletes  
     * concerning results of this query, otherwise false.
     */
    public boolean isModifiedQuery();

    /**
     * Sets flag coccerning information whether the query is modified.
     *
     * @param mod  true if there have been performed inserts, updates or deletes
     * concerning results of this query, otherwise false.
     */
    public void setModifiedQuery(boolean mod);
    
    /**
     * Returns time needed for query execution.
     *
     * @return Time needed for query execution.
     */
    public int getTime();

    /**
     * Sets time needed for query execution.
     *
     * @param time Time needed for query execution.
     */
    public void setTime(int time);

    /**
     * Returns array of query conditions conds.
     *
     * @return Array of query conditions.
     */
    public ArrayList  getConds();

    /**
     * Sets array of query conditions.
     *
     * @param conds Array of query conditions.
     */
    public void setConds(ArrayList  conds);

    /**
     * Adds condition to query.
     *
     * @param cond <code>Condition</code> that will be added to query.
     */
    public void addCond(Condition cond);

    /**
     * Returns query database.
     *
     * @deprecated Use get_OriginDatabase()
     * @return Query database.
     */
    public String  getOriginDatabase();

    /**
     * Returns query database.
     *
     * @return Query database.
     */
    public String  get_OriginDatabase();

    /**
     * Checks whether data object obj satisfies conditions of this query.
     *
     * @param obj Data object for which are checked conditions of this query.
     * @return true if data object obj satisfies conditions of this query,
     * otherwise false.
     */
    public boolean checkConditions(GenericDO obj);

    /**
     * Checks whether DataStruct object obj satisfies conditions of this query.
     *
     * @param obj DataStruct object for which are checked conditions of this
     * query.
     * @return true if DataStruct object obj satisfies conditions of this query,
     * otherwise false.
     */
    public boolean checkConditions(CoreDataStruct obj);

    /**
     * Inserts data object obj (or updates it if already exists) in array DOs,
     * if it satisfies this query.
     *
     * @param obj Data object which may be inserted (or updated) in array DOs.
     */
    public void update(GenericDO obj);

    /**
     * Inserts DataStruct object obj (or updates it if already exists) in array
     * DOs, if it satisfies this query.
     *
     * @param obj DataStruct object which may be inserted (or updated) in array
     * DOs.
     */
    public void update(CoreDataStruct obj);

    /**
     * Removes data object obj from array DOs, if present.
     *
     * @param obj Data object which will be removed from array DOs.
     */
    public void delete(GenericDO obj);

    /**
     * Removes DataStruct object obj from array DOs, if present.
     *
     * @param obj DataStruct object which will be removed from array DOs.
     */
    public void delete(CoreDataStruct obj);

    /**
     * Adds data object obj to array DOs.
     *
     * @param obj Data object which will be added to array DOs.
     */
    public void add(GenericDO obj);

    /**
     * Adds DataStruct object obj to array DOs.
     *
     * @param obj DataStruct object which will be added to array DOs.
     */
    public void add(CoreDataStruct obj);
}
