
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache;

import java.util.Map ;
import org.enhydra.dods.cache.ConfigurationAdministration;
import com.lutris.dods.builder.generator.dataobject.GenericDO;
import org.enhydra.dods.exceptions.CacheObjectException;

/**
 * This abstract class contains data and mechanisms needed for caching data
 * objects (or DataStruct objects) by their OIDs and provides cache
 * configuration and administration.
 *
 * @author    Tanja Jovanovic
 * @author    Sinisa Milosevic
 * @version   1.0  05.08.2003.
 *
 */
public abstract class DOCache extends ConfigurationAdministration {

    /**
     * Creates DOCache instance.
     *
     * @return created data cache instance (for storing data (or DataStruct)
     * objects.
     */
    public abstract DOCache newInstance() throws CacheObjectException;

    /**
     * Returns cache (data or DataStruct) content.
     *
     * @return Cache content as <code>Map</code> of data objects (or DataStruct
     * objects).
     */
    public abstract Map  getCacheContent();

    /**
     * Returns information if multi databases are supported.
     *
     * @return true if multi databases are used, otherwise false.
     */
    public abstract boolean isMulti();

    /**
     * Checks wheather cache reconfiguration needs to be done.
     *
     * @return true if cache reconfiguration needs to be done, otherwise false.
     */
    public abstract boolean toReconfigure();

    /**
     * Adds DO (data object) to the cache.
     *
     * @param newDO Data object that will be added to the cache.
     *
     * @return Added data object.
     */
    public abstract GenericDO addDO(GenericDO newDO);

    /**
     * Removes DO (data object) from the cache.
     *
     * @param DO Data object that will be removed from the cache.
     *
     * @return Removed data object, or <tt>null</tt> if there was no object
     * removed from the cache.
     */
    public abstract GenericDO removeDO(GenericDO DO);

    /**
     * Removes DO (data object) from the cache.
     *
     * @param handle Cache handle of DO (data object) that will be removed from
     * the cache. The form of cache handle is:
     * "<database_name>.<String_presentation_of_oid>".
     *
     * @return Removed data object, or <tt>null</tt> if there was no object
     * removed from the cache.
     */
    public abstract GenericDO removeDO(String  handle);

    /**
     * Updates cached DO, or inserts DO in the cache if it didn't exist in the
     * cache.
     *
     * @param DO Data object that will be updated (or inserted if didn't
     * exist in the cache).
     *
     * @return Updated or inserted data object.
     */
    public abstract GenericDO updateDO(GenericDO DO);

    /**
     * Deletes DO from the cache.
     *
     * @param DO Data object that will be deleted from the cache.
     *
     * @return Deleted data object, or <tt>null</tt> if there was no object
     * deleted from the cache.
     */
    public abstract GenericDO deleteDO(GenericDO DO);

    /**
     * Returns data object whose String representation of OID is parameter handle.
     *
     * @param handle String representation of OID of object that is being
     * searched in the cache.
     *
     * @return Data object whose String representation of OID is handle.
     */
    public abstract GenericDO getDOByHandle(String  handle);
}
