
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: StandardDBQuery.java,v 1.3 2005/05/26 08:08:10 predrag Exp $
 */
package com.lutris.appserver.server.sql.standard;

import java.sql.ResultSet ;
import java.sql.SQLException ;

import org.enhydra.dods.DODS;

import com.lutris.appserver.server.sql.DBConnection;
import com.lutris.appserver.server.sql.DBQuery;
import com.lutris.appserver.server.sql.ExtendedQuery;
import com.lutris.appserver.server.sql.ObjectIdException;
import com.lutris.appserver.server.sql.Query;
import com.lutris.logging.Logger;
import com.lutris.util.FatalExceptionError;

/**
 * Standard implementation of a DBQuery object.
 *
 * @see         DBQuery
 * @author      Kyle Clark
 * @since       LBS1.7
 * @version     $Revision: 1.3 $
 */
public class StandardDBQuery implements DBQuery {

    /**
     * Identifier for this query object.
     */
    private int id;

    /**
     * Next available identifier.
     */
    private static int nextId;

    /**
     * Connection used by this query object.
     */
    private DBConnection conn;

    /**
     * Result set after performing a query.
     */
    private ResultSet  resultSet = null;

    /**
     * The interface via which the query will be
     * executed.
     */
    private Query queryInterface = null;

    /**
     * Is this object still active, or has it been released.
     */
    private boolean released = false;
    private boolean releaseConnection = true;

    /**
     * The log channel.
     */
    //    private LogChannel channel;

    /**
     * Construct a query object for use on the supplied dB connection.
     *
     * @param dbConnection
     *   The database connection to use.
     * @exception SQLException
     *   If a database access error occurs.
     */
    protected StandardDBQuery(DBConnection dbConnection)
        throws SQLException  {
        id = nextId++;
        logDebug("new instance");
        conn = dbConnection;
    }

    /**
     * Query the database.
     *
     * @param q
     *   Query interface via which the query will be executed.
     * @exception SQLException
     *   If a database access error occurs.
     */
    public synchronized void query(Query q)
        throws SQLException  {
        logDebug("execute query");
        validate();
        conn.incrRequestCount();
        queryInterface = q;
        try {
            if (resultSet != null) {
                resultSet.close();
            }
            resultSet = queryInterface.executeQuery(conn);
        } catch (SQLException  e) {
            handleException(e);
            throw e;
        }
    }

    /**
     * Returns a new object representing the next result form
     * the query.
     *
     * @return
     *   New instance of object representing query results.
     *   <I>null</I> is returned when there are no more results.
     * @exception SQLException
     *   If a database access error occurs.
     * @exception ObjectIdException
     *   If ObjectId was not found.
     */
    public Object  next() throws SQLException , ObjectIdException {
        logDebug("get next result");
        validate();
        try {
            return queryInterface.next(resultSet);
        } catch (SQLException  e) {
            handleException(e);
            throw e;
        }
    }

    /**
     * Frees all resources consumed by this query.
     * Connections are returned to the connection pool.
     * Subsequent queries via this object,
     * will throw an exception.
     */
    public synchronized void release() {
        logDebug("release");
        try {
            validate();
        } catch (SQLException  except) {
            throw new FatalExceptionError(except);
        }
        try {
            //
 // Free all resources consumed by this connection
 // and return the connection to the connection
 // pool.
 //
 // NOTE: if a connection cannot handle and exception,
 // then it is automatically dropped from the connection
 // pool and closed.   This means that the connection
 // should no longer be used.
 //
 SQLException  sqlEx = null;

            try {
                java.sql.Statement  stmt=null;
                try {
                    if (resultSet != null) {
                        resultSet.close();
                    }
                }catch(SQLException  sqle) {
                    logDebug(sqle.toString());
                }
                stmt = ((ExtendedQuery)queryInterface).getStatement();
                if (stmt!=null){
                    stmt.close();
                }
            } catch (SQLException  e) {
                conn.handleException(e);
            }
            try {
                if (releaseConnection) {
                    conn.reset();
                }
            } catch (SQLException  e) {
                conn.handleException(e);
            }
            //
 // Release the connection.
 //
 if (releaseConnection) {
                conn.release();
            }
        }
        finally {
            released = true;
            resultSet = null;
            queryInterface = null;
            conn = null;
        }
    }

    /**
     * Exception handler.  This object is should not be
     * used for subsequent queries if this method returns
     * false.
     *
     * @return
     *   True if the exception can be handled and the object is
     *   still valid, false otherwise.
     */
    public synchronized boolean handleException(SQLException  e) {
        logDebug("handle exception");
        return conn.handleException(e);
    }

    /**
     * Method to ensure this object is still valid.
     * Once this object has been released it cannot be
     * used any more.
     *
     * @exception SQLException
     *   If a database access error occurs.
     */
    public void validate() throws SQLException  {
        if (released) {
            throw new SQLException ("Cannot access this object "
                                       + "once it has been released.");
        }
    }

    /**
     * Finalizer.
     * If this object has not been <A HREF=#release>released</A>,
     * this method ensures that garbage collection does so.
     */
    protected void finalize() {
        if (!released) {
            release();
        }
    }

    /**
     * Logging. For debuging only, since it effects all Query objects.
     *
     * @param str
     *   The data to log.
     */
    protected void logDebug(String  str) {
        if (DODS.getDatabaseManager().debug) {
            DODS.getLogChannel().write(Logger.DEBUG, "DBTransaction[" + id + "]: " + str);
        }
    }

    /**
     *
     */
    protected void setReleaseConnection(boolean rc) {
        releaseConnection = rc;
    }
}
