
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache;

import java.sql.ResultSet ;

import com.lutris.util.Config;

import org.enhydra.dods.CommonConstants;
import org.enhydra.dods.DODS;
import org.enhydra.dods.exceptions.AssertionDataObjectException;
import com.lutris.appserver.server.sql.StandardDatabaseManager;
import com.lutris.logging.Logger;
import com.lutris.dods.builder.generator.query.QueryBuilder;
import com.lutris.appserver.server.sql.standard.DatabaseConfiguration;
import com.lutris.appserver.server.sql.standard.DriverSpecificConstants;
import com.lutris.appserver.server.sql.standard.StandardLogicalDatabase;

/**
 * TableConfiguration contains parameters about table configuration. It reads 
 * these parameters from application's configuration file, sets them to new 
 * values or returns their current values.
 *
 * @author    Tanja Jovanovic
 * @author    Nenad Vico
 * @author    Zorica Suvajdzin
 * @version   2.0  15.06.2003.  
 */
 
public class TableConfiguration implements CacheConstants, DriverSpecificConstants {
    
    /**
     * If table is read-only, parameter readOnly is true, otherwise false.
     * Default value is false.
     */
    private boolean readOnly = ((StandardDatabaseManager)DODS.getDatabaseManager()).getAllReadOnly();


    /**
     * If table is caseSensitive, parameter caseSensitive is true, otherwise false.
     * Default value is false.
     */
    private boolean caseSensitive = CacheConstants.DEFAULT_CASE_SENSITIVE;


    /**
     * If table is read-only, parameter readOnly is true, otherwise false.
     * Default value is false.
     */
    private boolean originalAllReadOnly = ((StandardDatabaseManager)DODS.getDatabaseManager()).getAllReadOnly();    

    /**
     * If table supports lazy-loading, parameter lazyLoading is true, otherwise 
     * false.
     * Default value is false.
     */
    private boolean lazyLoading = CacheConstants.DEFAULT_LAZY_LOADING;
    
    /**
     * Maximal time in miliseconds for Event log.
     * If query is executed longer that this time, its select statement and 
     * execution time are written to Event log for future analyzes. 
     */
    private int maxExecuteTime = CacheConstants.DEFAULT_MAX_EXECUTE_TIME;
    
    private int defaultFetchSize = CacheConstants.DEFAULT_DEFAULT_FETCH_SIZE;            
    
    private int queryTimeout = CacheConstants.DEFAULT_QUERY_TIMEOUT;                

    private int fullCacheCountLimit = CacheConstants.DEFAULT_FULL_CACHE_COUNT_LIMIT;
    
    private String  initCachesResultSetType = null;
    
    private String  initCachesResultSetConcurrency = null;
    
    private Integer  queryTimeLimit = CommonConstants.DEFAULT_QUERY_TIME_LIMIT;

    /**
     * Optional select statement - used with mass Updates and/or deletes .
     * Default value is false.
     */
    private boolean selectOids = CacheConstants.DEFAULT_SELECT_OIDS;
    
    /**
     * Increment versions during mass Updates.
     * Default value is true.
     */
    private boolean incrementVersions = CacheConstants.DEFAULT_INCREMENT_VERSIONS;

    /**
     * Returns readOnly parameter.
     *
     * @return true if the table is read-only, otherwise false.
     */
    public boolean isReadOnly() {
        return readOnly;
    }
    
    /**
     * Sets readOnly parameter to value newReadOnly.
     *
     * @param newReadOnly New read-only value.
     * @exception AssertionDataObjectException If new value of read-only is 
     * false, but all application's tables are read-only.
     */
    protected void setReadOnly(boolean newReadOnly) throws AssertionDataObjectException {
        if (newReadOnly == false) {
            if (originalAllReadOnly) {
                throw new AssertionDataObjectException("Can't set read-only table to false: all tables are read-only");
            } else {
                Wrapper.getInstance().removeAllComplexQueries();
            }    
        }
        readOnly = newReadOnly;
    }
  
    /**
     * Return value of InitCachesResultSetType property
     * @return Value of InitCachesResultSetType property
     */ 
    public int getInitCachesResultSetType() {
        String  tmpUCrN = initCachesResultSetType;
        try {
            if (tmpUCrN==null)
                return ResultSet.TYPE_FORWARD_ONLY;
            if(tmpUCrN.equalsIgnoreCase("TYPE_SCROLL_SENSITIVE"))
                return ResultSet.TYPE_SCROLL_SENSITIVE;
            else if (tmpUCrN.equalsIgnoreCase("TYPE_SCROLL_INSENSITIVE"))
                return ResultSet.TYPE_SCROLL_INSENSITIVE;
            else if (tmpUCrN.equalsIgnoreCase("TYPE_FORWARD_ONLY"))
                return ResultSet.TYPE_FORWARD_ONLY;
            else {
                DODS.getLogChannel().write(Logger.DEBUG,"Invalid value for InitCachesResultSetType parameter :"+tmpUCrN);
                return ResultSet.TYPE_FORWARD_ONLY;
            }
        }catch(Exception  ex) {
            DODS.getLogChannel().write(Logger.DEBUG,"Use default value for InitCachesResultSetType parameter ");
            return ResultSet.TYPE_FORWARD_ONLY;
        }
    }

    /**
     * Return value of InitCachesResultSetConcurrency property
     * @return Value of InitCachesResultSetConcurrency property
     */
    public int getInitCachesResultSetConcurrency() {
        String  tmpUCrN = initCachesResultSetConcurrency;
        try{
            if (tmpUCrN==null)
                return ResultSet.CONCUR_READ_ONLY;
            if(tmpUCrN.equalsIgnoreCase("CONCUR_READ_ONLY"))
                return ResultSet.CONCUR_READ_ONLY;
            else if (tmpUCrN.equalsIgnoreCase("CONCUR_UPDATABLE"))
                return ResultSet.CONCUR_UPDATABLE;
            else {
                DODS.getLogChannel().write(Logger.DEBUG,"Invalid value for InitCachesResultSetConcurrency parameter :"+tmpUCrN);
                return ResultSet.CONCUR_READ_ONLY;
            }
        }catch(Exception  ex) {
            DODS.getLogChannel().write(Logger.DEBUG,"Use default value for InitCachesResultSetConcurrency parameter ");
            return ResultSet.CONCUR_READ_ONLY;
        }
    }
  
    /**
     * @param string
     */
    public void setInitCachesResultSetConcurrency(String  string) {
        initCachesResultSetConcurrency = string;
    }

    /**
     * @param string
     */
    public void setInitCachesResultSetType(String  string) {
        initCachesResultSetType = string;
    }

    /**
     * Returns lazyLoading parameter.
     *
     * @return true if the table supports lazy-loading, otherwise false.
     */
    public boolean isLazyLoading() {
        return lazyLoading;
    }
    
    /**
     * Sets readOnly parameter to value newlazyLoading.
     *
     * @param newlazyLoading New lazy-loading value.
     */
    protected void setLazyLoading(boolean newlazyLoading) {
        lazyLoading = newlazyLoading;
    }

    /**
     * Returns maxExecuteTime parameter.
     *
     * @return Maximal time in miliseconds for Event log.
     */
    public int getMaxExecuteTime() {
        return maxExecuteTime;
    }
    
    /**
     * Sets maxExecuteTime parameter to value newMaxExecuteTime.
     *
     * @param newMaxExecuteTime New maxExecuteTime value.
     */
    protected void setMaxExecuteTime(int newMaxExecuteTime) {
        maxExecuteTime = newMaxExecuteTime;
    }

    /**
     * Returns DefaultFetchSize parameter.
     *
     * @return DefaultFetchSize parameter value.
     */
    public int getDefaultFetchSize() {
        return defaultFetchSize;
    }
    
    /**
     * Sets DefaultFetchSize parameter to value newValue.
     *
     * @param newValue New DefaultFetchSize value.
     */
    private void setDefaultFetchSize(int newValue) {
        defaultFetchSize = newValue;
        QueryBuilder.setDefaultFetchSize(defaultFetchSize);
    }   
    
    /**
     * Returns QueryTimeout parameter.
     *
     * @return QueryTimeout parameter value.
     */
    public int getQueryTimeout() {
        return queryTimeout;
    }
    
    /**
     * Sets QueryTimeout parameter to value newValue.
     *
     * @param newValue New QueryTimeout value.
     */
    private void setQueryTimeout(int newValue) {
        queryTimeout = newValue;
        QueryBuilder.setDefaultQueryTimeout(queryTimeout);      
    }

    /**
     * Returns selectOids parameter.
     *  
     * @return selectOids parameter value.
     */
    public boolean getSelectOids() {
        return selectOids;
    }
    
    /**
     * Sets selectOids parameter.
     *
     * @param newValue New selectOids value.
     */
    private void setSelectOids(boolean newValue) {
        selectOids = newValue;
    }
    
    
    /**
     * Returns CaseSensitive parameter.
     * 
     * @return value of CaseSensitive parameter.
     */
    public boolean isCaseSensitive() {
        return caseSensitive;
    }

    /**
     * Set CaseSensitive parameter to newValue value.
     * 
     * @param newValue New value for CaseSensitive parameter
     */
    public void setCaseSensitive(boolean newValue) {
        caseSensitive = newValue;
    }


    /**
     * Returns IncrementVersions parameter.
     *  
     * @return IncrementVersions parameter value.
     */
    public boolean getIncrementVersions() {
        return incrementVersions;
    }
    
    /**
     * Sets IncrementVersions parameter.
     *
     * @param newValue New IncrementVersions value.
     */
    private void setIncrementVersions(boolean newValue) {
        incrementVersions = newValue;
    }

    public int getFullCacheCountLimit() {
        return fullCacheCountLimit;
    }
    public void setFullCacheCountLimit(int i) {
        fullCacheCountLimit = i;
    }
    
    
    /**
     * @return Returns the queryTimeLimit.
     */
    public Integer  getQueryTimeLimit() {
        return queryTimeLimit;
    }
    /**
     * @param queryTimeLimit The queryTimeLimit to set.
     */
    public void setQueryTimeLimit(Integer  queryTimeLimit) {
        this.queryTimeLimit = queryTimeLimit;
    }
    
    
    /**
     * Reads table configuration parameters.
     *
     * @param tableConfig Configuration file from which the parameters are read.
     */
    public void readTableConfiguration(Config tableConfig, String  dbName) {
        
        DatabaseConfiguration defaultDatabaseConfig = null;
        
        try {
            defaultDatabaseConfig = ((StandardLogicalDatabase) (DODS.getDatabaseManager().findLogicalDatabase(dbName))).getDatabaseConfiguration();
            
        } catch (Exception  ex) {}
        if (defaultDatabaseConfig != null) {

            try {
                readOnly = defaultDatabaseConfig.isAllReadOnly();
                originalAllReadOnly = readOnly;
            } catch (Exception  e) {}
            try {
                lazyLoading = defaultDatabaseConfig.isLazyLoading();
            } catch (Exception  e) {}
            try {
                caseSensitive = defaultDatabaseConfig.getCaseSensitive();
            } catch (Exception  e) {}
            try {
                maxExecuteTime = defaultDatabaseConfig.getMaxExecuteTime();
            } catch (Exception  e) {}
            try {
                defaultFetchSize = defaultDatabaseConfig.getDefaultFetchSize();
            } catch (Exception  e) {}
            try {
                queryTimeout = defaultDatabaseConfig.getQueryTimeout();
            } catch (Exception  e) {}
            try {
                selectOids = defaultDatabaseConfig.getSelectOids();
            } catch (Exception  e) {}
            try {
                incrementVersions = defaultDatabaseConfig.getIncrementVersions();
            } catch (Exception  e) {}
            try {
                fullCacheCountLimit = defaultDatabaseConfig.getFullCacheCountLimit();
            } catch (Exception  e) {}
            try {
                initCachesResultSetType = defaultDatabaseConfig.getInitCachesResultSetType();
            } catch (Exception  e) {}
            try {
                initCachesResultSetConcurrency = defaultDatabaseConfig.getInitCachesResultSetConcurrency();
            } catch (Exception  e) {}
            try {
                queryTimeLimit = defaultDatabaseConfig.getQueryTimeLimit();
            } catch (Exception  e) {}
            
            
        }
        if (tableConfig != null) {
            try {
                if (!readOnly) {
                    readOnly = tableConfig.getBoolean(CacheConstants.PARAMNAME_READ_ONLY);
                } else {
                    if (!tableConfig.getBoolean(CacheConstants.PARAMNAME_READ_ONLY)) {
                        DODS.getLogChannel().write(Logger.WARNING,
                                "Single table has readOnly=false property in config file, but AllReadOnly=true ");
                    }
                }    
            } catch (Exception  e) {}
            try {
                lazyLoading = tableConfig.getBoolean(CacheConstants.PARAMNAME_LAZY_LOADING);
            } catch (Exception  e) {}
            try {
                maxExecuteTime = tableConfig.getInt(CacheConstants.PARAMNAME_MAX_EXECUTE_TIME);
            } catch (Exception  e) {}
            try {
                defaultFetchSize = tableConfig.getInt(CacheConstants.PARAMNAME_DEFAULT_FETCH_SIZE);
            } catch (Exception  e) {}
            try {
                queryTimeout = tableConfig.getInt(CacheConstants.PARAMNAME_QUERY_TIMEOUT);
            } catch (Exception  e) {}
            try {
                selectOids = tableConfig.getBoolean(CacheConstants.PARAMNAME_SELECT_OIDS);
            } catch (Exception  e) {}
            try {
                incrementVersions = tableConfig.getBoolean(CacheConstants.PARAMNAME_INCREMENT_VERSIONS);
            } catch (Exception  e) {}
            try {
                caseSensitive = tableConfig.getBoolean(CacheConstants.PARAMNAME_CASE_SENSITIVE);
            } catch (Exception  e) {}
            try {
                fullCacheCountLimit = tableConfig.getInt(CacheConstants.FULL_CACHE_COUNT_LIMIT);
            } catch (Exception  e) {}
            try {
                initCachesResultSetType = tableConfig.getString(CommonConstants.INIT_CACHES_RESULT_SET_TYPE);
            } catch (Exception  e) {}
            try {
                initCachesResultSetConcurrency = tableConfig.getString(CommonConstants.INIT_CACHES_RESULT_SET_CONCURRENCY);
            } catch (Exception  e) {}
            try {
                queryTimeLimit = new Integer (tableConfig.getInt(CommonConstants.QUERY_TIME_LIMIT));
            } catch (Exception  e) {}
        }
        setQueryTimeout(queryTimeout);
        setDefaultFetchSize(defaultFetchSize);
    }
    
}
