
 

/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: StandardDatabaseManager.java,v 1.9 2005/05/26 08:08:10 predrag Exp $
 */
package com.lutris.appserver.server.sql;

import java.io.FileInputStream ;
import java.io.InputStream ;
import java.sql.SQLException ;
import java.lang.reflect.Method ;
import java.net.URL ;
import java.util.Date ;
import java.util.Enumeration ;
import java.util.Hashtable ;
import org.enhydra.dods.Common;
import org.enhydra.dods.CommonConstants;
import org.enhydra.dods.DODS;
import org.enhydra.dods.cache.CacheConstants;
import org.enhydra.dods.cache.Wrapper;
import com.lutris.appserver.server.sql.standard.StandardLogicalDatabase;
import com.lutris.appserver.server.sql.standard.DriverSpecificConstants;
import com.lutris.classloader.MultiClassLoader;
import com.lutris.util.Config;
import com.lutris.util.ConfigException;
import com.lutris.util.ConfigFile;
import org.enhydra.util.ConfigFileInterface;
import com.lutris.util.KeywordValueException;
import com.lutris.dods.builder.generator.query.QueryBuilder;
import com.lutris.logging.Logger;

import java.io.File ;

import javax.xml.parsers.DocumentBuilder ;
import javax.xml.parsers.DocumentBuilderFactory ;

import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.NodeList ;


/**
 * The standard database manager implementation. A database manager
 * manages logical databases. It provides a single object from which
 * database connections, object ids (OIDs), transactions and queries
 * can be obtained. The configuration file specifies what logical
 * database to create.
 * <P>
 * The configuration data is specified as follows:
 * <UL>
 * <LI> <B><CODE>DatabaseManager.Databases</CODE></B> -
 *  A list of logical SQL database names.
 * <LI> <B><CODE>DatabaseManager.DefaultDatabase</CODE></B> -
 *  The default logical database used by this application. Optional,
 *  if not specified, then the first entry in
 *  "DatabaseManager.Databases" is used.
 * <LI> <B><CODE>DatabaseManager.Debug</CODE></B> -
 *  Specify true to enable Query and Transaction logging, false to
 *  disable it. Optional, false if not specified.
 * </UL>
 * For each logical database, there is a set of entry names in the form
 * <B><CODE>DatabaseManager.DB.<I>dbname</I></CODE></B>.
 * Where <CODE><I>dbname</I></CODE> is one of the listed logical databases.
 * <P>
 * <B><CODE>DatabaseManager.DB.<I>dbname</I>.ClassType</CODE></B> -
 * This is an optional field which specifies the class of the logical
 * database implementation or a symbolic name if one on the standard types
 * are selected. This is recommended because although JDBC abstracts the
 * data access, the functionality of each database is slightly different
 * and this parameter allows for optimised usage. The following are
 * standard types:
 * <UL>
 * <LI> <I>Oracle</I> - For optimized Oracle 7/8 usage.
 * <LI> <I>Informix</I> - For optimized Informix usage.
 * <LI> <I>Sybase</I> - For optimized Sybase usage.
 * <LI> <I>Msql</I> - For optimized Microsoft MSQL usage.
 * <LI> <I>Standard</I> - For all other JDBC databases. <B>(Default)</B>
 * <LI> <I>class</I> - For vendor supplied database classes.
 * </UL>
 * <P>
 * Note that since a single SQL user is used to access the database by
 * the entire application, connections maybe kept open, thus saving this
 * overhead on each request.  Connections are opened as desired until
 * the maximum configured is reached.
 * <P>
 * If a thread needs to process a transaction, it requests a connection.
 * If none are available, then a new connection is created up to the
 * configured maximum.  A thread is queued waiting for a connection to be
 * returned if a new connection can't be created.
 * <P>
 * Note also that multiple logical names may map to the same actual
 * database, making it easy to migrate databases if necessary and to
 * provide access through multiple users.
 *
 * @version $Revision: 1.9 $
 * @since   LBS1.8
 * @author  Paul Morgan
 * @author  Kyle Clark
 */
public class StandardDatabaseManager implements DatabaseManager, CacheConstants, DriverSpecificConstants {

    /**
     * Table of named logical databases.
     */
    private Hashtable  logicalDatabases = new Hashtable ();

    /**
     * Default logical database.
     */
    private LogicalDatabase defaultLogicalDatabase = null;

    /**
     * Controls debugging for Transactions and Queries.
     */
    protected boolean debug = false;
    protected String  defaultDB = null;
    protected Config config = null;
    
    protected String  confDir = null;
    
    private String  appName = CacheConstants.DEFAULT_APP_NAME;    
    
    private DatabaseManagerConfiguration dbmConf = new DatabaseManagerConfiguration();
    
    private Config mainConfig = null;
    
    

    
    /**
     * Creates a new <code>DatabaseManager</code> object and configures
     * the logical databases defined in the config file.
     *
     * @param config
     *  The configuration data for logical databases.
     * @exception ConfigException
     *  If there is an error in the configuration file.
     * @exception DatabaseManagerException
     *  If a logical database name is specified twice in the configuration file.
     * @exception SQLException
     *  If a SQL error occurs.
     */
    public StandardDatabaseManager(Config config)
        throws ConfigException, DatabaseManagerException, SQLException  {
            
        this.config = config;
        String [] databases = config.getStrings("Databases");

        if (databases == null) {
            return;
        }
            
        try {
            confDir = config.getString("ConfigurationDir", null);
        } catch (KeywordValueException except) {
            confDir = null;
        }
        try {
            appName = config.getString(PARAMNAME_APP_NAME, DEFAULT_APP_NAME);            
        } catch (KeywordValueException except) {
            appName = DEFAULT_APP_NAME;
        }
        Common.setConfigDir(confDir);
 
        try {
            Config userConfig = (Config) config.getSection("DB.User");
            if (userConfig != null) {
                String  value;
                if ((value = userConfig.getString("userWildcard")) != null) {
                    dbmConf.setUserConfigWildcard(value);
                }
                if ((value = userConfig.getString("userSingleWildcard")) != null) {
                    dbmConf.setUserConfigSingleWildcard(value);
                }
                if ((value = userConfig.getString("userSingleWildcardEscape"))
                        != null) {
                    dbmConf.setUserConfigSingleWildcardEscape(value);
                }
                if ((value = userConfig.getString("userWildcardEscape")) != null) {
                    dbmConf.setUserConfigWildcardEscape(value);
                }
            }
        } catch (Exception  except) {}

 
        Config defaultsConfig;

        try {
            defaultsConfig = (Config) config.getSection("defaults");
        } catch (KeywordValueException except) {
            throw new ConfigException("No DatabaseManager.defaults defined in config file.");
        }
        
        if (defaultsConfig != null) {
            dbmConf.setAllReadOnly(defaultsConfig.getBoolean(PARAMNAME_ALL_READ_ONLY,
                    CacheConstants.DEFAULT_All_READONLY));
            dbmConf.setLazyLoading(defaultsConfig.getBoolean(PARAMNAME_LAZY_LOADING,
                    false));
            dbmConf.setCaseSensitive(defaultsConfig.getBoolean(PARAMNAME_CASE_SENSITIVE,
                    DEFAULT_CASE_SENSITIVE));                    
            dbmConf.setMaxExecuteTime(defaultsConfig.getInt(PARAMNAME_MAX_EXECUTE_TIME,
                    0));
            dbmConf.setTransactionCheck(defaultsConfig.getBoolean(PARAMNAME_TRANSACTION_CHECK,
                    false));
            dbmConf.setDeleteCheckVersion(defaultsConfig.getBoolean(PARAMNAME_DELETE_CHECK_VERSION,
                    false));
            dbmConf.setAutoSave(defaultsConfig.getBoolean(PARAMNAME_AUTO_SAVE,
                    false));
            dbmConf.setAutoSaveCreateVirgin(defaultsConfig.getBoolean(PARAMNAME_AUTO_SAVE_CREATE_VIRGIN,
                    false));
            dbmConf.setAutoWrite(defaultsConfig.getBoolean(PARAMNAME_AUTO_WRITE,
                    false));
            dbmConf.setTransactionCaches(defaultsConfig.getBoolean(PARAMNAME_TRANSACTION_CACHES,
                    false));
            dbmConf.setDeadlockWaitTime(defaultsConfig.getInt(PARAMNAME_DEADLOCK_READ_TIME,
                    0));
            dbmConf.setDeadlockRetryCount(defaultsConfig.getInt(PARAMNAME_DEADLOCK_RETRY_NUMBER,
                    0));
            QueryBuilder.setDefaultQueryTimeout(config.getInt(PARAMNAME_QUERY_TIMEOUT,
                    0));
            dbmConf.setQueryTimeout(defaultsConfig.getInt(PARAMNAME_QUERY_TIMEOUT,
                    0));
            dbmConf.setSelectOids(defaultsConfig.getBoolean(PARAMNAME_SELECT_OIDS,
                    false));
            dbmConf.setIncrementVersions(defaultsConfig.getBoolean(PARAMNAME_INCREMENT_VERSIONS,
                    true));
            QueryBuilder.setDefaultFetchSize(config.getInt(PARAMNAME_DEFAULT_FETCH_SIZE,
                    -1));
            dbmConf.setDefaultFetchSize(defaultsConfig.getInt(PARAMNAME_DEFAULT_FETCH_SIZE,
                    -1));
            dbmConf.setDBTransactionFactoryName(defaultsConfig.getString(CommonConstants.TRANSACTION_FACTORY,
                    null));     
            dbmConf.setConnectionAllocatorName(defaultsConfig.getString(CommonConstants.CONNECTION_ALLOCATOR,
                    null)); 
            dbmConf.setDBConnectionFactoryName(defaultsConfig.getString(CommonConstants.CONNECTION_FACTORY,
                    null));    
            dbmConf.setQueryCacheImplClass(defaultsConfig.getString(CommonConstants.QUERY_CACAHE_IMPL_CLASS,
                    null));
            dbmConf.setFullCacheCountLimit(defaultsConfig.getInt(CacheConstants.FULL_CACHE_COUNT_LIMIT,
                    CacheConstants.DEFAULT_FULL_CACHE_COUNT_LIMIT));   
            dbmConf.setCaseInsensitiveDatabase(defaultsConfig.getBoolean(CommonConstants.CASE_INSENSITIVE_DATABASE,
                    CommonConstants.DEFAULT_CASE_INSENSITIVE_DATABASE));
            dbmConf.setXaDefaultTimeout(defaultsConfig.getInt(CommonConstants.XA_DEFAULT_TIMEOUT
                    ,CommonConstants.DEFAULT_XA_DEFAULT_TIMEOUT));
            dbmConf.setXaTransactonManagerLookupName(defaultsConfig.getString(CommonConstants.XA_TM_LOOKUP_NAME
                    ,CommonConstants.DEFAULT_XA_TM_LOOKUP_NAME));
            dbmConf.setXaUsageCase(defaultsConfig.getInt(CommonConstants.XA_USAGE_CASE
                    ,CommonConstants.DEFAULT_XA_USAGE_CASE));
            dbmConf.setXaWrappedTransImplFactory(defaultsConfig.getString(CommonConstants.XA_WRAPPED_TRANS_IMPL_FACTORY
                    ,CommonConstants.DEFAULT_XA_WRAPPED_TRANS_IMPL_FACTORY));
            dbmConf.setXaUserTransactonLookupName(defaultsConfig.getString(CommonConstants.XA_USER_TRANSACTION_LOOKUP_NAME
                    ,CommonConstants.DEFAULT_XA_USER_TRANSACTION_LOOKUP_NAME)); 
            dbmConf.setXaJtaSupport(defaultsConfig.getString(CommonConstants.XA_JTA_SUPPORT
                    ,CommonConstants.DEFAULT_XA_JTA_SUPPORT));
            try {
                dbmConf.setUseCursorName(defaultsConfig.getBoolean(PARAMNAME_USE_CURSOR_NAME));                    
            } catch (ConfigException e) {
                dbmConf.setUseCursorName(null);
            }
            dbmConf.setInitCachesResultSetType(defaultsConfig.getString(CommonConstants.INIT_CACHES_RESULT_SET_TYPE,
                    null));
            dbmConf.setInitCachesResultSetConcurrency(defaultsConfig.getString(CommonConstants.INIT_CACHES_RESULT_SET_CONCURRENCY,
                    null));
            dbmConf.setSqlBatch(defaultsConfig.getBoolean(CommonConstants.SQL_BATCH,
                    CommonConstants.DEFAULT_SQL_BATCH));
            try {
                dbmConf.setQueryTimeLimit(new Integer (defaultsConfig.getInt(CommonConstants.QUERY_TIME_LIMIT)));                    
            } catch (ConfigException e) {
                dbmConf.setQueryTimeLimit(null);
            }
        }
        try {
            defaultsConfig = (Config) config.getSection("defaults.cache");
        } catch (KeywordValueException except) {
            throw new ConfigException("No DatabaseManager.defaults defined in config file.");
        }   
        
        if (defaultsConfig != null) {
            dbmConf.setReserveFactor(defaultsConfig.getDouble(PARAMNAME_RESERVE_FACTOR,
                    CacheConstants.DEFAULT_RESERVE_FACTOR));
            dbmConf.setCachePercentage(defaultsConfig.getDouble(PARAMNAME_CACHE_PERCENTAGE,
                    CacheConstants.DEFAULT_CACHE_PERCENTAGE));
            dbmConf.setMaxCacheSize(defaultsConfig.getInt(PARAMNAME_MAX_CACHE_SIZE,
                    CacheConstants.DEFAULT_MAX_CACHE_SIZE));
            dbmConf.setMaxSimpleCacheSize(defaultsConfig.getInt(PARAMNAME_MAX_SIMPLE_CACHE_SIZE,
                    CacheConstants.DEFAULT_MAX_SIMPLE_QUERY_CACHE_SIZE));
            dbmConf.setMaxComplexCacheSize(defaultsConfig.getInt(PARAMNAME_MAX_COMPLEX_CACHE_SIZE,
                    CacheConstants.DEFAULT_MAX_COMPLEX_QUERY_CACHE_SIZE));
            dbmConf.setMaxMultiJoinCacheSize(defaultsConfig.getInt(PARAMNAME_MAX_MULTI_JOIN_CACHE_SIZE,
                    CacheConstants.DEFAULT_MAX_MULTI_JOIN_QUERY_CACHE_SIZE));
            dbmConf.setInitAllCaches(defaultsConfig.getBoolean(PARAMNAME_INITIAL_ALL_CACHES,
                    CacheConstants.DEFAULT_INITIAL_ALL_CACHES));
            dbmConf.setInitialCacheFetchSize(defaultsConfig.getInt(CacheConstants.PARAMNAME_INITIAL_CACHE_FETCH_SIZE,
                    CacheConstants.DEFAULT_INITIAL_CACHE_FETCH_SIZE));
            dbmConf.setInitialDSCacheSize(defaultsConfig.getInt(CacheConstants.PARAMNAME_INITIAL_DS_CACHE_SIZE,
                    CacheConstants.DEFAULT_INITIAL_DS_CACHE_SIZE));
            dbmConf.setDodsCacheFactory(defaultsConfig.getString(CacheConstants.PARAMNAME_DODS_CACHE_FACTORY
                            ,CacheConstants.DEFAULT_DODS_CACHE_FACTORY));
        }
        
        isDODSConfigured = null != System.getProperty("DODS_HOME", null)
                || null != config.getString("ConfigurationDir", null);
//        mustConfigureDODS = config.getBoolean("DoFullInit", false);
          mustConfigureDODS = false;

        /**
         * Configure each logical database.
         */
        Config dbConfig;

        for (int idx = 0; idx < databases.length; idx++) {
            String  dbName = databases[idx];

            try {
                dbConfig = (Config) config.getSection("DB." + dbName);
            } catch (KeywordValueException except) {
                throw new ConfigException("No DatabaseManager.DB." + dbName
                        + " defined in config file.");
            }
            
            if (logicalDatabases.get(dbName) != null) {
                throw new DatabaseManagerException("duplicate logical database name: \""
                        + dbName + "\"");
            }
            
            LogicalDatabase logicalDatabase = loadLogicalDatabase(dbName,dbConfig);

            logicalDatabases.put(dbName, logicalDatabase);
        }

        /**
         * Set default database if supplied.
         */
        defaultDB = config.getString("DefaultDatabase", databases[0]);
        setDefaultDatabase(defaultDB);

        /**
         * Set debug logging of Queries and Transactions.
         */
        boolean debugLogging = config.getBoolean("Debug", false);

        setDebugLogging(debugLogging);
        Wrapper.getInstance().setTimeout(config.getLong("MainCacheLockTimeout",
                100));

        /**
         * Set the oid and version column names in CoreDO.
         * Note that these could be configured for each logical
         * database but this would require some complex extensions
         * to DBQuery and DBTransaction in order to ensure that
         * the correct values were always being used without
         * creating race conditions.
         * This means that the limitation is currently that all
         * databases accessed by your application must use the
         * same column names.
         */
        String  oidColumnName = config.getString("ObjectIdColumnName", null);

        if (oidColumnName != null) {
            CoreDO.set_OIdColumnName(oidColumnName);
        }
        String  versionColumnName = config.getString("VersionColumnName", null);

        if (versionColumnName != null) {
            CoreDO.set_versionColumnName(versionColumnName);
        }
     
        if(mainConfig == null) {
         ConfigFileInterface mainConfigFile = null;
            if(config!=null)
               mainConfigFile = config.getConfigFile();
            if(mainConfigFile!=null)
               mainConfig = mainConfigFile.getConfig();
        } 
    }




    public void initChaches(ClassLoader  clsLoader) {
            try{
                initAllDodsCaches(this.config, clsLoader);
        }catch(Exception  e){
                DODS.getLogChannel().write(Logger.DEBUG,"Unable to load caches at startup");
            }
    }



    
    /**
     * Creates new <code>StandardDatabaseManager</code> instance.
     *
     * @param confURL Additional path to folder or *.jar file with configuration file. If null use DODS classpath
     *
     * @param confFile Name of conf file relativ to *.jar file or to specifide folder (from confURL/DODS claspath).
     * 
     * @return new <code>StandardDatabaseManager</code> instance.
     *    
     */
    public static DatabaseManager newInstance(URL  confURL, String  confFile) 
        throws ConfigException, DatabaseManagerException, SQLException  {
        try { 
            InputStream  configIS = Common.getConfFileFromURL(confURL, confFile);                                               
            ConfigFile configFile = new ConfigFile(configIS);                 
            Config config = configFile.getConfig();
            configIS.close();                                                 
            Config dbConfig = new Config(config.getSection("DatabaseManager"),configFile);    
            return new StandardDatabaseManager(dbConfig); 
        } catch (Exception  e) {                                                
            throw new ConfigException(e);  
        }                                                                      
    }   
        

    
    /**
     * Creates new <code>StandardDatabaseManager</code> instance.
     *
     * @param fileName full path name of the application configuration file.
     *
     * @return new <code>StandardDatabaseManager</code> instance.
     *    
     */
    public static DatabaseManager newInstance(String  fileName) 
        throws ConfigException, DatabaseManagerException, SQLException  {
        try {       
            Config config  = null;
            ConfigFile configFile = null;       
            if(fileName!=null) {
                try {
                    File inputFile = new File(fileName); 
                    if(inputFile!=null && inputFile.isFile()) {
                        configFile = new ConfigFile(inputFile); 
                        config = configFile.getConfig();  
                    }
                } catch (Exception  e) {
                    System.out.println("Error during configuration file reading: '"+fileName+"'");
                }
            }          
            if (config==null){
                InputStream  configIS=Common.getConfFileFromURL(null,null);
                configFile = new ConfigFile(configIS);
                config = configFile.getConfig();
                configIS.close();         
            }   
            if (config==null){
                throw new Exception ("Can't create configuration object from file:'"+fileName+"'");  
            }
            Config dbConfig = new Config(config.getSection("DatabaseManager"),configFile); 
            return new StandardDatabaseManager(dbConfig); 
        } catch (Exception  e) {                                               
            throw new ConfigException(e);  
        }                                                                     
    }   
        
    /**
     * Return main configuration class 
     *  
     * @return
     *   main configuration class.
     */ 
    public Config getConfig() {
        return config;
    }

    /**
     * Return main application configuration class
     *  
     * @return
     *   main application configuration class.
     */ 

    public Config getParentConfig() {
     
            return mainConfig;    
    }



    boolean isDODSConfigured = false;
    boolean mustConfigureDODS = false;

    /**
     * Actually load the specified logical database. This method provides
     * an easy way to override the default behavour.
     *
     * @return
     *   The logical database.
     * @exception DatabaseManagerException
     *   if an error occurs creating the logical database.
     */
    public LogicalDatabase loadLogicalDatabase(String  dbName, Config dbConfig)
        throws DatabaseManagerException {
        LogicalDatabase lDB = null;

        try {
            if (mustConfigureDODS && !isDODSConfigured) {
                throw new DatabaseManagerException("DODS is not configured properly.");
            }
            if (isDODSConfigured) {
                String  dbClassName = dbConfig.getString("ClassType", "Standard");
                String  classPath = Common.getDodsConfProperty("ClassPath",
                        dbClassName);
                String  className = Common.getDodsConfProperty("ClassName",
                        dbClassName);

                if (classPath != null && className != null) {
                    MultiClassLoader loader = new MultiClassLoader(null);

                    loader.setClassPath(classPath);
                    Class  dbClass = loader.loadClass(className);

                    lDB = (LogicalDatabase) dbClass.newInstance();
                    lDB.init(dbName, dbConfig);
                }
            }
            if (lDB == null) {
                lDB = new StandardLogicalDatabase(dbName, dbConfig, dbmConf);
            }
        } catch (Exception  except) {
            throw new DatabaseManagerException("Could not create logical database "
                    + dbName,
                    except);
        }
        return lDB;
    }

    /**
     * Return default database name (given in config file)
     * @return
     *   default database name
     */
    public String  getDefaultDB() {
        return defaultDB;
    }

    /**
     * Allocate a connection to a thread. The connection should be returned
     * to the allocator by calling its
     * <a HREF=com.lutris.appserver.server.sql.DBConnection#release>
     * release()</a> function.  A thread will wait if
     * no connections are available.
     * Interupted exceptions are converted to
     * errors.
     * N.B. Can't be synchronized, as connection allocator
     * <CODE>allocate</CODE> may wait.
     *
     * @param dbName
     *   Logical name of the database to allocate a connection to.
     * @return
     *   The allocated connection object.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     * @exception SQLException
     *   If a SQL error occures.
     */
    public DBConnection allocateConnection(String  dbName)
        throws DatabaseManagerException, SQLException  {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        return logicalDatabase.allocateConnection();
    }

    /**
     * Allocate a connection to a thread. The connection should be returned
     * to the allocator by calling its
     * <a HREF=com.lutris.appserver.server.sql.DBConnection#release>
     * release()</a> function.  A thread will wait if
     * no connections are available.
     * Interupted exceptions are converted to
     * errors.  The connection is allocated from the default logical
     * database.
     *
     * @return
     *   The allocated connection object.
     * @exception DatabaseManagerException
     *   If no default logical database has been set.
     * @exception SQLException
     *   If a SQL error occurs.
     * @see
     *   #setDefaultDatabase
     */
    public DBConnection allocateConnection()
        throws DatabaseManagerException, SQLException  {
        if (defaultLogicalDatabase == null) {
            throw new DatabaseManagerException("Default logical database "
                    + "has not been specified.");
        }
        return defaultLogicalDatabase.allocateConnection();
    }

    /**
     * Allocate an object id from the specified logical database.
     *
     * @param dbName
     *   Logical name of the database from which to obtain an object id.
     * @return The allocated unique OID
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     * @exception ObjectIdException
     *   If a problem (e.g. SQL error) occured in obtaining the OID.
     */
    public ObjectId allocateObjectId(String  dbName)
        throws DatabaseManagerException, ObjectIdException {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        return logicalDatabase.allocateObjectId();
    }

    /**
     * Allocate an object id from the specified logical database.
     *
     * @return The allocated connection object.
     * @exception DatabaseManagerException
     *   If a nonexistent default logical database has been set.
     * @exception ObjectIdException
     *   If a problem (e.g. SQL error) occured in obtaining the OID.
     * @see
     *   #setDefaultDatabase
     */
    public ObjectId allocateObjectId()
        throws DatabaseManagerException, ObjectIdException {
        if (defaultLogicalDatabase == null) {
            throw new DatabaseManagerException("Default logical database "
                    + "has not been specified.");
        }
        return defaultLogicalDatabase.allocateObjectId();
    }

    /**
     * Check does oid belong to Object id's range [minOId, currentOId]
     *
     * @param dbName
     *   Logical name of the database from which to check an object id.
     * @param oid
     *   oid which will be checked.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     * @exception ObjectIdException
     *   If a oid does't belong to range.
     */
    public void checkOId(String  dbName, ObjectId oid)
        throws DatabaseManagerException, ObjectIdException {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        logicalDatabase.checkOId(oid);
    }

    /**
     * Check does oid belong to Object id's range [minOId, currentOId] for default database
     *
     * @param oid
     *   oid which will be checked.
     * @exception DatabaseManagerException
     *   If a nonexistent default logical database has been set.
     * @exception ObjectIdException
     *   If a oid does't belong to range.
     */
    public void checkOId(ObjectId oid)
        throws DatabaseManagerException, ObjectIdException {
        if (defaultLogicalDatabase == null) {
            throw new DatabaseManagerException("Default logical database "
                    + "has not been specified.");
        }
        defaultLogicalDatabase.checkOId(oid);
    }

    /**
     * Create a transaction object for the specified logical database.
     *
     * @param dbName
     *   Logical name of the database from which to obtain a transaction.
     * @return The transaction
     * @exception DatabaseManagerException
     *   If a nonexistent or invalid logical database name is supplied.
     * @exception SQLException
     *   If a problem occured creating the transaction.
     */
    public DBTransaction createTransaction(String  dbName)
        throws DatabaseManagerException, SQLException  {
        DBTransaction ret;

        if (dbName == null) { 
            return createTransaction();
        }
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        ret = logicalDatabase.createTransaction();
        ret.setDatabaseName(dbName);
        return ret;
    }

    /**
     * Create a transaction object for the default logical database.
     *
     * @return The transaction
     * @exception DatabaseManagerException
     *   If a nonexistent default logical database has been set.
     * @exception SQLException
     *   If a problem occured creating the transaction.
     * @see
     *   #setDefaultDatabase
     */
    public DBTransaction createTransaction()
        throws DatabaseManagerException, SQLException  {
        DBTransaction ret;

        if (defaultLogicalDatabase == null) {
            throw new DatabaseManagerException("Default logical database "
                    + "has not been specified.");
        }
        ret = defaultLogicalDatabase.createTransaction();
        ret.setDatabaseName(defaultDB);
        return ret;
    }

    /**
     * Create a query object for the specified logical database.
     *
     * @param dbName
     *   Logical name of the database from which to obtain a query.
     * @return The query
     * @exception DatabaseManagerException
     *   If a nonexistent or invalid logical database name is supplied.
     * @exception SQLException
     *   If a problem occured creating the query.
     */
    public DBQuery createQuery(String  dbName)
        throws DatabaseManagerException, SQLException  {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        return logicalDatabase.createQuery();
    }

    /**
     * Create a query object for the default logical database.
     *
     * @return The query
     * @exception DatabaseManagerException
     *   If a nonexistent default logical database has been set.
     * @exception SQLException
     *   If a problem occured creating the query.
     * @see
     *   #setDefaultDatabase
     */
    public DBQuery createQuery()
        throws DatabaseManagerException, SQLException  {
        if (defaultLogicalDatabase == null) {
            throw new DatabaseManagerException("Default logical database "
                    + "has not been specified.");
        }
        return defaultLogicalDatabase.createQuery();
    }

    /**
     * Return a loical database type for the specified logical database.
     *
     * @param dbName
     *   Logical name of the database from which to obtain a query.
     * @return loical database type
     * @exception DatabaseManagerException
     *   If a nonexistent or invalid logical database name is supplied.
     * @exception SQLException
     *   If a problem occured creating the query.
     */
    public String  logicalDatabaseType(String  dbName)
        throws DatabaseManagerException, SQLException  {
        LogicalDatabase logicalDatabase = null;

        if (dbName != null) {
            logicalDatabase = findLogicalDatabase(dbName);
            return logicalDatabase.getType();
        } else {
            return logicalDatabaseType();
        }
    }

    /**
     * Return a loical database type for the default logical database.
     *
     * @return loical database type
     * @exception DatabaseManagerException
     *   If a nonexistent default logical database has been set.
     * @exception SQLException
     *   If a problem occured creating the query.
     * @see
     *   #setDefaultDatabase
     */
    public String  logicalDatabaseType()
        throws DatabaseManagerException, SQLException  {
        if (defaultLogicalDatabase == null) {
            throw new DatabaseManagerException("Default logical database "
                    + "has not been specified.");
        }
        return defaultLogicalDatabase.getType();
    }
 
    /**
     * Find the named logical database in hash table.
     *
     * @param dbName Logical name of the database to locate.
     * @exception DatabaseManagerException If a nonexistant logical database
     *  name is supplied.
     */
    public LogicalDatabase findLogicalDatabase(String  dbName)
        throws DatabaseManagerException {
        LogicalDatabase logicalDatabase = (LogicalDatabase) logicalDatabases.get(dbName);

        if (logicalDatabase == null) {
            throw new DatabaseManagerException
                    ("unknown logical database name: \"" + dbName + "\"");
        }
        return logicalDatabase;
    }

    /**
     * Set the default logical database. This should only be called
     * after the logical database
     * (<A HREF=com.lutris.appserver.server.sql.LogicalDatabase></A>
     * has been established.
     *
     * @param dbName The default logical database.
     * @exception DatabaseManagerException
     *   if the logical database name is invalid or not found.
     */
    public void setDefaultDatabase(String  dbName)
        throws DatabaseManagerException {
        defaultLogicalDatabase = findLogicalDatabase(dbName);
    }

    /**
     * Shutdown the database manager. All logical databases will be
     * shutdown and all connections closed.
     */
    public void shutdown() {
        for (Enumeration  keys = logicalDatabases.keys();
                keys.hasMoreElements();) {
            LogicalDatabase logicalDatabase = (LogicalDatabase)
                    logicalDatabases.get((String ) keys.nextElement());

            logicalDatabase.shutdown();
        }
    }

    // ====================================================================
 // The following are primarily for management purposes...
 // ====================================================================
 /**
     * Returns the list of managed logical databases.
     *
     * @return  List of logical database names.
     */
    public String [] getLogicalDatabaseNames() {
        String [] names = new String [logicalDatabases.size()];
        int idx = 0;
        Enumeration  keys = logicalDatabases.keys();

        while (keys.hasMoreElements()) {
            names[idx++] = (String ) keys.nextElement();
        }
        return names;
    }

    /**
     * Returns a description of the logical database type.
     *
     * @param dbName
     *   The logical database name.
     * @return
     *   A text description of the logical database type.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     */
    public String  getType(String  dbName)
        throws DatabaseManagerException {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        return logicalDatabase.getType();
    }

    /**
     * Gets the number of requests made to the database since startup time.
     *
     * @param dbName The name of the logical database.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     * @return  The number of database requests since the server started.
     */
    public long getRequestCount(String  dbName)
        throws DatabaseManagerException {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        return logicalDatabase.getRequestCount();
    }

    /**
     * Gets the number of currently active connections.
     *
     * @param   dbName The name of the logical database.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     * @return  The number of currently active connections.
     */
    public int getActiveConnectionCount(String  dbName)
        throws DatabaseManagerException {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        return logicalDatabase.getActiveConnectionCount();
    }

    /**
     * Gets the maximum number of concurent connections that existed
     * at any time since this object was created, or
     * <CODE>resetMaxConnectionCount()</CODE> was called.
     * This is a historical highwater mark.
     * If you do not implement this feature, return -1.
     *
     * @param   dbName The name of the logical database.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     * @return The highwater mark for number of connections, or -1.
     */
    public int getMaxConnectionCount(String  dbName)
        throws DatabaseManagerException {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        return logicalDatabase.getMaxConnectionCount();
    }

    /**
     * Gets the time when the maximum refered to by
     * <CODE>maxConnectionCount()</CODE> occured.
     *
     * @param   dbName The name of the logical database.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     * @return The Date of when the maximum number of connections occured.
     */
    public Date  getMaxConnectionCountDate(String  dbName)
        throws DatabaseManagerException {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        return logicalDatabase.getMaxConnectionCountDate();
    }

    /**
     * Reset the maximum connection count. See
     * <CODE>maxConnectionCount()</CODE>. The highwater mark should be
     * reset to the current number of connections.
     *
     * @param   dbName The name of the logical database.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     */
    public void resetMaxConnectionCount(String  dbName)
        throws DatabaseManagerException {
        LogicalDatabase logicalDatabase = findLogicalDatabase(dbName);

        logicalDatabase.resetMaxConnectionCount();
        return;
    }

    /**
     * Turn debugging on or off.
     *
     * @param   condition on of off.
     */
    public void setDebugLogging(boolean condition) {
        debug = condition;
    }

    /**
     * Return allReadOnly parameter (given in config file)
     * @return
     *   allReadOnly parameter
     */
    public boolean getAllReadOnly() {
        return dbmConf.isAllReadOnly();
    }  
   
    /**
     * Return AppName parameter (given in config file)
     * @return
     *   AppName parameter
     */
    public String  getAppName() {
        return appName;
    }  

    public DatabaseManagerConfiguration getDatabaseManagerConfiguration() {
        return dbmConf;  
    } 
     
    
    /**
     * Init chaches for specified Database
     * @param dbConfig Database section of app. configuration file.
     * @param fileName File (path) with list of data layer classes. 
     */
    private void initDatabaseCaches(Config dbConfig,String  fileName, ClassLoader  cls){
        String  tablesXmlFile=null;
        try {
            tablesXmlFile = dbConfig.getString("ClassList",fileName);
        } catch (ConfigException e) {}
        InputStream  tablesList = null;
        if (tablesXmlFile!=null){
            tablesList = getTablesXML(tablesXmlFile,cls);
        }else{
            tablesList = getTablesXML((String )null,cls);
        }
        if(tablesList!=null){
            initCaches(tablesList, cls);
        }
    }
    
    /**
     * Init chaches for all Databases specified in DatabaseManager section of application conf file.
     * @param config DatabaseManager section of app configuration file. 
     * @param cls application classLoader
     */
    public void initAllDodsCaches(Config config, ClassLoader  cls){
        Config dbConfig = null;
        String [] databases = null;
        String  tablesXmlFile = null;
        try {
            databases = config.getStrings("Databases");
            tablesXmlFile = config.getString("ClassList",(String )null);
        } catch (ConfigException e) {
            DODS.getLogChannel().write(Logger.DEBUG,"Unable to init caches: Error reading application configuation file");                  
        }
        if(databases!=null){
            for (int idx = 0; idx < databases.length; idx++) {
                String  dbName = databases[idx];
                try {
                    dbConfig = (Config) config.getSection("DB." + dbName);
                    initDatabaseCaches(dbConfig,tablesXmlFile, cls);
                } catch (Exception  except) {
                    DODS.getLogChannel().write(Logger.DEBUG,"Unable to init caches for database: "+dbName);         
                }
            }
        }
    }

    /**
     * Read xml file with list of classes in data layer.
     * @param fileName Path of xml file with data layer class list
     * @return InputStream with xml file.
     */
    private InputStream  getTablesXML(String  fileName, ClassLoader  cls){
        InputStream  res=null;
        if(fileName!=null){
            try{    
                File tablesFile = new File(fileName);
                if (tablesFile.isFile()) {
                    res= new FileInputStream (fileName);                 
                }
            }catch(Exception  e){
                DODS.getLogChannel().write(Logger.DEBUG,"Unable to read table names from file: "+fileName);
            }
        }else{
            try{
//              res = this.getClass().getClassLoader().getResourceAsStream("org/enhydra/dods/DODSClassList.xml");
                res = cls.getResourceAsStream("org/enhydra/dods/DODSClassList.xml");
            }catch(Exception  e){
                DODS.getLogChannel().write(Logger.DEBUG,"Unable to read table names from resources: org.enhydra.dods.DODSClassList.xml");           
            }       
        }
        return res;
    }
    
    /**
     * Initialize caches for all data layer classes specified in tablesList (xml file)
     * @param tablesList InputStream of XML file with data layer class names.
     */
    private void initCaches(InputStream  tablesList, ClassLoader  cls){
        Document  doc = null; 
        try {
            DocumentBuilderFactory  factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder  builder = factory.newDocumentBuilder();
            doc = builder.parse(tablesList);        
        } catch (Exception  e) {
            DODS.getLogChannel().write(Logger.DEBUG,"Unable to parse xml file with cache initialisation tables list.");         
        } 
        if(doc!=null){
            NodeList  classNodes = doc.getElementsByTagName("CLASS");
            for (int i = 0; i < classNodes.getLength(); i++) {
                String  className = ((Element )classNodes.item(i)).getAttribute("name");
                try {
                    Class  ce = cls.loadClass(className);
                    Method  mth = ce.getMethod("getCacheDodsTableName",new Class [] {});
                    mth.invoke(null, new Object []{});
                } catch (Exception  e1) {
                    e1.printStackTrace();
                    DODS.getLogChannel().write(Logger.DEBUG,"Unable to invoke caches for class: "+className);
                }
            }       
        }
    }

}
