
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 */
package org.enhydra.dods.statistics;

import java.util.Date ;

/**
 *
 * This class provides information about table and its statistics.
 *
 * @author    Tanja Jovanovic
 * @author    Nenad Vico
 * @author    Zorica Suvajdzin
 * @version   2.0  15.06.2003.
 */
public class TableStatistics implements Statistics {

    /**
     * Number of insert statements performed on the table.
     */
    protected int insertNum = 0;

    /**
     * Number of update statements performed on the table.
     */
    protected int updateNum = 0;

    /**
     * Number of delete statements performed on the table.
     */
    protected int deleteNum = 0;

    /**
     * Number of lazy loadings performed on the table.
     */
    protected int lazyLoadingNum = 0;

    /**
     * Time when the statistics was started.
     */
    protected Date  startTime;

    /**
     * Time when the statistics was stopped.
     */
    protected Date  stopTime;

    /**
     * Total number of non-oid queries performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     */
    protected int queryNum = 0;
    
    /**
     * Total number of queries by oid performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     */
    protected int queryByOIdNum = 0;

    /**
     * Average time for executing non-oid queries in milliseconds.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     */
    protected int averageQueryTime = 0;

    /**
     * Average time for executing queries by oid in milliseconds.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     */
    protected int averageQueryByOIdTime = 0;

    /**
     * Constructor().
     */
    public TableStatistics() {
        this.clears();
    }

    /**
     * Returns type of the statistics. In this case, this is TABLE_STATISTICS.
     *
     * @return Type of the statistics: TABLE_STATISTICS.
     */
    public int getStatisticsType() {
        return TABLE_STATISTICS;
    }

    /**
     * Returns number of insert statements performed on the table.
     *
     * @return Number of insert statements performed on the table.
     */
    public int getInsertNum() {
        return insertNum;
    }

    /**
     * Sets number of insert statements performed on the table.
     *
     * @param newInsertNum New number of insert statements performed on the
     * table.
     */
    public void setInsertNum(int newInsertNum) {
        this.insertNum = newInsertNum;
    }

    /**
     * Increases number of insert statemenst performed on the table.
     */
    public void incrementInsertNum() {
        insertNum++;
    }

    /**
     * Returns number of update statements performed on the table.
     *
     * @return Number of update statements performed on the table.
     */
    public int getUpdateNum() {
        return updateNum;
    }

    /**
     * Sets number of update statements performed on the table.
     *
     * @param newUpdateNum New number of update statements performed on the
     * table.
     */
    public void setUpdateNum(int newUpdateNum) {
        this.updateNum = newUpdateNum;
    }

    /**
     * Increases number of update statements performed on the table.
     */
    public void incrementUpdateNum() {
        updateNum++;
    }

    /**
     * Returns number of delete statements performed on the table.
     *
     * @return Number of delete statements performed on the table.
     */
    public int getDeleteNum() {
        return deleteNum;
    }

    /**
     * Sets number of delete statements performed on the table.
     *
     * @param newDeleteNum New number of delete statements performed on the
     * table.
     */
    public void setDeleteNum(int newDeleteNum) {
        this.deleteNum = newDeleteNum;
    }

    /**
     * Increases number of delete statements performed on table.
     */
    public void incrementDeleteNum() {
        deleteNum++;
    }

    /**
     * Returns number of DML operations (inserts, updates and deletes) performed
     * on the table.
     *
     * @return Number of DML operations performed on the table.
     */
    public int getDMLNum() {
        return insertNum + updateNum + deleteNum;
    }

    /**
     * Returns number of lazy loadings performed on the table.
     *
     * @return Number of lazy loadings performed on the table.
     */
    public int getLazyLoadingNum() {
        return lazyLoadingNum;
    }

    /**
     * Sets number of lazy loadings performed on the table.
     *
     * @param newLazyLoadingNum New number of lazy loadings performed on the
     * table.
     */
    public void setLazyLoadingNum(int newLazyLoadingNum) {
        this.lazyLoadingNum = newLazyLoadingNum;
    }

    /**
     * Increases number of lazy loadings performed on the table for one.
     */
    public void incrementLazyLoadingNum() {
        lazyLoadingNum++;
    }

    /**
     * Returns time when the statistics was started.
     *
     * @return Time when the statistics was started.
     */
    public Date  getStartTime() {
        return startTime;
    }

    /**
     * Sets time when the statistics starts.
     *
     * @param startTime Time when the statistics starts.
     */
    public void setStartTime(Date  startTime) {
        this.startTime = startTime;
    }

    /**
     * Returns time when the statistics was stopped.
     *
     * @return Time when the statistics was stopped.
     */
    public Date  getStopTime() {
        return stopTime;
    }

    /**
     * Sets time when the statistics stops.
     *
     * @param stopTime time when the statistics was stops.
     */
    public void setStopTime(Date  stopTime) {
        this.stopTime = stopTime;
    }

    /**
     * Sets stop time to current time.
     */
    public void stopTime() {
        stopTime = new Date ();
    }

    /**
     * Returns total number of non-oid queries performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     *
     * @return Total number of non-oid queries performed on the table.
     */
    public int getQueryNum() {
        return queryNum; 
    } 

    /**
     * Sets total number of non-oid queries performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     *
     * @param newQueryNum New total number of non-oid queries performed on
     * the table.
     */
    public void setQueryNum(int newQueryNum) {
        this.queryNum = newQueryNum;
    }

    /**
     * Increases total number of non-oid queries performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     */
    public void incrementQueryNum() {
        queryNum++;
    }

    /**
     * Returns total number of queries by oid performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     *
     * @return Total number of queries by oid performed performed on the table.
     */
    public int getQueryByOIdNum() {
        return queryByOIdNum;
    }

    /**
     * Sets total number of queries by oid performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     *
     * @param newQueryByOIdNum New total number of queries by oid performed on
     * the table.
     */
    public void setQueryByOIdNum(int newQueryByOIdNum) {
        this.queryByOIdNum = newQueryByOIdNum;
    }

    /**
     * Increases total number of queries by oid performed on the table for one.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     */
    public void incrementQueryByOIdNum() {
        queryByOIdNum++;
    }

    /**
     * Returns average time needed for executing non-oid query.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     *
     * @return Average time needed for executing non-oid query.
     */
    public int getQueryAverageTime() {
        return averageQueryTime;
    }

    /**
     * Updates average time needed for executing non-oid queries.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     *
     * @param newTime New query time in miliseconds.
     */
    public void updateQueryAverageTime(int newTime) {
        long sum = 0;

        if (queryNum > 0) {
            sum = (queryNum - 1) * averageQueryTime + newTime;
            sum = sum / queryNum;
            averageQueryTime = (new Long (sum)).intValue();
        }
    }

    /**
     * Returns average time needed for executing oid query.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     *
     * @return Average time needed for executing oid query.
     */
    public int getQueryByOIdAverageTime() {
        return averageQueryByOIdTime;
    }

    /**
     * Updates average time for executing OId queries and increments number
     * of them by paramether <code>no</code>.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     *
     * @param newTime New query time in miliseconds for no queries by OId.
     * @param no Number of queries by OId.
     */
    public void updateQueryByOIdAverageTime(int newTime, int no) {
        if (no > 0) {
            long sum = 0;

            if (queryByOIdNum > 0) {
                sum = (queryByOIdNum) * averageQueryByOIdTime + newTime;
                queryByOIdNum += no;
                sum = sum / queryByOIdNum;
                averageQueryByOIdTime = (new Long (sum)).intValue();
            } else {
                sum = newTime;
                queryByOIdNum += no;
                sum = sum / queryByOIdNum;
                averageQueryByOIdTime = (new Long (sum)).intValue();
            }
        }
    }

    /**
     * Clears statistics.
     */
    public void clears() {
        insertNum = 0;
        updateNum = 0;
        deleteNum = 0;
        lazyLoadingNum = 0;
        startTime = new Date ();
        stopTime = new Date ();
        queryNum = 0;
        queryByOIdNum = 0;
        averageQueryTime = 0;
        averageQueryByOIdTime = 0;
    }

    /**
     * Clears statistics.
     */
    public void clear() {
        this.clears();
    }

    /**
     * Returns query statistics for table without cache.
     * Since this is table without cache parameter type has no meaning.
     *
     * @param type Value 0 (org.enhydra.dods.cache.CacheConstants.DATA_CACHE)
     * for DO (data object) cache,
     * value 1 (org.enhydra.dods.cache.CacheConstants.SIMPLE_QUERY_CACHE) for
     * simple query cache and value 2
     * (org.enhydra.dods.cache.CacheConstants.COMPLEX_QUERY_CACHE) for complex
     * query cache.
     * @return Query statistics table.
     */
    public CacheStatistics getCacheStatistics(int type) {
        return null;
    }
}
