
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache;

import java.util.Vector ;

/**
 * This abstract class contains data and mechanisms needed for caching data
 * objects (or DataStruct objects) and queries and provides cache configuration
 * and administration.
 *
 * @author    Tanja Jovanovic
 * @author    Sinisa Milosevic
 * @version   1.0  05.08.2003.
 */
public abstract class TransactionQueryCache extends DOCache {

    /**
     * Creates new QueryCacheItem instance.
     *
     * @param dbName Database name.
     * @return Created QueryCacheItem.
     */
    public abstract QueryCacheItem newQueryCacheItemInstance(String  dbName);

    /**
     * Returns QueryCacheItem object for specified database and simple query,
     * if exists, otherwise null.
     *
     * @param dbName Database name.
     * @param query Query in form of String.
     * @return QueryCacheItem object.
     */
    public abstract QueryCacheItem getSimpleQueryCacheItem(String  dbName, String  query);

    /**
     * Returns QueryCacheItem object for specified database and complex query,
     * if exists, otherwise null.
     *
     * @param dbName Database name.
     * @param query Query in form of String.
     * @return QueryCacheItem object.
     */
    public abstract QueryCacheItem getComplexQueryCacheItem(String  dbName, String  query);

    /**
     * Adds simple query to simple query cache.
     *
     * @param queryItem Query that will be added to simple query cache.
     * @return Query added to simple query cache.
     */
    public abstract QueryCacheItem addSimpleQuery(QueryCacheItem queryItem);

    /**
     * Removes simple query from simple query cache.
     *
     * @param queryItem QueryItem that will be removed from simple query cache.
     * @return QueryItem removed from simple query cache.
     */
    public abstract QueryCacheItem removeSimpleQuery(QueryCacheItem queryItem);

    /**
     * Adds complex query to complex query cache.
     *
     * @param queryItem Query that will be added to complex query cache.
     * @return Query added to complex query cache.
     */
    public abstract QueryCacheItem addComplexQuery(QueryCacheItem queryItem);

    /**
     * Removes complex query from complex query cache.
     *
     * @param queryItem Query that will be removed from complex query cache.
     * @return Query removed from complex query cache.
     */
    public abstract QueryCacheItem removeComplexQuery(QueryCacheItem queryItem);

    /**
     * Returns query results from simple query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in simple query cache.
     * @return Query results retrieved from simple cache, or null, if there are
     * no results retrieved from simple query cache.
     */
    public abstract QueryResult getSimpleQueryResults(String  dbName, String  query);

    /**
     * Returns query results from simple query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in simple query cache.
     * @param limit Specified number of results (database limit and read skip).
     * @param maxdb Number of rows retrieved from database (or cache).
     * @return Query results retrieved from simple cache, or null, if there are
     * no results retrieved from simple query cache.
     */
    public abstract QueryResult getSimpleQueryResults(String  dbName, String  query, int limit, int maxdb);

    /**
     * Returns query results from simple query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in simple query cache.
     * @param limit Specified number of results (database limit and read skip).
     * @param maxdb Number of rows retrieved from database (or cache).
     * @param unique If true, only unique results are returned.
     * @return Query results retrieved from simple cache, or null, if there are
     * no results retrieved from simple query cache.
     */
    public abstract QueryResult getSimpleQueryResults(String  dbName, String  query, int limit, int maxdb, boolean unique);

    /**
     * Returns query results from complex query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in complex query cache.
     * @return Query results retrieved from complex cache, or null, if there are
     * no results retrieved from complex query cache.
     */
    public abstract QueryResult getComplexQueryResults(String  dbName, String  query);

    /**
     * Returns query results from complex query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in complex query cache.
     * @param limit Specified number of results (database limit and read skip).
     * @param maxdb Number of rows retrieved from database (or cache).
     * @return Query results retrieved from complex cache, or null, if there are
     * no results retrieved from complex query cache.
     */
    public abstract QueryResult getComplexQueryResults(String  dbName, String  query, int limit, int maxdb);

    /**
     * Returns query results from complex query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in complex query cache.
     * @param limit Specified number of results (database limit and read skip).
     * @param maxdb Number of rows retrieved from database (or cache).
     * @param unique If true, only unique results are returned.
     * @return Query results retrieved from complex cache, or null, if there are
     * no results retrieved from complex query cache.
     */
    public abstract QueryResult getComplexQueryResults(String  dbName, String  query, int limit, int maxdb, boolean unique);

    /**
     * Returns query results from simple or complex query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in simple and complex
     * query cache.
     * @return Query results retrieved from simple or complex cache, or null,
     * if there are no results retrieved from simple and complex query cache.
     */
    public abstract QueryResult getQueryResults(String  dbName, String  query);
    
    /**
     *
     */
    public abstract void removeEntries(Vector  vec); 

    /**
     *
     * @param tableClass -
     */
    public abstract void removeEntries(Class  tableClass);

    /**
     *
     */
    public abstract void emptyEntries(Vector  vec, boolean incrementVersion);
    
    /**
     * Dumps data structs for all instances of tableClass in transaction
     * cache.
     *
     * @param tableClass - Class object whose instances will be emptied
     */
    public abstract void emptyEntries(Class  tableClass) ;
    
}
