
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: RDBRow.java,v 1.1 2004/09/03 13:42:41 sinisa Exp $
 */
package com.lutris.dods.builder.generator.query;

/**
 * An instance of RDBRow is returned by the QueryBuilder.getNextRow() method.
 * An RDBRow contains RDBColumnValue objects corresponding to the 
 * RDBColumn objects used to construct the instance of QueryBuilder.
 * in a particular table in the database.
 * @see QueryBuilder
 * @author Jay Gunter
 */
public class RDBRow {
    private RDBColumnValue[] values;

    /**
     * Constructor used only by QueryBuilder, never the application developer.
     * 
     * @param vals the column values in the current row
     * @see QueryBuilder
     * author Jay Gunter
     */
    public RDBRow(RDBColumnValue[] vals) {
        if (null == vals) {
            values = new RDBColumnValue[0];
        } else {
            values = vals;
        }
    }

    /**
     * Returns an RDBColumnValue object for a column in the current row.
     * Application developers are encouraged to use the 
     * RDBRow.get(RDBColumn) method instead because specifying the 
     * desired column value using RDBColumn is less error-prone
     * than using an int offset into the columns.
     * 
     * @return the value of the ith column
     * @param i the index into the row of the column value desired.
     * @exception InvalidRowColumnException If i is not a valid column number.
     * @see QueryBuilder
     * author Jay Gunter
     */
    public RDBColumnValue get(int i) 
        throws InvalidRowColumnException {
        if (!(0 <= i && i < values.length)) {
            throw new InvalidRowColumnException("" + i);
        }
        return values[i];
    }

    /**
     * Returns an RDBColumnValue object for a column in the current row.
     * This is the preferred method for use by application developers.
     * 
     * @return the value of the specified column
     * @param col the column value desired.
     * @exception InvalidRowColumnException 
     * If col is not a valid column in the row.
     * @see QueryBuilder
     * author Jay Gunter
     */
    public RDBColumnValue get(RDBColumn col)
        throws InvalidRowColumnException {
        if (null == col) {
            throw new InvalidRowColumnException("Null RDBColumn");
        }
        for (int i = 0; i < values.length; i++) {
            if (values[i].equals(col)) {
                return values[i];
            }
        }
        throw new InvalidRowColumnException(col.getFullColumnName());
    }
}
