
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: RDBColumnValue.java,v 1.1 2004/09/03 13:42:41 sinisa Exp $
 */
package com.lutris.dods.builder.generator.query;

import java.math.BigDecimal ;

/**
 * After a Vector of RDBColumn objects is used to construct
 * an instance of QueryBuilder, the QueryBuilder.getNextRow() method
 * is used to return an RDBRow object which contains RDBColumnValue objects
 * representing the column values in the returned row.
 * An instance of RDBColumnValue is returned by the RDBRow.get() method.
 * The value contained in the RDBColumnValue object is extracted
 * using RDBColumnValue.get methods.
 * @see QueryBuilder
 * @author Jay Gunter
 */
public class RDBColumnValue extends RDBColumn {

    /**
     * The constructor for RDBColumnValue is used only by QueryBuilder,
     * never by the application developer.
     * 
     * return the RDBColumnValue object containing the value for the 
     * RDBColumn specified.
     * @param column the column specified by QueryBuilder during row access.
     * @param val the value for the column in the currently accessed row.
     * @see QueryBuilder 
     * author Jay Gunter
     */
    public RDBColumnValue(RDBColumn column, Object  val) {
        super(column.table, column.name);
        value = val;
    }
    private Object  value;

    /**
     * Return the column value as an Object.
     * Rarely used by the application developer
     * (only when the type of the column value is not known.)
     * 
     * @return the column value as an Object whose type can be interrogated.
     * @see QueryBuilder
     * author Jay Gunter
     */
    public Object  getValue() {
        return value;
    }

    /**
     * Used only by the QueryBuilder.
     * Sets the value for the RDBColumnValue object.
     * 
     * @param o the value as an Object
     * author Jay Gunter
     */
    public void   setValue(Object  o) {
        value = o;
    }

    /**
     * Retreive column value as a String.
     * 
     * @return the column value as a String.
     * @exception ColumnTypeException If the column value is not a String
     * (if the column type is not CHAR, VARCHAR, VARCHAR2, etc.)
     * @see QueryBuilder
     * author Jay Gunter
     */
    public String  getString()
        throws ColumnTypeException {
        if (!(value instanceof String )) {
            throw new ColumnTypeException(this, "String");
        }
        return (String ) value;
    }

    /**
     * Retreive column value as an int.
     * 
     * @return the column value as an int.
     * @param nullOk if false and the column in the row contains 
     * a database NULL value, a ColumnNullException is thrown.
     * If true and the column in the row contains a NULL, 0 is returned.
     * @exception ColumnTypeException If the column value is not an Integer
     * (if the column type is not INTEGER, LONG, etc.) or if NULL is not
     * an allowed value (nullOk==false.)
     * @see QueryBuilder
     * author Jay Gunter
     */
    public int getInteger(boolean nullOk)
        throws ColumnTypeException, ColumnNullException {
        if (!(value instanceof Integer )) {
            throw new ColumnTypeException(this, "Integer");
        }
        if (null == value) {
            if (!nullOk) {
                throw new ColumnNullException(this);
            } else {
                return 0;
            }
        }
        return ((Integer ) value).intValue();
    }

    /**
     * Retreive column value as an int.
     * 
     * @return the column value as an int.  If the column value is NULL,
     * 0 is returned.
     * @exception ColumnTypeException If the column value is not an Integer
     * (if the column type is not INTEGER, LONG, etc.)
     * @see QueryBuilder
     * author Jay Gunter
     */
    public Integer  getInteger()
        throws ColumnTypeException {
        if (!(value instanceof Integer )) {
            throw new ColumnTypeException(this, "Integer");
        }
        return (Integer ) value;
    }

    /**
     * Retreive column value as an double.
     * 
     * @return the column value as an double.
     * @param nullOk if false and the column in the row contains 
     * a database NULL value, a ColumnNullException is thrown.
     * If true and the column in the row contains a NULL, 0 is returned.
     * @exception ColumnTypeException If the column value is not a Double
     * (if the column type is not FLOAT, DOUBLE, NUMERIC, DECIMAL, etc.)
     * or if NULL is not an allowed value (nullOk==false.)
     * @see QueryBuilder
     * author Jay Gunter
     */
    public double getDouble(boolean nullOk)
        throws ColumnTypeException, ColumnNullException {
        if (null == value) {
            if (!nullOk) {
                throw new ColumnNullException(this);
            } else {
                return 0;
            }
        }
        Double  d = getDouble();

        return d.doubleValue();
    }

    /**
     * Retreive column value as an double.
     * 
     * @return the column value as an double.  If the column value is NULL,
     * 0 is returned.
     * @exception ColumnTypeException If the column value is not a Double
     * (if the column type is not FLOAT, DOUBLE, NUMERIC, DECIMAL, etc.)
     * @see QueryBuilder
     * author Jay Gunter
     */
    public Double  getDouble()
        throws ColumnTypeException {
        if (value instanceof Double ) {
            return (Double ) value;
        }
        if (value instanceof BigDecimal ) {
            BigDecimal  b = (BigDecimal ) value;

            return (Double ) new Double (b.toString());
        }
        throw new ColumnTypeException(this, "Double/BigDecimal");
    }

    /**
     * Retreive column value as a BigDecimal.
     * 
     * @return the column value as a BigDecimal.
     * If the column value is NULL, 0 is returned.
     * @exception ColumnTypeException If the column value is not a BigDecimal
     * (if the column type is not NUMERIC, or DECIMAL.)
     * @see QueryBuilder
     * author Jay Gunter
     */
    public BigDecimal  getBigDecimal()
        throws ColumnTypeException {
        // Accomodate InstantDB which returns Long objects for DECIMAL columns.
 if (value instanceof Long ) {
            return new BigDecimal (value.toString());
        }
        if (!(value instanceof BigDecimal )) {
            throw new ColumnTypeException(this, "BigDecimal");
        }
        return (BigDecimal ) value;
    }
}
