
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package com.lutris.appserver.server.sql.standard;

import org.enhydra.dods.CommonConstants;
import org.enhydra.dods.DODS;
import org.enhydra.dods.cache.CacheConstants;
import org.enhydra.dods.exceptions.AssertionDataObjectException;

import com.lutris.appserver.server.sql.DatabaseManagerConfiguration;
import com.lutris.appserver.server.sql.StandardDatabaseManager;
import com.lutris.util.Config;


/**
 * TableConfiguration contains parameters about table configuration. It reads 
 * these parameters from application's configuration file, sets them to new 
 * values or returns their current values.
 *
 * @author    Sinisa Milosevic
 * @version   1.0  01.09.2003.  
 */
 
public class DatabaseConfiguration implements CacheConstants, DriverSpecificConstants {
    
    /**
     * 
     * Database name
     */
    private String  dbName = null;
    
    /**
     * If table is read-only, parameter readOnly is true, otherwise false.
     * Default value is false.
     */
    private boolean readOnly = CacheConstants.DEFAULT_All_READONLY;
    
    /**
     * If table supports lazy-loading, parameter lazyLoading is true, otherwise 
     * false.
     * Default value is false.
     */
    private boolean lazyLoading = CacheConstants.DEFAULT_LAZY_LOADING;
    
    
    
    /**
     * If table supports caseSensitive, parameter caseSensitive is true, otherwise 
     * false.
     * Default value is false.
     */
    private boolean caseSensitive = CacheConstants.DEFAULT_CASE_SENSITIVE;
    
    
    
    
    /**
     * Maximal time in miliseconds for Event log.
     * If query is executed longer that this time, its select statement and 
     * execution time are written to Event log for future analyzes. 
     */
    private int maxExecuteTime = CacheConstants.DEFAULT_MAX_EXECUTE_TIME;
    
    private double reserveFactor = CacheConstants.DEFAULT_RESERVE_FACTOR;
   
    private int maxCacheSize = CacheConstants.DEFAULT_MAX_CACHE_SIZE;
   
    private int maxSimpleCacheSize = CacheConstants.DEFAULT_MAX_SIMPLE_QUERY_CACHE_SIZE;
   
    private int maxComplexCacheSize = CacheConstants.DEFAULT_MAX_COMPLEX_QUERY_CACHE_SIZE;  
    
    private int maxMultiJoinCacheSize = CacheConstants.DEFAULT_MAX_MULTI_JOIN_QUERY_CACHE_SIZE;  
   
    private double cachePercentage = CacheConstants.DEFAULT_CACHE_PERCENTAGE;
    private boolean transactionCheck = CacheConstants.DEFAULT_TRANSACTION_CHECK;      
   
    private boolean deleteCheckVersion = CacheConstants.DEFAULT_DELETE_CHECK_VERSION;      
  
    private boolean autoSave = CacheConstants.DEFAULT_AUTO_SAVE;      
   
    private boolean autoSaveCreateVirgin = CacheConstants.DEFAULT_AUTO_SAVE_CREATE_VIRGIN;      
   
    private boolean autoWrite = CacheConstants.DEFAULT_AUTO_WRITE;      
    
    private boolean transactionCaches = CacheConstants.DEFAULT_TRANSACTION_CACHES;      
   
    private int deadlockWaitTime = CacheConstants.DEFAULT_DEADLOCK_READ_TIME;      
   
    private int deadlockRetryCount = CacheConstants.DEFAULT_DEADLOCK_RETRY_NUMBER;
          
    private boolean initialAllCaches = CacheConstants.DEFAULT_INITIAL_ALL_CACHES;     
   
    private int defaultFetchSize = CacheConstants.DEFAULT_DEFAULT_FETCH_SIZE;         
   
    private int defaultQueryTimeout = CacheConstants.DEFAULT_QUERY_TIMEOUT;            
                            
    private boolean defaultSelectOids = CacheConstants.DEFAULT_SELECT_OIDS;            
    
    private boolean defaultIncrementVersions = CacheConstants.DEFAULT_INCREMENT_VERSIONS;                

    private Boolean  useCursorName = null ;
    
    private boolean disableFetchSizeWithMaxRows = DriverSpecificConstants.DEFAULT_DISABLE_FETCH_SIZE_WITH_MAX_ROWS;
    
    private int resultSetType = DriverSpecificConstants.DEFAULT_RESULT_SET_TYPE;
    
    private int resultSetConcurrency = DriverSpecificConstants.DEFAULT_RESULT_SET_CONCURRENCY;
  
    private String  DBTransactionFactoryName = null; 
    
    private String  connectionAllocatorName = null;
    
    private String  DBConnectionFactoryName = null;
    
    private String  queryCacheImplClass = null;

    private int fullCacheCountLimit = CacheConstants.DEFAULT_FULL_CACHE_COUNT_LIMIT;

    private int initialCacheFetchSize = CacheConstants.DEFAULT_INITIAL_CACHE_FETCH_SIZE;

    private int initialDSCacheSize = CacheConstants.DEFAULT_INITIAL_DS_CACHE_SIZE;

    private String  initCachesResultSetType = null;
    
    private String  initCachesResultSetConcurrency = null;       
    
    private boolean sqlBatch = CommonConstants.DEFAULT_SQL_BATCH;
    
    private boolean caseInsensitiveDatabase = CommonConstants.DEFAULT_CASE_INSENSITIVE_DATABASE;
    
    private Integer  queryTimeLimit = CommonConstants.DEFAULT_QUERY_TIME_LIMIT;

    // - XA Support -
 
    private int xaDefaultTimeout = CommonConstants.DEFAULT_XA_DEFAULT_TIMEOUT;
    
    private int xaUsageCase      = CommonConstants.DEFAULT_XA_USAGE_CASE;
    
    private String  xaTransactonManagerLookupName = CommonConstants.DEFAULT_XA_TM_LOOKUP_NAME;
    
    private String  xaUserTransactonLookupName    = CommonConstants.DEFAULT_XA_USER_TRANSACTION_LOOKUP_NAME;
    
    private String  xaWrappedTransImplFactory     = CommonConstants.DEFAULT_XA_WRAPPED_TRANS_IMPL_FACTORY;
    
    private String  xaJtaSupport = CommonConstants.DEFAULT_XA_JTA_SUPPORT;
    
    // - XA Support -

    private String  dodsCacheFactory = CacheConstants.DEFAULT_DODS_CACHE_FACTORY;
    
    /**
     * Returns readOnly parameter.
     *
     * @return true if the table is read-only, otherwise false.
     */
    public boolean isAllReadOnly() {
        return readOnly;
    }
    
    public DatabaseConfiguration(String  database) {
        if (database != null) {
            dbName = database;
        } else {
            dbName = DODS.getDatabaseManager().getDefaultDB();
        } 
        init(((StandardDatabaseManager)DODS.getDatabaseManager()).getDatabaseManagerConfiguration());     
    }
   
    public DatabaseConfiguration(String  database, DatabaseManagerConfiguration DbManagerConf) {
        if (database != null) {
            dbName = database;
        } else {
            dbName = DODS.getDatabaseManager().getDefaultDB();
        }
        init(DbManagerConf);     
    }

   
    private void init(DatabaseManagerConfiguration stmc) {     
        readOnly       = stmc.isAllReadOnly();
        lazyLoading    = stmc.isLazyLoading();
        caseSensitive  = stmc.getCaseSensitive();
        maxExecuteTime = stmc.getMaxExecuteTime();
        reserveFactor  = stmc.getReserveFactor();
        cachePercentage = stmc.getCachePercentage();
        maxCacheSize    = stmc.getMaxCacheSize();
        maxSimpleCacheSize    = stmc.getMaxSimpleCacheSize();
        maxComplexCacheSize   = stmc.getMaxComplexCacheSize();
        maxMultiJoinCacheSize = stmc.getMaxMultiJoinCacheSize();
        transactionCheck   = stmc.getTransactionCheck();
        deleteCheckVersion = stmc.getDeleteCheckVersion();
        autoSave = stmc.getAutoSave();
        autoSaveCreateVirgin = stmc.getAutoSaveCreateVirgin();
        autoWrite = stmc.getAutoWrite();
        transactionCaches  = stmc.getTransactionCaches();
        deadlockWaitTime   = stmc.getDeadlockWaitTime();
        deadlockRetryCount = stmc.getDeadlockRetryCount();
        initialAllCaches   = stmc.getInitAllCaches();
        defaultFetchSize   = stmc.getDefaultFetchSize();
        defaultQueryTimeout = stmc.getQueryTimeout();                  
        defaultSelectOids = stmc.getSelectOids();                  
        defaultIncrementVersions = stmc.getIncrementVersions();
        useCursorName = stmc.getUseCursorName();  
        DBTransactionFactoryName = stmc.getDBTransactionFactoryName();     
        connectionAllocatorName  = stmc.getConnectionAllocatorName(); 
        DBConnectionFactoryName  = stmc.getDBConnectionFactoryName();     
        queryCacheImplClass      = stmc.getQueryCacheImplClass();      
        dodsCacheFactory         = stmc.getDodsCacheFactory();
        fullCacheCountLimit      = stmc.getFullCacheCountLimit();     
        initialCacheFetchSize    = stmc.getInitialCacheFetchSize();
        initialDSCacheSize       = stmc.getInitialDSCacheSize();    
        initCachesResultSetType        = stmc.getInitCachesResultSetType();
        initCachesResultSetConcurrency = stmc.getInitCachesResultSetConcurrency();
        sqlBatch = stmc.isSqlBatch();  
        caseInsensitiveDatabase = stmc.isCaseInsensitiveDatabase(); 
        queryTimeLimit          = stmc.getQueryTimeLimit();
        xaDefaultTimeout              = stmc.getXaDefaultTimeout();
        xaUsageCase                   = stmc.getXaUsageCase();
        xaTransactonManagerLookupName = stmc.getXaTransactonManagerLookupName();
        xaUserTransactonLookupName    = stmc.getXaUserTransactonLookupName();
        xaWrappedTransImplFactory     = stmc.getXaWrappedTransImplFactory();
        xaJtaSupport                  = stmc.getXaJtaSupport();
    }  

    /**
     * Sets readOnly parameter to value newReadOnly.
     *
     * @param newReadOnly New read-only value.
     * @exception AssertionDataObjectException If new value of read-only is 
     * false, but all application's tables are read-only.
     */
    protected void setReadOnly(boolean newReadOnly) throws AssertionDataObjectException {
        readOnly = newReadOnly ;
    }
    
    /**
     * Returns lazyLoading parameter.
     *
     * @return true if the table supports lazy-loading, otherwise false.
     */
    public boolean isLazyLoading() {
        return lazyLoading;
    }
    
    /**
     * Sets lazyLoading parameter to value newlazyLoading.
     *
     * @param newlazyLoading New lazy-loading value.
     */
    protected void setLazyLoading(boolean newlazyLoading) {
        lazyLoading = newlazyLoading;
    }

    /**
     * Returns maxExecuteTime parameter.
     *
     * @return Maximal time in miliseconds for Event log.
     */
    public int getMaxExecuteTime() {
        return maxExecuteTime;
    }
    
    /**
     * Sets maxExecuteTime parameter to value newMaxExecuteTime.
     *
     * @param newMaxExecuteTime New maxExecuteTime value.
     */
    protected void setMaxExecuteTime(int newMaxExecuteTime) {
        maxExecuteTime = newMaxExecuteTime;
    }
    
    /**
     * Returns maxCacheSize parameter.
     *
     * @return Maximal cache size.
     */
    public int getMaxCacheSize() {
        return maxCacheSize;
    }
    
    /**
     * Sets maxCacheSize parameter to value newMaxCacheSize.
     *
     * @param newMaxCacheSize New newMaxCacheSize value.
     */
    protected void setMaxCacheSize(int newMaxCacheSize) {
        maxCacheSize = newMaxCacheSize;
    }
    
    /**
     * Returns reserveFactor parameter.
     *
     * @return reserveFactor double value
     */
    public double getReserveFactor() {
        return reserveFactor;
    }
    
    /**
     * Sets reserveFactor parameter to value newReserveFactor.
     *
     * @param newReserveFactor New reserveFactor value.
     */
    protected void setReserveFactor(double newReserveFactor) {
        reserveFactor = newReserveFactor;
    }   

    /**
     *
     */
    protected void setCachePercentage(double percent) {
        cachePercentage = percent;
    }

    /**
     *
     */
    public double getCachePercentage() {
        return cachePercentage;
    }

    /**
     * Returns maxComplexCacheSize parameter.
     *
     * @return Maximal complex query cache size.
     */
    public int getMaxComplexCacheSize() {
        return maxComplexCacheSize;
    }
    
    /**
     * Sets maxComplexCacheSize parameter to value newMaxCacheSize.
     *
     * @param newMaxCacheSize New newMaxComplexCacheSize value.
     */
    protected void setMaxComplexCacheSize(int newMaxCacheSize) {
        maxComplexCacheSize = newMaxCacheSize;
    }
    /**
     * Returns maxMultiJoinCacheSize parameter.
     *
     * @return Maximal multi join query cache size.
     */
    public int getMaxMultiJoinCacheSize() {
        return maxMultiJoinCacheSize;
    }
    
    /**
     * Sets maxMultiJoinCacheSize parameter to value newMaxCacheSize.
     *
     * @param newMaxCacheSize New maxMultiJoinCacheSize value.
     */
    protected void setMaxMultiJoinCacheSize(int newMaxCacheSize) {
        maxMultiJoinCacheSize = newMaxCacheSize;
    }
    
    /**
     * Returns maxSimpleCacheSize parameter.
     *
     * @return Maximal cache size.
     */
    public int getMaxSimpleCacheSize() {
        return maxSimpleCacheSize;
    }
    
    /**
     * Sets maxSimpleCacheSize parameter to value newMaxCacheSize.
     *
     * @param newMaxCacheSize New newMaxCacheSize value.
     */
    protected void setMaxSimpleCacheSize(int newMaxCacheSize) {
        maxSimpleCacheSize = newMaxCacheSize;
    }       

    /**
     * Returns TransactionCheck parameter.
     *
     * @return TransactionCheck value.
     */
    public boolean getTransactionCheck() {
        return transactionCheck;
    }
    
    /**
     * Sets TransactionCheck parameter to value newCheck.
     *
     * @param newCheck New TransactionCheck value.
     */
    protected void setTransactionCheck(boolean newCheck) {
        transactionCheck = newCheck;
    }       

    /**
     * Returns DeleteCheckVersion parameter.
     *
     * @return DeleteCheckVersion parameter value.
     */
    public boolean getDeleteCheckVersion() {
        return deleteCheckVersion;
    }
    
    /**
     * Sets DeleteCheckVersion parameter to value newCheck.
     *
     * @param newCheck New DeleteCheckVersion value.
     */
    protected void setDeleteCheckVersion(boolean newCheck) {
        deleteCheckVersion = newCheck;
    }       

    /**
     * Returns AutoSave parameter.
     *
     * @return AutoSave parameter value.
     */
    public boolean getAutoSave() {
        return autoSave;
    }
    
    /**
     * Sets AutoSave parameter to value newCheck.
     *
     * @param newCheck New AutoSave value.
     */
    protected void setAutoSave(boolean newCheck) {
        autoSave = newCheck;
    }
    
    /**
     * Returns AutoSaveCreateVirgin parameter.
     *
     * @return AutoSaveCreateVirgin parameter value.
     */
    public boolean getAutoSaveCreateVirgin() {
        return autoSaveCreateVirgin;
    }
    
    /**
     * Sets AutoSaveCreateVirgin parameter to value newCheck.
     *
     * @param newCheck New AutoSaveCreateVirgin value.
     */
    protected void setAutoSaveCreateVirgin(boolean newCheck) {
        autoSaveCreateVirgin = newCheck;
    }

    /**
     * Returns AutoWrite parameter.
     *
     * @return AutoWrite parameter value.
     */
    public boolean getAutoWrite() {
        return autoWrite;
    }
    
    /**
     * Sets AutoWrite parameter to value newCheck.
     *
     * @param newCheck New AutoWrite value.
     */
    protected void setAutoWrite(boolean newCheck) {
        autoWrite = newCheck;
    }       
    
    /**
     * Returns TransactionCaches parameter.
     *
     * @return TransactionCaches parameter value.
     */
    public boolean getTransactionCaches() {
        return transactionCaches;
    }
    
    /**
     * Sets TransactionCaches parameter to value newCheck.
     *
     * @param newCheck New TransactionCaches value.
     */
    protected void setTransactionCaches(boolean newCheck) {
        transactionCaches = newCheck;
    }       
    
    /**
     * Returns DeadlockWaitTime parameter.
     *
     * @return DeadlockWaitTime parameter value.
     */
    public int getDeadlockWaitTime() {
        return deadlockWaitTime;
    }
    
    /**
     * Sets DeadlockWaitTime parameter to value waitTime.
     *
     * @param waitTime New DeadlockWaitTime value.
     */
    protected void setDeadlockWaitTime(int waitTime) {
        deadlockWaitTime = waitTime;
    }       
    
    /**
     * Returns DeadlockRetryCount parameter.
     *
     * @return DeadlockRetryCount parameter value.
     */
    public int getDeadlockRetryCount() {
        return deadlockRetryCount;
    }
    
    /**
     * Sets DeadlockRetryCount parameter to value newValue.
     *
     * @param newValue New DeadlockRetryCount value.
     */
    protected void setDeadlockRetryCount(int newValue) {
        deadlockRetryCount = newValue;
    }                           

    /**
     * Returns InitAllCaches parameter.
     *
     * @return InitAllCaches parameter value.
     */
    public boolean getInitAllCaches() {
        return initialAllCaches;
    }
    
    /**
     * Sets InitAllCaches parameter to value newValue.
     *
     * @param newValue New InitAllCaches value.
     */
    protected void setInitAllCaches(boolean newValue) {
        initialAllCaches = newValue;
    }

    /**
     * Returns DefaultFetchSize parameter.
     *
     * @return DefaultFetchSize parameter value.
     */
    public int getDefaultFetchSize() {
        return defaultFetchSize;
    }
    
    /**
     * Sets DefaultFetchSize parameter to value newValue.
     *
     * @param newValue New DefaultFetchSize value.
     */
    protected void setDefaultFetchSize(int newValue) {
        defaultFetchSize = newValue;
    }   
    
    /**
     * Returns QueryTimeout parameter.
     *
     * @return QueryTimeout parameter value.
     */
    public int getQueryTimeout() {
        return defaultQueryTimeout;
    }
    
    /**
     * Sets QueryTimeout parameter to value newValue.
     *
     * @param newValue New QueryTimeout value.
     */
    protected void setQueryTimeout(int newValue) {
        defaultQueryTimeout = newValue;
    }
        
        
    /**
     * Returns selectOids parameter.
     *  
     * @return selectStatement parameter value.
     */
    public boolean getSelectOids() {
        return defaultSelectOids;
    }
    
    /**
     * Sets selectOids parameter.
     *
     * @param newValue New selectOids value.
     */
    protected void setSelectOids(boolean newValue) {
        defaultSelectOids = newValue;
    }
        


    /**
     * Returns IncrementVersions parameter.
     *  
     * @return IncrementVersions parameter value.
     */
    public boolean getIncrementVersions() {
        return defaultIncrementVersions;
    }
    
    /**
     * Sets IncrementVersions parameter.
     *
     * @param newValue New IncrementVersions value.
     */
    protected void setIncrementVersions(boolean newValue) {
        defaultIncrementVersions = newValue;
    }
    
    
    /**
     * Returns UseCursorName parameter.
     * 
     * @return UseCursorName parameter value.
     */
    public Boolean  getUseCursorName() {
        return useCursorName;
    }

    /**
     * Sets UseCursorName parameter.
     * 
     * @param newValue New UseCursorName value.
     */
    public void setUseCursorName(boolean newValue) {
        useCursorName = new Boolean (newValue);
    }

    /**
     * Sets UseCursorName parameter.
     * 
     * @param newValue New UseCursorName value.
     */
    public void setUseCursorName(Boolean  newValue) {
        useCursorName = newValue;
    }


    /**
     * Returns UseCursorName parameter.
     * 
     * @return newValue New UseCursorName value.
     */
    public boolean getDisableFetchSizeWithMaxRows() {
        return disableFetchSizeWithMaxRows;
    }

    /**
     * Sets DisableFetchSizeWithMaxRows parameter.
     * 
     * @param newValue New UseCursorName value.
     */
    public void setDisableFetchSizeWithMaxRows(boolean newValue) {
        disableFetchSizeWithMaxRows = newValue;
    }


    /**
     * Returns ResultSetType parameter.
     * 
     * @return newValue New ResultSetType value.
     */
    public int getResultSetType() {
        return resultSetType;
    }

    /**
     * Sets ResultSetType parameter.
     * 
     * @param newValue New ResultSetType value.
     */
    public void setResultSetType(int newValue) {
        resultSetType = newValue;
    }

    

    /**
     * Returns ResultSetConcurrency parameter.
     * 
     * @return newValue New ResultSetConcurrency value.
     */
    public int getResultSetConcurrency() {
        return resultSetConcurrency;
    }

    /**
     * Sets ResultSetConcurrency parameter.
     * 
     * @param newValue New ResultSetConcurrency value.
     */
    public void setResultSetConcurrency(int newValue) {
        resultSetConcurrency = newValue;
    }
    
    
    /**
     * @return
     */
    public String  getDBTransactionFactoryName() {
        return DBTransactionFactoryName;
    }


    /**
     * @param string
     */
    public void setDBTransactionFactoryName(String  string) {
        DBTransactionFactoryName = string;
    }


    /**
     * @return
     */
    public String  getConnectionAllocatorName() {
        return connectionAllocatorName;
    }


    /**
     * @param string
     */
    public void setConnectionAllocatorName(String  string) {
        connectionAllocatorName = string;
    }


    public int getFullCacheCountLimit() {
        return fullCacheCountLimit;
    }
    public void setFullCacheCountLimit(int i) {
        fullCacheCountLimit = i;
    }


    
    /**
     * Reads database configuration parameters.
     *
     */
    public void readDatabaseConfiguration() {
        Config databaseConfig = null;

        try {
            databaseConfig = (Config) DODS.getDatabaseManager().getConfig().getSection("DB."
                    + dbName);
        } catch (Exception  ex) {}
        if (databaseConfig != null) {
            try {
                readOnly = databaseConfig.getBoolean(PARAMNAME_READ_ONLY);
            } catch (Exception  e) {}
            try {
                lazyLoading = databaseConfig.getBoolean(PARAMNAME_LAZY_LOADING);
            } catch (Exception  e) {}
            try {
                caseSensitive = databaseConfig.getBoolean(PARAMNAME_CASE_SENSITIVE);
            } catch (Exception  e) {}
            try {
                maxExecuteTime = databaseConfig.getInt(PARAMNAME_MAX_EXECUTE_TIME);
            } catch (Exception  e) {}
            
            try {
                transactionCheck = databaseConfig.getBoolean(PARAMNAME_TRANSACTION_CHECK);
            } catch (Exception  e) {}
            try {
                deleteCheckVersion = databaseConfig.getBoolean(PARAMNAME_DELETE_CHECK_VERSION);
            } catch (Exception  e) {}
            try {
                autoSave = databaseConfig.getBoolean(PARAMNAME_AUTO_SAVE);
            } catch (Exception  e) {}
            try {
                autoSaveCreateVirgin = databaseConfig.getBoolean(PARAMNAME_AUTO_SAVE_CREATE_VIRGIN);
            } catch (Exception  e) {}
            try {
                autoWrite = databaseConfig.getBoolean(PARAMNAME_AUTO_WRITE);
            } catch (Exception  e) {}
            try {
                transactionCaches = databaseConfig.getBoolean(PARAMNAME_TRANSACTION_CACHES);
            } catch (Exception  e) {}
            try {
                deadlockWaitTime = databaseConfig.getInt(PARAMNAME_DEADLOCK_READ_TIME);
            } catch (Exception  e) {}
            try {
                deadlockRetryCount = databaseConfig.getInt(PARAMNAME_DEADLOCK_RETRY_NUMBER);
            } catch (Exception  e) {}
            try {
                defaultFetchSize = databaseConfig.getInt(PARAMNAME_DEFAULT_FETCH_SIZE);
            } catch (Exception  e) {}
            try {
                defaultQueryTimeout = databaseConfig.getInt(PARAMNAME_QUERY_TIMEOUT);
            } catch (Exception  e) {}
            try {
                defaultSelectOids = databaseConfig.getBoolean(PARAMNAME_SELECT_OIDS);
            } catch (Exception  e) {}
            try {
                defaultIncrementVersions = databaseConfig.getBoolean(PARAMNAME_INCREMENT_VERSIONS);
            } catch (Exception  e) {}
            try {
                boolean tmpUCrN = (databaseConfig.getBoolean(PARAMNAME_USE_CURSOR_NAME));
                useCursorName = new Boolean (tmpUCrN);
            } catch (Exception  e) {
                useCursorName = null;
            }
            try {
                DBTransactionFactoryName = (databaseConfig.getString(CommonConstants.TRANSACTION_FACTORY));
            } catch (Exception  e) {
                DBTransactionFactoryName = null;
            }
            try {
                connectionAllocatorName = (databaseConfig.getString(CommonConstants.CONNECTION_ALLOCATOR));
            } catch (Exception  e) {
                connectionAllocatorName = null;
            }
            try {
                DBConnectionFactoryName = (databaseConfig.getString(CommonConstants.CONNECTION_FACTORY));
            } catch (Exception  e) {
                DBConnectionFactoryName = null;
            }
            try {
                queryCacheImplClass = (databaseConfig.getString(CommonConstants.QUERY_CACAHE_IMPL_CLASS));
            } catch (Exception  e) {
                queryCacheImplClass = null;
            }           
            try {
                fullCacheCountLimit = (databaseConfig.getInt(CacheConstants.FULL_CACHE_COUNT_LIMIT));
            } catch (Exception  e) {}
            try {
                initCachesResultSetType = databaseConfig.getString(CommonConstants.INIT_CACHES_RESULT_SET_TYPE);
            } catch (Exception  e) {}    
            try {
                initCachesResultSetConcurrency = databaseConfig.getString(CommonConstants.INIT_CACHES_RESULT_SET_CONCURRENCY);
            } catch (Exception  e) {}
            try {
                sqlBatch = databaseConfig.getBoolean(CommonConstants.SQL_BATCH);
            } catch (Exception  e) {}
            try {
                caseInsensitiveDatabase = databaseConfig.getBoolean(CommonConstants.CASE_INSENSITIVE_DATABASE);
            } catch (Exception  e) {}
            try {
                xaDefaultTimeout = databaseConfig.getInt(CommonConstants.XA_DEFAULT_TIMEOUT);
            } catch (Exception  e) { }

            try {
                queryTimeLimit = new Integer (databaseConfig.getInt(CommonConstants.QUERY_TIME_LIMIT));
            } catch (Exception  e) {}
            try {
                xaTransactonManagerLookupName = databaseConfig.getString(CommonConstants.XA_TM_LOOKUP_NAME);
            } catch (Exception  e) { }
            try {
                xaUsageCase = databaseConfig.getInt(CommonConstants.XA_USAGE_CASE);
            } catch (Exception  e) { }
            try {
                xaWrappedTransImplFactory = databaseConfig.getString(CommonConstants.XA_WRAPPED_TRANS_IMPL_FACTORY);
            } catch (Exception  e) { }
            try {
                xaUserTransactonLookupName = databaseConfig.getString(CommonConstants.XA_USER_TRANSACTION_LOOKUP_NAME);
            } catch (Exception  e) { }
            try {
                xaJtaSupport = databaseConfig.getString(CommonConstants.XA_JTA_SUPPORT);
            } catch (Exception  e) { }

        }   
        try {
            databaseConfig = (Config) DODS.getDatabaseManager().getConfig().getSection("DB."
                    + dbName + ".cache");
        } catch (Exception  ex) {}
        if (databaseConfig != null) {   
            try {
                maxCacheSize = databaseConfig.getInt(PARAMNAME_MAX_CACHE_SIZE);
            } catch (Exception  e) {}
            try {
                reserveFactor = databaseConfig.getDouble(PARAMNAME_RESERVE_FACTOR);
            } catch (Exception  e) {}
            try {
                cachePercentage = databaseConfig.getDouble(PARAMNAME_CACHE_PERCENTAGE);
            } catch (Exception  e) {}
            try {
                maxSimpleCacheSize = databaseConfig.getInt(PARAMNAME_MAX_SIMPLE_CACHE_SIZE);
            } catch (Exception  e) {}
            try {
                maxComplexCacheSize = databaseConfig.getInt(PARAMNAME_MAX_COMPLEX_CACHE_SIZE);
            } catch (Exception  e) {}
            try {
                maxMultiJoinCacheSize = databaseConfig.getInt(PARAMNAME_MAX_MULTI_JOIN_CACHE_SIZE);
            } catch (Exception  e) {}
            try {
                initialAllCaches = databaseConfig.getBoolean(PARAMNAME_INITIAL_ALL_CACHES);
            } catch (Exception  e) {}    
            try {
                initialCacheFetchSize = databaseConfig.getInt(PARAMNAME_INITIAL_CACHE_FETCH_SIZE);
            } catch (Exception  e) {}            
            try {
                initialDSCacheSize = databaseConfig.getInt(PARAMNAME_INITIAL_DS_CACHE_SIZE);
            } catch (Exception  e) {}    
            try {
                dodsCacheFactory = databaseConfig.getString(PARAMNAME_DODS_CACHE_FACTORY);
            } catch (Exception  e) {}    
            
        }
    }
    
    
    /**
     * Returns CaseSensitive parameter.
     * 
     * @return value of CaseSensitive parameter.
     */
    public boolean getCaseSensitive() {
        return caseSensitive;
    }

    /**
     * Set CaseSensitive parameter to newValue value.
     * 
     * @param newValue New value for CaseSensitive parameter
     */
    public void setCaseSensitive(boolean newValue) {
        caseSensitive = newValue;
    }

    /**
     * @return
     */
    public String  getDBConnectionFactoryName() {
        return DBConnectionFactoryName;
    }

    /**
     * @param string
     */
    public void setDBConnectionFactoryName(String  string) {
        DBConnectionFactoryName = string;
    }

    /**
     * @return
     */
    public int getInitialCacheFetchSize() {
        return initialCacheFetchSize;
    }

    /**
     * @return
     */
    public int getInitialDSCacheSize() {
        return initialDSCacheSize;
    }

    /**
     * @param i
     */
    public void setInitialCacheFetchSize(int i) {
        initialCacheFetchSize = i;
    }

    /**
     * @param i
     */
    public void setInitialDSCacheSize(int i) {
        initialDSCacheSize = i;
    }

    /**
     * @return
     */
    public String  getQueryCacheImplClass() {
        return queryCacheImplClass;
    }

    /**
     * @param string
     */
    public void setQueryCacheImplClass(String  string) {
        queryCacheImplClass = string;
    }

    /**
     * @return
     */
    public String  getInitCachesResultSetConcurrency() {
        return initCachesResultSetConcurrency;
    }

    /**
     * @return
     */
    public String  getInitCachesResultSetType() {
        return initCachesResultSetType;
    }

    /**
     * @param string
     */
    public void setInitCachesResultSetConcurrency(String  string) {
        initCachesResultSetConcurrency = string;
    }

    /**
     * @param string
     */
    public void setInitCachesResultSetType(String  string) {
        initCachesResultSetType = string;
    }
    

    /**
     * @return
     */
    public boolean isSqlBatch() {
        return sqlBatch;
    }

    
    /**
     * @param b
     */
    public void setSqlBatch(boolean b) {
        sqlBatch = b;
    }

    
    /**
     * @return Returns the caseInsensitiveDatabase.
     */
    public boolean isCaseInsensitiveDatabase() {
        return caseInsensitiveDatabase;
    }
    
    
    /**
     * @param caseInsensitiveDatabase The caseInsensitiveDatabase to set.
     */
    public void setCaseInsensitiveDatabase(boolean caseInsensitiveDatabase) {
        this.caseInsensitiveDatabase = caseInsensitiveDatabase;
    }

    
    /**
     * @return Returns the queryTimeLimit.
     */
    public Integer  getQueryTimeLimit() {
        return queryTimeLimit;
    }
    
    
    /**
     * @param queryTimeLimit The queryTimeLimit to set.
     */
    public void setQueryTimeLimit(Integer  queryTimeLimit) {
        this.queryTimeLimit = queryTimeLimit;
    }
    
    
    /**
     * @return Returns the xaDefaultTimeout.
     */
    public int getXaDefaultTimeout() {
        return xaDefaultTimeout;
    }
    
    
    /**
     * @param xaDefaultTimeout The xaDefaultTimeout to set.
     */
    public void setXaDefaultTimeout(int xaDefaultTimeout) {
        this.xaDefaultTimeout = xaDefaultTimeout;
    }
    
    
    /**
     * @return Returns the xaTransactonManagerLookupName.
     */
    public String  getXaTransactonManagerLookupName() {
        return xaTransactonManagerLookupName;
    }
    
    
    /**
     * @param xaTransactonManagerLookupName The xaTransactonManagerLookupName to set.
     */
    public void setXaTransactonManagerLookupName(
            String  xaTransactonManagerLookupName) {
        this.xaTransactonManagerLookupName = xaTransactonManagerLookupName;
    }
    
    
    /**
     * @return Returns the xaUsageCase.
     */
    public int getXaUsageCase() {
        return xaUsageCase;
    }
    
    
    /**
     * @param xaUsageCase The xaUsageCase to set.
     */
    public void setXaUsageCase(int xaUsageCase) {
        this.xaUsageCase = xaUsageCase;
    }
    
    
    /**
     * @return Returns the xaWrappedTransImplFactory.
     */
    public String  getXaWrappedTransImplFactory() {
        return xaWrappedTransImplFactory;
    }
    
    
    /**
     * @param xaWrappedTransImplFactory The xaWrappedTransImplFactory to set.
     */
    public void setXaWrappedTransImplFactory(String  xaWrappedTransImplFactory) {
        this.xaWrappedTransImplFactory = xaWrappedTransImplFactory;
    }
    
    
    /**
     * @return Returns the xaUserTransactonLookupName.
     */
    public String  getXaUserTransactonLookupName() {
        return xaUserTransactonLookupName;
    }
    
    
    /**
     * @param xaUserTransactonLookupName The xaUserTransactonLookupName to set.
     */
    public void setXaUserTransactonLookupName(String  xaUserTransactonLookupName) {
        this.xaUserTransactonLookupName = xaUserTransactonLookupName;
    }
    
    
    /**
     * @return Returns the xaJtaSupport.
     */
    public String  getXaJtaSupport() {
        return xaJtaSupport;
    }
    
    
    /**
     * @param xaJtaSupport The xaJtaSupport to set.
     */
    public void setXaJtaSupport(String  xaJtaSupport) {
        this.xaJtaSupport = xaJtaSupport;
    }
    
    /**
     * @return Returns the dods_cache_factory.
     */
    public String  getDodsCacheFactory() {
        return this.dodsCacheFactory;
    }
    /**
     * @param dods_cache_factory The dods_cache_factory to set.
     */
    public void setDodsCacheFactory(String  dodsCacheFactory) {
        this.dodsCacheFactory = dodsCacheFactory;
    }
}
