
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache;

import java.util.ArrayList ;
import java.util.Collection ;
import java.util.HashMap ;
import org.enhydra.dods.cache.Condition;
import org.enhydra.dods.cache.QueryCacheItem;
import com.lutris.appserver.server.sql.DatabaseManagerException;
import com.lutris.dods.builder.generator.dataobject.GenericDO;
import com.lutris.appserver.server.sql.CoreDataStruct;

/**
 * This class stores one query and its necessary data, for query array.
 *
 * @author    Tanja Jovanovic
 * @author    Nenad Vico
 * @version   2.0  15.06.2003.
 */
public class QueryCacheItemImpl implements QueryCacheItem {

    /**
     * Query id: String "query_database_name.String_presentation_of_query".
     */
    protected String  queryId;

    /**
     * HashMap of data (and DataStruct) object IDs which are results of the
     * query.
     */
    protected HashMap  OIds;

    /**
     * Number of cached query results.
     */
    protected int resultNum = 0;

    /**
     * True if all query results are cached, otherwise false.
     */
    protected boolean completeRes = false;

    /**
     * Head of object ID list in which are query results.
     */
    protected ListItem head;

    /**
     * Tail of object ID list in which are query results.
     */
    protected ListItem tail;
    
    /**
     * This attribute indicates whether there have been performed inserts, 
     * updates or deletes on results of this query.
     *
     */
    protected boolean modifiedQuery = false;

    /**
     * Time needed for query execution.
     */
    protected int time;

    /**
     * Array of conditions declared in WHERE part of the query
     * (array of org.enhydra.dods.cache.Condition objects).
     */
    protected ArrayList  conds;

    /**
     * Database of the query.
     */
    protected String  originDatabase;

    /**
     * Constructor (String).
     *
     * @param origDb Query database.
     */
    public QueryCacheItemImpl(String  origDb) {
        OIds = new HashMap ();
        resultNum = 0;
        completeRes = false;
        modifiedQuery = false;
        this.time = 0;
        conds = new ArrayList ();
        originDatabase = origDb;
    }

    /**
     * Constructor (String, LinkedHashSet, int, ArrayList, String).
     *
     * @param qId Query id.
     * @param OIds Array of object IDs which are results of the Query.
     * @param time Query execution time.
     * @param conditions Conditions (WHERE part of the query).
     * @param origDb Query database.
     */
    public QueryCacheItemImpl(String  qId, HashMap  OIds, int time, ArrayList  conditions, String  origDb) {
        queryId = qId;
        this.OIds = OIds;
        resultNum = 0;
        completeRes = false;
        modifiedQuery = false;
        this.time = time;
        conds = conditions;
        originDatabase = origDb;
    }

    /**
     * Returns query id (String it the form:
     * query_database_name.String_presentation_of_query).
     *
     * @return Query id.
     */
    public String  getQueryId() {
        return queryId;
    }

    /**
     * Sets query id (String it the form:
     * query_database_name.String_presentation_of_query).
     *
     * @param queryId Query id.
     */
    public void setQueryId(String  queryId) {
        this.queryId = queryId;
    }

    /**
     * Returns OIds (Collection of object IDs which are results of the query).
     *
     * @return Collection of object IDs which are results of the query.
     */
    public Collection  getOIds() {
        ArrayList  list = new ArrayList ();
        ListItem iter = head;

        while (iter != null) {
            list.add(iter.handle);
            iter = iter.next;
        }
        return list;
    }

    /**
     * Returns number of cached query results.
     *
     * @return Number of cached query results.
     */
    public int getResultNum() {
        return resultNum;
    }

    /**
     * Returns true if all query results are cached, otherwise false.
     *
     * @return true if all query results are cached, otherwise false.
     */
    public boolean isCompleteResult() {
        return completeRes;
    }

    /**
     * Sets new boolean value about the cached query results (true if all query
     * results are cached, otherwise false).
     *
     * @param newCompleteRes true if all query results are cached, otherwise
     * false.
     */
    public void setCompleteResult(boolean newCompleteRes) {
        completeRes = newCompleteRes;
    }
  
    /**
     * Returns true if there have been performed inserts, updates or deletes  
     * concerning results of this query, otherwise false.
     *
     * @return true if there have been performed inserts, updates or deletes  
     * concerning results of this query, otherwise false.
     */
    public boolean isModifiedQuery() {
        return modifiedQuery;  
    }

    /**
     * Sets modifiedQuery attribute.
     *
     * @param mod New value of attribute modifiedQuery.
     * concerning results of this query, otherwise false.
     */
    public void setModifiedQuery(boolean mod) {
        modifiedQuery = mod;
    }

    /**
     * Returns time needed for query execution.
     *
     * @return Time needed for query execution.
     */
    public int getTime() {
        return time;
    }

    /**
     * Sets time needed for query execution.
     *
     * @param time Time needed for query execution.
     */
    public void setTime(int time) {
        this.time = time;
    }

    /**
     * Returns array of query conditions conds.
     *
     * @return Array of query conditions.
     */
    public ArrayList  getConds() {
        return conds;
    }

    /**
     * Sets array of query conditions.
     *
     * @param conds Array of query conditions.
     */
    public void setConds(ArrayList  conds) {
        this.conds = conds;
    }

    /**
     * Adds condition to query.
     *
     * @param cond <code>Condition</code> that will be added to query.
     */
    public void addCond(Condition cond) {
        conds.add(cond);
    }

    /**
     * Returns query database.
     *
     * @deprecated Use get_OriginDatabase()
     * @return Query database.
     */
    public String  getOriginDatabase() {
        return get_OriginDatabase();
    }

    /**
     * Returns query database.
     *
     * @return Query database.
     */
    public String  get_OriginDatabase() {
        return originDatabase;
    }

    /**
     * Checks whether data object obj satisfies conditions of this query.
     *
     * @param obj Data object for which are checked conditions of this query.
     * @return true if data object obj satisfies conditions of this query,
     * otherwise false.
     */
    public boolean checkConditions(GenericDO obj) {
        for (int i = 0; i < conds.size(); i++) {
            if (!obj.compareCond((Condition) conds.get(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * Checks whether DataStruct object obj satisfies conditions of this query.
     *
     * @param obj DataStruct object for which are checked conditions of this
     * query.
     * @return true if DataStruct object obj satisfies conditions of this query,
     * otherwise false.
     */
    public boolean checkConditions(CoreDataStruct obj) {
        for (int i = 0; i < conds.size(); i++) {
            if (!obj.compareCond((Condition) conds.get(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * Inserts data object obj (or updates it if already exists) in array DOs,
     * if it satisfies this query.
     *
     * @param obj Data object which may be inserted (or updated) in array DOs.
     */
    public void update(GenericDO obj) {
        if (obj.get_OriginDatabase().equals(originDatabase)) {
            try {
                String  key = obj.get_Handle();

                if (checkConditions(obj)) {
                    ListItem tmp = null;

                    tmp = (ListItem) OIds.get(key);
                    if (tmp == null) {
                        addHandle(key);
                    }
                } else {
                    removeHandle(key);
                }
            } catch (DatabaseManagerException e) {}
        }
    }

    /**
     * Inserts DataStruct object obj (or updates it if already exists) in array
     * DOs, if it satisfies this query.
     *
     * @param obj DataStruct object which may be inserted (or updated) in array
     * DOs.
     */
    public void update(CoreDataStruct obj) {
        if (obj.get_Database().equals(originDatabase)) {
            try {
                String  key = obj.get_Handle();

                if (checkConditions(obj)) {
                    ListItem tmp = null;

                    tmp = (ListItem) OIds.get(key);
                    if (tmp == null) {
                        addHandle(key);
                    }
                } else {
                    removeHandle(key);
                }
            } catch (DatabaseManagerException e) {}
        }
    }

    /**
     * Removes data object obj from array DOs, if present.
     *
     * @param obj Data object which will be removed from array DOs.
     */
    public void delete(GenericDO obj) {
        if (obj.get_OriginDatabase().equals(originDatabase)) {
            try {
                String  key = null;

                key = obj.get_Handle();
                if (key != null) {
                    removeHandle(key);
                }
            } catch (DatabaseManagerException e) {}
        }
    }

    /**
     * Removes DataStruct object obj from array DOs, if present.
     *
     * @param obj DataStruct object which will be removed from array DOs.
     */
    public void delete(CoreDataStruct obj) {
        if (obj.get_Database().equals(originDatabase)) {
            try {
                String  key = null;

                key = obj.get_Handle();
                if (key != null) {
                    removeHandle(key);
                }
            } catch (DatabaseManagerException e) {}
        }
    }

    /**
     * Adds data object obj to array DOs.
     *
     * @param obj Data object which will be added to array DOs.
     */
    public void add(GenericDO obj) {
        if (obj.get_OriginDatabase().equals(originDatabase)) {
            try {
                String  key = null;

                key = obj.get_Handle();
                if (key != null) {
                    addHandle(key);
                }
            } catch (DatabaseManagerException e) {}
        }
    }

    /**
     * Adds DataStruct object obj to array DOs.
     *
     * @param obj DataStruct object which will be added to array DOs.
     */
    public void add(CoreDataStruct obj) {
        if (obj != null && obj.get_Database() != null) {    
            if (obj.get_Database().equals(originDatabase)) {
                try {
           
                    String  key = null;

                    key = obj.get_Handle();
                    if (key != null) {
                        addHandle(key);
                    }
                } catch (DatabaseManagerException e) {}
            }
        }  
    }

    /**
     * Shows content of this class.
     * Can be used for debugging.
     */
    public String  toString() {
        StringBuffer  ret = new StringBuffer ();
        GenericDO DO = null;

        ret.append("\n QueryCacheItemImpl: ");
        ret.append("\n queryId: " + queryId);
        ret.append("\n OIds : " + OIds);
        if (OIds != null) {
            for (ListItem iter = head; iter != null; iter = iter.next) {
                ret.append(" " + iter.handle);
            }
        }
        ret.append("\n time : " + time);
        ret.append("\n conds : " + conds);
        ret.append("\n originDatabase : " + originDatabase);
        return ret.toString();
    }

    protected void addHandle(String  handle) {
        ListItem item = new ListItem(handle);

        item.prev = tail;
        if (tail != null) {
            tail.next = item;
        }
        if (head == null) {
            head = item;
        }
        tail = item;
        ListItem value = (ListItem) OIds.get(handle);

        if (value == null) {
            OIds.put(handle, item);
        } else {
            item.sameNext = value;
            OIds.put(handle, item);
        }
        resultNum++;
    }

    protected void removeHandle(String  handle) {
        ListItem curr = (ListItem) OIds.get(handle);

        if (curr != null) {
            if (head == curr) {
                head = curr.next;
            }
            if (tail == curr) {
                tail = curr.prev;
            }
        }
        ListItem temp;

        while (curr != null) {
            temp = curr;
            curr = curr.unlink().sameNext;
            temp.sameNext = null;
            resultNum--;
        }
    }
    static class ListItem {
        public String  handle;
        public ListItem prev = null;
        public ListItem next = null;
        public ListItem sameNext = null;
        ListItem() {}

        ListItem(String  hnd) {
            handle = hnd;
        }

        ListItem unlink() {
            if (prev != null) {
                prev.next = next;
            }
            if (next != null) {
                next.prev = prev;
            }
            return this;
        }
    }
}
