
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: QueryBuilder.java,v 1.10 2005/05/26 08:08:10 predrag Exp $
 *
 * contains WebDocWf extension for extended wildcard support
 *
 * conatins WebDocWf extension for row counters
 *
 */
package com.lutris.dods.builder.generator.query;

import java.math.BigDecimal ;
import java.sql.PreparedStatement ;
import java.sql.ResultSet ;
import java.sql.SQLException ;
import java.util.BitSet ;
import java.util.Date ;
import java.util.Enumeration ;
import java.util.Hashtable ;
import java.util.Vector ;

import org.enhydra.dods.Common;
import org.enhydra.dods.CommonConstants;
import org.enhydra.dods.DODS;
import org.enhydra.dods.cache.CacheConstants;

import com.lutris.appserver.server.sql.CoreDataStruct;
import com.lutris.appserver.server.sql.DBConnection;
import com.lutris.appserver.server.sql.DBQuery;
import com.lutris.appserver.server.sql.DatabaseManagerException;
import com.lutris.appserver.server.sql.ExtendedDBConnection;
import com.lutris.appserver.server.sql.ExtendedQuery;
import com.lutris.appserver.server.sql.ObjectId;
import com.lutris.appserver.server.sql.ObjectIdException;
import com.lutris.appserver.server.sql.StandardDatabaseManager;
import com.lutris.appserver.server.sql.standard.DriverSpecificConstants;
import com.lutris.appserver.server.sql.standard.StandardLogicalDatabase;
import com.lutris.dods.builder.generator.dataobject.GenericDO;
import com.lutris.logging.Logger;

/**
 * @author Jay Gunter
 * <PRE>
 *      A QueryBuilder is a helper-object used by the xxxQuery classes.
 *
 *      Stand-alone usage:
 *
 *              QueryBuilder qb;
 *              qb = new QueryBuilder( "cats" );
 *                      or
 *              qb = new QueryBuilder( "cats", "name, age" );
 *                      ...
 *              qb.addWhereClause( "owner.name", "Smith",
 *                                      QueryBuilder.NOT_NULL,
 *                                      QueryBuilder.EXACT_MATCH );
 *              qb.addWhereClause( "paws", 3, QueryBuilder.GREATER_THAN );
 *              qb.addTwoColumnWhereClause(
 *                      "cat.ownerId", "owner.Id", QueryBuilder.EQUAL );
 *              ResultSet rs = qb.executeQuery( conn );
 *                      ...
 *              qb.reset();
 *
 * </PRE>
 *
 *
 * @version $Revision: 1.10 $
 */
public class QueryBuilder implements ExtendedQuery {
    // Special values for whereClauses:
 private static final String  OR = " OR ";
    private static final String  OPEN_PAREN = " ( ";
    private static final String  CLOSE_PAREN = " ) ";
    // Comparison operator passed as 3rd arg to addWhereClause( int|float|double )
 public static final String   EQUAL = "=";
    public static String   NOT_EQUAL = "!=";
    public static final String   LESS_THAN = "<";
    public static final String   LESS_THAN_OR_EQUAL = "<=";
    public static final String   GREATER_THAN = ">";
    public static final String   GREATER_THAN_OR_EQUAL = ">=";
    public static final String   IS_NULL = " IS NULL ";
    public static final String   IS_NOT_NULL = " IS NOT NULL ";
    public static final String   OPEN_IN = " IN ( ";
    public static final String   OPEN_NOT_IN = " NOT IN ( ";
    public static final String   OPEN_EXISTS = " EXISTS ( ";
    public static final String   OPEN_NOT_EXISTS = " NOT EXISTS ( ";
    public static final String   CLOSE_IN = " ) ";
    public static final String   CASE_SENSITIVE_CONTAINS = "%X%";
    public static final String   CASE_INSENSITIVE_CONTAINS = "%x%";
    public static final String   CASE_SENSITIVE_STARTS_WITH = "X%";
    public static final String   CASE_INSENSITIVE_STARTS_WITH = "x%";
    public static final String   CASE_SENSITIVE_ENDS_WITH = "%X";
    public static final String   CASE_INSENSITIVE_ENDS_WITH = "%x";
    public static final String   CASE_INSENSITIVE_EQUAL = "%xxxx";
    // WebDocWf extension for extended wildcard support
 // The following lines have been added:
 public static final String   CASE_SENSITIVE_MATCH = "X";
    public static final String   CASE_INSENSITIVE_MATCH = "x";
    public static final String   USER_CASE_SENSITIVE_MATCH = "U";
    public static final String   USER_CASE_INSENSITIVE_MATCH = "u";
    // end ofWebDocWf extension for extended wildcard support
 /**
     * Null-acceptability indicator passed as 3rd arg to addWhereClause( String )
     * @deprecated Use addWhere() methods instead of addWhereClause() methods.
     */
    public static final boolean NULL_OK = true;

    /**
     * Null-acceptability indicator passed as 3rd arg to addWhereClause( String )
     * @deprecated Use addWhere() methods instead of addWhereClause() methods.
     */
    public static final boolean NOT_NULL = false;

    /**
     * Wild-card          indicator passed as 4th arg to addWhereClause( String )
     * @deprecated Use addWhere() methods instead of addWhereClause() methods.
     */
    public static final boolean EXACT_MATCH = true;

    /**
     * Wild-card          indicator passed as 4th arg to addWhereClause( String )
     * @deprecated Use addWhere() methods instead of addWhereClause() methods.
     */
    public static final boolean NOT_EXACT = false;
    // Default inexact string match keyword (e.g. can be MATCHES or LIKE)
 public static final String  DEFAULT_MATCHES_KEYWORD = "MATCHES";
    private String  likeKeyword = DEFAULT_MATCHES_KEYWORD;
    // Default inexact string wildcard (e.g. can be * or %)
 public static final String  DEFAULT_WILDCARD = "*";
    private String  wildcard = DEFAULT_WILDCARD;
    // WebDocWf extension for extended wildcard support
 // The following lines have been added:
 // Do NOT use "\" or "\\" or "\\\\" ... as wildcard/escape characters
 // this would lead to runtime errrors because of multiple string parsing in the DODS code
 public static final String  DEFAULT_SINGLE_WILDCARD = "_";
    public static final String  DEFAULT_SINGLE_WILDCARD_ESCAPE = "?";
    public static final String  DEFAULT_WILDCARD_ESCAPE = "?";
    public static final String  DEFAULT_WILDCARD_ESCAPE_CLAUSE = "ESCAPE '?'";
    public static final String  DEFAULT_USER_WILDCARD = "*";
    public static final String  DEFAULT_USER_SINGLE_WILDCARD = "?";
    public static final String  DEFAULT_USER_SINGLE_WILDCARD_ESCAPE = "?";
    public static final String  DEFAULT_USER_WILDCARD_ESCAPE = "?";
    public static final boolean DEFAULT_USER_APPEND_WILDCARD = false;
    public static final boolean DEFAULT_USER_TRIM_STRING = false;
    public String  userConfigWildcard = DEFAULT_USER_WILDCARD;
    public String  userConfigSingleWildcard = DEFAULT_USER_SINGLE_WILDCARD;
    public String  userConfigSingleWildcardEscape = DEFAULT_USER_SINGLE_WILDCARD_ESCAPE;
    public String  userConfigWildcardEscape = DEFAULT_USER_WILDCARD_ESCAPE;
    // end of first part WebDocWf extension for extended wildcard support
 // See debug() method.
 static private boolean debugAllSQL = false;
    private boolean debugSQL = false;
    // true if it is used at least one method that provides possibility of join between tables
 private boolean multiTableJoin = false;
    private boolean unionTableJoin = false;
    // true if select <table>.primaryKey can be performed
 private boolean preventPrimaryKeySelect = false;
    // WebDocWf extension cursor-handling
 // Value -1 means not defined
 private int iCurrentFetchSize = -1;
    private static int iDefaultFetchSize = -1;
    private int iCurrentQueryTimeout = CacheConstants.DEFAULT_QUERY_TIMEOUT;
    private static int iDefaultQueryTimeout = CacheConstants.DEFAULT_QUERY_TIMEOUT;
    // WebDocWf extension cursor-handling end
 private String  databaseName = null;

    private int DEFAULT_RS_TYPE = -100;
    private int DEFAULT_RS_CONCURRENCY = -100;
    private int iResultSetConcurrency = DEFAULT_RS_CONCURRENCY;
    private int iResultSetType = DEFAULT_RS_TYPE;
    
    private String  customNotEqualSqlOperator = null;
    
    private boolean caseInsensitiveDatabase = CommonConstants.DEFAULT_CASE_INSENSITIVE_DATABASE;
    
    /*
     The DODS (Data Object Design Studio) development tool
     generates data-layer source code which uses QueryBuilder objects
     to perform database queries.
     This is the original (1st, old) purpose of QueryBuilder.
     QueryBuilder can also be used by application developers
     to perform straight SQL queries.  Certain static final data members
     of the DO classes generated by DODS are used to specify the tables and columns
     for the query.  This allows developers to write straight SQL queries
     but still enjoy the compile-time checking.
     This is the 2nd (new) purpose of QueryBuilder.
     ================= Usage 1: DODS generated code =================================
     NOTE!  Application developers, please see "Usage 2" section below.
     1st, old, String-based style (used by DODS-generated Query classes):
     Query creation:
     The ctor takes a String tablename
     (and optional String list of comma-separated column names).
     E.g.:
     // In PersonQuery class:
     QueryBuilder qb = new QueryBuilder( "person", "person.LastName" );
     addWhereClause() methods:
     A where-clause is added by calling one of the addWhereClause() methods
     which takes a String column name a String value for comparison.
     Because table and column names are Strings, the application code
     can become out-of-sync with a changing database schema,
     and the developer gets no warning from the compiler.
     This scheme is ok for use by DODS since the application code
     (DO, Query classes) are (re)generated every time the
     schema (the DODS project file) is changed.
     However, this scheme is non-ideal for manual coding.
     E.g.:
     // In PersonQuery class:
     qb.addWhereClause( "FirstName", "Bob", QueryBuilder.EQUAL );
     Obtaining results:
     The results of the query are returned as a JDBC ResultSet.
     Column values are retrieved by their postion within each returned row,
     or by the String name of the column in the returned row.
     Again, no hope for compile-time checking.
     E.g.:
     // PersonQuery class creates a DBQuery object to run the query:
     DBQuery dbQuery = DODS.getDatabaseManager().createQuery();
     dbQuery.query( this );  // invokes PersonQuery.executeQuery();
     // PersonQuery class extends Query and implements executeQuery():
     ResultSet rs = qb.executeQuery( connection );
     while ( rs.next() ) {
     String lastname = rs.getString( 1 );
     }
     ================= Usage 2: Application Developers =============================
     2nd, new, Object Oriented style (used by application developers):
     Query creation:
     The ctor takes a Vector of RDBColumn objects.
     Each RDBColumn object knows which RDBTable it belongs to.
     In each DO class generated by DODS, there is a static RDBColumn object
     for each column in database table for the DO.
     E.g.:
     Vector fields = new Vector();
     fields.addElement( PersonDO.LastName );
     QueryBuilder qb = new QueryBuilder( fields );
     Or:
     QueryBuilder qb = new QueryBuilder();
     qb.select( PersonDO.LastName );
     addWhere() methods:
     A where-clause is added by passing an RDBColumn object and a value.
     Because objects (not Strings) are used to specify columns in the query,
     errors in hand-written application code can be caught by the compiler.
     E.g.:
     qb.addWhere( PersonDO.FirstName, "Bob" );
     Obtaining results:
     The results of the query are returned as a series of RDBRow objects.
     Each RDBRow contains an RDBColumnValue object for each returned column.
     An RDBColumnValue object is retrieved from the RDBRow using
     a one of the RDBColumn objects passed to the QueryBuilder ctor.
     The value is extracted from an RDBColumnValue object via its
     getString or getInteger (...) method.
     These methods throw an exception if the application attempts to
     extract an incorrect type for the RDBColumnValue.
     E.g.:
     RDBRow row;
     while ( null != ( row = qb.getNextRow() ) ) {
     String lastname = row.get( PersonDO.LastName ).getString();
     }
     qb.close();  // always release the resources
     Note:
     The templates used by the DODS sourceGenerator_Query class
     could be converted to use this 2nd style.
     */

    /**
     * Construct a QueryBuilder object that will return specified fields of a table.
     * Style 2 (see above.)
     *
     * @param fields Vector of RDBColumn objects,
     * one for each column to be returned.  Fields can refer to different tables.
     */
    public QueryBuilder(Vector  fields) {
        init();
        reset();
        if (null != fields) {
            selectedFields = (Vector ) fields.clone();
        }
    }

    private void init(){
        try {
            String  value;

            value=((StandardDatabaseManager)DODS.getDatabaseManager())
                .getDatabaseManagerConfiguration().getUserConfigWildcard();
            if (value!= null) {
                userConfigWildcard = value;
            }

            value=((StandardDatabaseManager)DODS.getDatabaseManager())
                .getDatabaseManagerConfiguration().getUserConfigSingleWildcard();
            if (value!= null) {
                userConfigSingleWildcard = value;
            }

            value=((StandardDatabaseManager)DODS.getDatabaseManager())
                .getDatabaseManagerConfiguration().getUserConfigSingleWildcardEscape();
            if (value!= null) {
                userConfigSingleWildcardEscape = value;
            }
        } catch (Exception  except) {}
        initDefaultWildCards();
        
        try {
            caseInsensitiveDatabase=((StandardLogicalDatabase)DODS.getDatabaseManager()
                     .findLogicalDatabase(getDatabaseName())).getDatabaseConfiguration().isCaseInsensitiveDatabase();
        } catch (Exception  e) {
            caseInsensitiveDatabase = CommonConstants.DEFAULT_CASE_INSENSITIVE_DATABASE;
        }
        
        customNotEqualSqlOperator = null;
        try {
            String  customNotEqualSqlOperatorStr = ((StandardLogicalDatabase)DODS.getDatabaseManager()
                     .findLogicalDatabase(getDatabaseName())).getDriverProperty(DriverSpecificConstants.PARAMNAME_CUSTOM_NOT_EQUAL_SQL_OPERATOR);
            if(customNotEqualSqlOperatorStr!=null) {
                    customNotEqualSqlOperator=customNotEqualSqlOperatorStr;
            }else {
                customNotEqualSqlOperator = DriverSpecificConstants.DEFAULT_CUSTOM_NOT_EQUAL_SQL_OPERATOR;
            }
        }catch(Exception  e) {}
        if(customNotEqualSqlOperator!=null) NOT_EQUAL=customNotEqualSqlOperator;
    }

    /**
     * Construct a QueryBuilder object that will return specified fields of a table.
     * Style 2 (see above.)
     *
     * @param fields array of RDBColumn objects,
     * one for each column to be returned.  Fields can refer to different tables.
     */
    public QueryBuilder(RDBColumn[] fields) {
        init();
        reset();
        for (int i = 0; i < fields.length; i++) {
            select(fields[i]);
        }
    }

    /**
     * Construct a QueryBuilder object.
     * When no fields are specified in the constructor,
     * you must use calls to select() to specify the RDBColumns
     * that will be returned.
     * This can be more convenient than constructing the Vector before
     * calling the QueryBuilder(Vector) constructor.
     * Style 2 (see above.)
     */
    public QueryBuilder() {
        init();
        reset();
    }

    /**
     * Add a field to be returned by this QueryBuilder.
     * Style 2 (see above.)
     *
     * @param field RDBColumn object to be returned.
     * Fields can refer to different tables.
     */
    public void select(RDBColumn field) {
        if (null == field) {
            return;
        }
        for (int i = 0; i < selectedFields.size(); i++) {
            if (field.equals((RDBColumn) selectedFields.elementAt(i))) {
                return;
            }
        }
        selectedFields.addElement(field);
    }

    /**
     * Returns an array of RDBRow objects.
     *
     * @return array of RDBRow objects.
     * @exception ATException If the data is not accessible
     * @exception BusinessRuleException If x is invalid.
     * @see #getNextRow
     * author Jay Gunter
     */
    public RDBRow[] getRows()
        throws QueryException {
        Vector  v = new Vector ();
        RDBRow row;

        while (null != (row = getNextRow())) {
            v.addElement(row);
        }
        RDBRow[] ret = new RDBRow[ v.size() ];

        v.copyInto(ret);
        return ret;
    }

    /**
     * Returns an RDBRow object containing the fields specified in the ctor.
     * On first call, executes the query.
     * NOTE:  to use this method, the QueryBuilder object must be created
     * using the
     *  QueryBuilder( RDBColumn[] )
     * constructor.
     *
     * @return row the RDBRow representing 1 row from the ResultSet.
     * @exception ATException If the data is not accessible
     * @exception BusinessRuleException If x is invalid.
     * author Jay Gunter
     */
    private DBQuery dbQuery = null;
    private ResultSet  rs = null;
    private Boolean    rsClosed = null;
    private boolean done = false;
    public RDBRow getNextRow() throws QueryException {
        if (done) {
            return null;
        }
        if (null == rs) {
            try {
                if (null == dbQuery) {
                    // May not be null if I later implement
 // a ctor that takes a DBQuery.
                    dbQuery = DODS.getDatabaseManager().createQuery();
                }
            } catch (Exception  e) {
                throw new QueryException("SQL=[" + sql
                                             + "]: Unable to create query",
                                         e);
            }
            try {
                // invokes executeQuery which sets ResultSet rs
                dbQuery.query(this);
            } catch (Exception  e) {
                throw new QueryException("SQL=[" + sql
                                             + "]: Unable to run query",
                                         e);
            }
            if (null == rs) {
                throw new QueryException("No ResultSet for Query.");
            }
        }
        try {
            if (!rs.next()) {
                // ResultSet.next() only returns false once.
 // After that, drivers typically throw "result set empty" exception.
 // That isn't very friendly or useful to an application programmer.
 // Setting done=true allows us to just keep returning nulls.
                done = true;
                rsClosed=new Boolean (true);
                rs.close();
                dbQuery.release();
                return null;
            }
        } catch (Exception  e) {
            throw new QueryException("SQL=[" + sql
                                         + "]: Unable to get query results",
                                     e);
        }
        RDBColumnValue[] vals = new RDBColumnValue[ selectedFields.size() ];
        RDBColumn field = null;

        for (int i = 0; i < selectedFields.size(); i++) {
            try {
                field = (RDBColumn) selectedFields.elementAt(i);
                vals[i] = new RDBColumnValue(field, rs.getObject(i + 1));
            } catch (Exception  e) {
                throw new QueryException("SQL=[" + sql
                                             + "]: Unable to get query result for column " + field,
                                         e);
            }
        }
        return new RDBRow(vals);
    }

    /**
     * WARNING:
     * This method is disabled and should never be called.
     * It's implementation is forced by the Query interface.
     * Use getNextRow instead().
     */
    public Object  next(ResultSet  rs)
        throws SQLException , ObjectIdException {
        return null;
    }

    // /////////////////////////////////////////////////////////////////////////////
 // / Constructors used by DODS-generated Query classes. ////////////////////////
 // /////////////////////////////////////////////////////////////////////////////
 /**
     * Construct a QueryBuilder object that will return all fields of a table.
     * Style 1 (see above.)
     * WARNING: QueryBuilder objects created with this constructor
     * cannot be used with the getNextRow() method.
     * @see #getNextRow
     *
     * @param tableName Name of the table on which to perform the SQL 'select'.
     */
    public QueryBuilder(String  tableName) {
        this(tableName, tableName + ".*");
    }

    /**
     * Construct a QueryBuilder object that will return specified fields of a table.
     * Style 1 (see above.)
     *
     * @param tableName     Name of the table on which to perform the SQL 'select'.
     * @param fieldList Comma-separated string of field names to retrieve.
     */
    public QueryBuilder(String  tableName, String  fieldList) {
        reset();
        init();
        /* mainTableName = tableName.toLowerCase(); */
        mainTableName = tableName;
        storeTableName(mainTableName);
        selectClause = fieldList;
    }

    /**
     * Determine table name of column.
     * For example, we return 'tableX' from the following column specification:
     *     UPPER(tableX.columnY)
     * WARNING: trickier column specifications will cause this method to fail.
     *
     * @param column     A column (field) name, possibly with a leading table name.
     */
    private String  getTableName(String  column) {
        int offset = 0;
        int end = 0;
        String  str = column;
        int index = str.lastIndexOf("(");

        if (-1 != index) {
            str = str.substring(index + 1);
        }
        index = str.indexOf(")");
        if (-1 != index) {
            str = str.substring(0, index);
        }
        index = str.indexOf(".");
        if (-1 == index) {
            return null;
        }
        return str.substring(0, index);
    }

    /**
     * Returnes multiTableJoin attribute.
     *
     * @return multiTableJoin attribute.
     */
    public boolean isMultiTableJoin() {
        return multiTableJoin;
    }
    
    /**
     * Returnes unionTableJoin attribute.
     *
     * @return unionTableJoin attribute.
     */
    public boolean isUnionTableJoin() {
        return unionTableJoin;
    }
    /**
     * Returnes preventPrimaryKeySelect attribute.
     *
     * @return preventPrimaryKeySelect attribute.
     */
    public boolean getPreventPrimaryKeySelect() {
        return preventPrimaryKeySelect;
    }

    /**
     * Resets selected fields.
     */
    public void resetSelectedFields() {
        selectedFields = new Vector ();
        selectClause = "";
    }

    /**
     * Store all table names that occur as prefix of column specifications.
     * All table names must appear in the "from" clause.
     *
     * @param column     A column (field) name, possibly with a leading table name.
     */
    private void storeTableNameForColumn(String  column) {
        String  table = getTableName(column);

        storeTableName(table);
    }

    /**
     * Store all table names that occur as prefix of column specifications.
     * All table names must appear in the "from" clause.
     *
     * @param column     RDBColumn containing table name.
     */
    private void storeTableNameForColumn(RDBColumn column) {
        storeTableName(column.getTableName());
    }

    /**
     * Store a table name.
     * All table names must appear in the "from" clause.
     *
     * @param table name.
     */
    private void storeTableName(String  table) {
        if (null != table) {
            tableNames.put(table, "");
        }
    }

    /**
     * <PRE>
     * Specify that the next conditional expression (where-clause)
     * should be preceded by OR instead of AND.  (AND is the default).
     * For example:
     *          addWhereClause( "person.fname", "Bob", EQUAL );
     *          addWhereClause( "person.lname", "Smith", EQUAL );
     *                  produces
     *          person.fname = 'Bob' AND person.lname = 'Smith'
     * But:
     *          addWhereClause( "person.fname", "Bob", EQUAL );
     *          addWhereOr();
     *          addWhereClause( "person.lname", "Smith", EQUAL );
     *                  produces
     *          person.fname = 'Bob' OR person.lname = 'Smith'
     * Disallows 'OR OR'.
     * Trailing ORs are removed by prepareStatement().
     *
     * </PRE>
     *
     * author Jay Gunter
     */
    public void addWhereOr() {
        int n = whereClauses.size();

        if (0 == n) {
            return;
        }
        String  prev = (String ) whereClauses.lastElement();

        if (!OR.equals(prev)) {
            whereClauses.addElement(OR);
        }
    }

    /**
     * Specify that the next conditional expression (where-clause)
     * should be preceded by '('.
     * Trailing OPEN_PARENs are removed by prepareStatement().
     *
     * author Jay Gunter
     */
    public void addWhereOpenParen() {
        whereClauses.addElement(OPEN_PAREN);

        /*
         for ( int i = 0; i < whereClauses.size(); i++ ) {
         System.err.println("whereClauses.elementAt("+i+")="+whereClauses.elementAt(i));
         }
         */
    }

    /**
     * Close a previous addWhereOpenParen.
     * Automatically disallows bad sequences like 'OR )' and '( )'.
     *
     * author Jay Gunter
     */
    public void addWhereCloseParen() {
        int n = whereClauses.size();

        if (0 == n) {
            return;
        }
        String  prev = (String ) whereClauses.lastElement();

        if (!(OR.equals(prev) || OPEN_PAREN.equals(prev))) {
            whereClauses.addElement(CLOSE_PAREN);
        }
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   int value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, GenericDO value, String  cmp_op) {
        BigDecimal  id = null;

        if (null != value) {
            id = value.get_OId().toBigDecimal();
        }
        _addWhereClause(column, id, cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a byte array.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   byte array to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, byte[] value, String  cmp_op) {
        String  v = null;

        if (null != value) {
            v = new String (value);
        }
        _addWhereClause(column, v, cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   int value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, java.sql.Time  value, String  cmp_op) {
        _addWhereClause(column, value, cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   int value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, java.sql.Timestamp  value, String  cmp_op) {
        _addWhereClause(column, value, cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   long value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, long value, String  cmp_op) {
        _addWhereClause(column, new Long (value), cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   double value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, double value, String  cmp_op) {
        _addWhereClause(column, new Double (value), cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   float value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, float value, String  cmp_op) {
        _addWhereClause(column, new Float (value), cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   BigDecimal value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, BigDecimal  value, String  cmp_op) {
        _addWhereClause(column, value, cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   int value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, int value, String  cmp_op) {
        _addWhereClause(column, new Integer (value), cmp_op);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   boolean value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, boolean value, String  cmp_op) {
        _addWhereClause(column, new Boolean (value), cmp_op);
    }

    /**
     * Add a where-clause that tests if a column is equal to a given String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   int value to compare against.
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, int value) {
        addWhere(column, value, EQUAL);
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   String value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhereLower(RDBColumn column, String  value, String  cmp_op) {
        if (caseInsensitiveDatabase) {
            _addWhereClause(column.getFullColumnName(), value, cmp_op);
        }else {
            _addWhereClause("LOWER(" + column.getFullColumnName() + ")", value, cmp_op);
        }
    }

    /**
     * Add a where-clause that compares a column against a String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   String value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, String  value, String  cmp_op) {
        _addWhereClause(column, value, cmp_op);
    }

    /**
     * Add a where-clause that tests if a column is equal to a given String.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   String value to compare against.
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, String  value) {
        addWhere(column, value, EQUAL);
    }

    /**
     * Add a where-clause that compares a column against a Date.
     * @param column  A column (field) name to use as a search constraint.
     * @param value   Date value to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column, java.sql.Date  value, String  cmp_op) {
        _addWhereClause(column, value, cmp_op);
    }

    /**
     * Add a where-clause that compares a column against another column.
     * @param column1  A column (field) name to use as a search constraint.
     * @param column2  A column (field) name to use as a search constraint.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column1, RDBColumn column2, String  cmp_op) {

        /*
         storeTableNameForColumn( column1 );
         storeTableNameForColumn( column2 );
         addTwoColumnWhereClause(       column1.getFullColumnName(),
         column2.getFullColumnName(),
         cmp_op );
         */
        String  cn1 = column1.getFullColumnName();
        String  cn2 = column2.getFullColumnName();

        storeTableNameForColumn(cn1);
        storeTableNameForColumn(cn2);
        whereClauses.addElement(cn1 + " " + cmp_op + " " + cn2);
        if(!column1.getTableName().equals(column2.getTableName()))
            multiTableJoin = true;
    }

    /**
     * Add a where-clause that tests if a column is equal to another column.
     * @param column1  A column (field) name to use as a search constraint.
     * @param column2  A column (field) name to use as a search constraint.
     *
     * author Jay Gunter
     */
    public void addWhere(RDBColumn column1, RDBColumn column2) {
        addWhere(column1, column2, EQUAL);
    }

    /**
     * Add a where-clause that tests if a column is null.
     * @param column  A column (field) name to use as a search constraint.
     *
     * author kapusta@grecodata.sk
     */
    public void addWhereIsNull(RDBColumn column) {
        _addWhereClause(column, null, IS_NULL);
    }

    /**
     * Add a where-clause that tests if a column is null.
     * @param column  A column (field) name to use as a search constraint.
     *
     * author kapusta@grecodata.sk
     */
    public void addWhereIsNotNull(RDBColumn column) {
        _addWhereClause(column, null, IS_NOT_NULL);
    }

    /**
     * Add a where-clause that tests if a column is null.
     * @param column  A column (field) name to use as a search constraint.
     * @deprecated  Use addWhereIsNull(RDBColumn column ) instead.
     */
    public void addWhereIsNull(String  column) {
        _addWhereClause(column, null, IS_NULL);
    }

    /**
     * Add a where-clause that tests if a column is null.
     * @param column  A column (field) name to use as a search constraint.
     * @deprecated  Use addWhereIsNotNull(RDBColumn column ) instead.
     */
    public void addWhereIsNotNull(String  column) {
        _addWhereClause(column, null, IS_NOT_NULL);
    }

    /**
     * Add a where-clause that imposes a RIGHT OUTER JOIN to
     * test if a column is equal to another column.
     * @param column1  A column (field) name to use as a search constraint.
     * @param column2  A column (field) name to use as a search constraint.
     *
     * author Jay Gunter
     */
    public void addWhereRightOuter(RDBColumn column1, RDBColumn column2) {
        String  cn1 = column1.getFullColumnName() + "(+)";
        String  cn2 = column2.getFullColumnName();

        storeTableNameForColumn(column1);
        storeTableNameForColumn(column2);
        whereClauses.addElement(cn1 + " = " + cn2);

        /*
         addTwoColumnWhereClause(
         column1.getFullColumnName() + "(+)",
         column2.getFullColumnName(), QueryBuilder.EQUAL );
         */
    }

    /**
     * Add a where-clause that imposes a LEFT OUTER JOIN to
     * test if a column is equal to another column.
     * @param column1  A column (field) name to use as a search constraint.
     * @param column2  A column (field) name to use as a search constraint.
     *
     * author Jay Gunter
     */
    public void addWhereLeftOuter(RDBColumn column1, RDBColumn column2) {
        String  cn1 = column1.getFullColumnName();
        String  cn2 = column2.getFullColumnName() + "(+)";

        storeTableNameForColumn(column1);
        storeTableNameForColumn(column2);
        whereClauses.addElement(cn1 + " = " + cn2);
    }

    /**
     * <PRE>
     * Add a where-clause to the SQL command.
     * The where-clause compare the values of two columns
     * using the specified comparison operator.
     * E.g., we want to find cats with too many paws:
     *      column1 = "cat.paws"
     *      column2 = "animalNorms.catPaws"
     *      cmp_op  = QueryBuilder.GREATER_THAN
     * generates the where-clause:
     *      cat.paws > animalNorms.catPaws
     * and remembers the table names (cat, animalNorms)
     * for inclusion in the from-clause.
     * </PRE>
     *
     * @param column1  A column (field) name to use as a search constraint.
     * @param column2  Another column to compare against.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     * @deprecated  Use addWhere(RDBColumn c1, RDBColumn c1, String cmp_op) instead.
     */
    public void addTwoColumnWhereClause(
        String  column1, String  column2, String  cmp_op) {
        storeTableNameForColumn(column1);
        storeTableNameForColumn(column2);
        whereClauses.addElement(column1 + " " + cmp_op + " " + column2);
    }

    /**
     * <PRE>
     * Add a where-clause to the SQL command.
     * The where-clause compare the values of two columns
     * using the EQUAL operator.
     * E.g., we want to find cats with the normal number of paws:
     *      column1 = "cat.paws"
     *      column2 = "animalNorms.catPaws"
     * generates the where-clause:
     *      cat.paws = animalNorms.catPaws
     * and remembers the table names (cat, animalNorms)
     * for inclusion in the from-clause.
     *
     * </PRE>
     *
     * @param column1  A column (field) name to use as a search constraint.
     * @param column2  Another column to compare against.
     * @deprecated  Use addWhere(RDBColumn c1, RDBColumn c1) instead.
     */
    public void addTwoColumnWhereClause(String  column1, String  column2) {
        addTwoColumnWhereClause(column1, column2, EQUAL);
    }

    /**
     * Replace a substring with another string.
     *
     * @return the resulting string after replacement.
     * Returns null if any params are null.
     * Returns original string if 'find' not found.
     * @param s the original string.
     * @param find the substring to replace.
     * @param replace the replacement string.
     * author Jay Gunter
     */
    public static String  stringReplace(String  s, String  find, String  replace) {
        if (null == s || null == find || null == replace) {
            return s;
        }
        int x;

        if (-1 == (x = s.indexOf(find))) {
            return s;
        }
        String  ret = s.substring(0, x);

        ret += replace;
        ret += s.substring(x + find.length());
        return ret;
    }

    private void _addWhereClause(RDBColumn column, Object  value, String  cmp_op) {
        storeTableNameForColumn(column);
        buildWhereClause(column.getFullColumnName(), value, cmp_op);
    }

    private void _addWhereClause(String  column, Object  value, String  cmp_op) {
        storeTableNameForColumn(column);
        buildWhereClause(column, value, cmp_op);
    }

    private void buildWhereClause(String  column, Object  value, String  cmp_op) {
        // WebDocWf extension for extended wildcard support
 // The following line has been changed:
 String  escapeString = "";

        // end of WebDocWf extension for extended wildcard support
 if (null == value) {
            if (EQUAL.equals(cmp_op)) {
                cmp_op = IS_NULL;
            } else if (NOT_EQUAL.equals(cmp_op)) {
                cmp_op = IS_NOT_NULL;
            }
            whereClauses.addElement(column + cmp_op);
        } else if (IS_NULL.equals(cmp_op) || IS_NOT_NULL.equals(cmp_op)) {
            whereClauses.addElement(column + cmp_op);
        } else {
            // WebDocWf extension for extended wildcard support
 // The following line has been changed:
 if ((-1 != cmp_op.indexOf('%'))
                || (cmp_op.equals(CASE_INSENSITIVE_MATCH))
                || (cmp_op.equals(CASE_SENSITIVE_MATCH))
                || (cmp_op.equals(USER_CASE_INSENSITIVE_MATCH))
                || (cmp_op.equals(USER_CASE_SENSITIVE_MATCH))) {  // CASE_...
 // before:      if ( -1 != cmp_op.indexOf( '%' ) ) {  // CASE_...
 // end of WebDocWf extension for extended wildcard support
 // WebDocWf extension for extended wildcard support
 // The following line has been changed:
 // There are 11 wildcarded string comparison operators:
 // before:          // There are 7 wildcarded string comparison operators:
 // end of WebDocWf extension for extended wildcard support
 // CASE_INSENSITIVE_EQUAL               = "%xxxx";
 // CASE_SENSITIVE_CONTAINS              = "%X%";
 // CASE_INSENSITIVE_CONTAINS    = "%x%";
 // CASE_SENSITIVE_STARTS_WITH   = "X%";
 // CASE_INSENSITIVE_STARTS_WITH = "x%";
 // CASE_SENSITIVE_ENDS_WITH     = "%X";
 // CASE_INSENSITIVE_ENDS_WITH   = "%x";
 // WebDocWf extension for extended wildcard support
 // The following lines have been added:
 // CASE_INSENSITIVE_MATCH       = "x";
 // CASE_SENSITIVE_MATCH = "X";
 // USER_CASE_INSENSITIVE_MATCH  = "u";
 // USER_CASE_SENSITIVE_MATCH    = "U";
 // end of WebDocWf extension for extended wildcard support
 // A lower-case 'x' means a case-insensitive comparison is desired.
 String  s = (String ) value;

                // WebDocWf extension for extended wildcard support
 // The following lines have been added:
 if (-1 != cmp_op.indexOf('u')) {
                    s = convertUserSearchValue(s);
                    if (userTrimString) {
                        s = s.trim();
                    }
                    if (userAppendWildcard) {
                        s = s + wildcard;
                    }
                    cmp_op = stringReplace(cmp_op, "u", "x");
                }
                if (-1 != cmp_op.indexOf('U')) {
                    s = convertUserSearchValue(s);
                    if (userTrimString) {
                        s = s.trim();
                    }
                    if (userAppendWildcard) {
                        s = s + wildcard;
                    }
                    cmp_op = stringReplace(cmp_op, "U", "X");
                }
                // end of WebDocWf extension for extended wildcard support
 if (-1 != cmp_op.indexOf('x')) {
                    // for case-insensitive comparisons,
 // push both column and value to lower case.
                    s = s.toLowerCase();
                    // Or, could do
 // s = "LOWER(" + s + ")";
 // David Corbin thought that might be better somehow.
 // I'd rather not count on the database optimizer
 // any more than necessary.
 if (!caseInsensitiveDatabase) {
                         column = "LOWER( " + column + ")";
                    }
                }
                // now we transform the CASE_ operator into the value
 // and set the comparison operator
 if (cmp_op.equals(CASE_INSENSITIVE_EQUAL)) {
                    cmp_op = EQUAL;
                    value = s;
                } else {
                    value = stringReplace(cmp_op.toLowerCase(), "x", s);
                    // WebDocWf extension for extended wildcard support
 // The following lines has been added:
 if (containsWildcards((String ) value)) {
                        if (((String ) value).equals(wildcard)) {
                            return;
                        }
                        escapeString = " " + wildcardEscapeClause;
                        // end of WebDocWf extension for extended wildcard support
                        cmp_op = likeKeyword;
                        // WebDocWf extension for extended wildcard support
 // The following lines has been added:
                    } else {
                        cmp_op = EQUAL;
                    }
                    // end of WebDocWf extension for extended wildcard support
                }
            }
            // WebDocWf extension for extended wildcard support
 // The following line has been changed:
            whereClauses.addElement(column + " " + cmp_op + " ?" + escapeString);
            // before : whereClauses.addElement( column + " " + cmp_op + " ?" );
 // end of WebDocWf extension for extended wildcard support
            parms.addElement(value);
        }
    }

    /**
     * Add a where-clause that compares a column against a string value
     * to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value to search for.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     * @deprecated  Use addWhere(RDBColumn column, String value, String cmp_op ) instead.
     */
    public void addWhereClause(String  column, String  value, String  cmp_op) {
        _addWhereClause(column, value, cmp_op);
    }

    // WebDocWf extension for extended wildcard support
 // All following lines have been added:
 /**
     * Add a DB specific match/like clause to the query
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value containing DB (vendor) specific wildcards to search for.
     * @return Whether a where-clause was generated.
     * @deprecated Use comparison operators instead
     */
    public boolean addMatchClause(String  column, String  value) {
        String  append = "";
        String  pattern = value;
        boolean added = false;

        if (pattern != null) {
            if (userTrimString) {
                pattern = pattern.trim();
            }
            if (userAppendWildcard) {
                pattern = pattern + wildcard;
            }
            if (!pattern.equals(wildcard)) {
                append = " " + column + " " + likeKeyword + " ? "
                    + wildcardEscapeClause;
                added = true;
                parms.addElement(pattern);
            }
        }
        if (added) {
            storeTableNameForColumn(column);
            whereClauses.addElement(append);
        }
        return added;
    }

    /**
     * Add a User specific match/like clause to the query
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value containing User specific wildcards to search for.
     * @return Whether a where-clause was generated.
     * @deprecated Use comparison operators instead
     */
    public boolean addUserMatchClause(String  column, String  value) {
        boolean ret = false;
        String  pattern = convertUserSearchValue(value);

        if (containsWildcards(pattern) || userAppendWildcard) {
            ret = addMatchClause(column, pattern);
        } else {
            addWhereClause(column, value, QueryBuilder.EQUAL);
            ret = true;
        }
        return ret;
    }

    /**
     * Add a DB specific match/like clause to the query
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param column   The column (field) to use as a search constraint.
     * @param value    The value containing DB (vendor) specific wildcards to search for.
     * @return Whether a where-clause was generated.
     * @deprecated Use comparison operators instead
     */
    public boolean addMatchClause(RDBColumn column, String  value) {
        return addMatchClause(column.getFullColumnName(), value);
    }

    /**
     * Add a User specific match/like clause to the query
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param column   The column (field) to use as a search constraint.
     * @param value    The value containing User specific wildcards to search for.
     * @return Whether a where-clause was generated.
     * @deprecated Use comparison operators instead
     */
    public boolean addUserMatchClause(RDBColumn column, String  value) {
        return addUserMatchClause(column.getFullColumnName(), value);
    }

    /**
     * Evaluate if string contains Wildcards
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param value    The value containing DB (vendor) specific wildcards to search for.
     * @return Whether the value contains wildcards
     */
    public boolean containsWildcards(String  value) {
        boolean ret = false;

        if (-1 != value.indexOf(wildcard)) {
            ret = true;
        }
        if (-1 != value.indexOf(singleWildcard)) {
            ret = true;
        }
        return ret;
    }

    /**
     * Replace User wildcards with DB vendor specific wildcards according to dods.conf
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param value    The string with User wildcards to be changed
     * @return The string with DB vendor specific wildcards
     */
    public String  convertUserSearchValue(String  value) {
        String  ret;

        if (value == null) {
            ret = null;
        } else {
            ret = "";
            for (int x = 0; x < value.length(); x++) {
                if (value.startsWith(userSingleWildcard, x)) {
                    ret = ret + singleWildcard;
                } else {
                    if (value.startsWith(userWildcard, x)) {
                        ret = ret + wildcard;
                    } else {
                        if (value.startsWith(wildcard, x)) {
                            ret = ret + wildcardEscape + wildcard;
                        } else {
                            if (value.startsWith(singleWildcard, x)) {
                                ret = ret + singleWildcardEscape
                                    + singleWildcard;
                            } else {
                                if (x + 1 < value.length()) {
                                    if (value.startsWith(userSingleWildcardEscape
                                                             + userSingleWildcard,
                                                         x)) {
                                        ret = ret + userSingleWildcard;
                                        x++;
                                    } else {
                                        if (value.startsWith(userWildcardEscape
                                                                 + userWildcard,
                                                             x)) {
                                            ret = ret + userWildcard;
                                            x++;
                                        } else {
                                            if (value.startsWith(userWildcardEscape
                                                                     + userWildcardEscape,
                                                                 x)) {
                                                ret = ret + userWildcardEscape;
                                                if (userWildcardEscape
                                                    == wildcardEscape) {
                                                    ret = ret + wildcardEscape;
                                                }
                                                x++;
                                            } else {
                                                if (value.startsWith(userSingleWildcardEscape
                                                                         + userSingleWildcardEscape,
                                                                     x)) {
                                                    ret = ret
                                                        + userSingleWildcardEscape;
                                                    if (userSingleWildcardEscape
                                                        == singleWildcardEscape) {
                                                        ret = ret
                                                            + singleWildcardEscape;
                                                    }
                                                    x++;
                                                } else {
                                                    ret = ret
                                                        + value.substring(x,
                                                                          x + 1);
                                                    if (value.startsWith(singleWildcardEscape,
                                                                         x)) {
                                                        ret = ret
                                                            + singleWildcardEscape;
                                                    } else {
                                                        if (value.startsWith(wildcardEscape,
                                                                             x)) {
                                                            ret = ret
                                                                + wildcardEscape;
                                                        }
                                                    }
                                                }
                                            }
                                        }
                                    }
                                } else {
                                    ret = ret + value.substring(x, x + 1);
                                    if (value.startsWith(singleWildcardEscape, x)) {
                                        ret = ret + singleWildcardEscape;
                                    } else {
                                        if (value.startsWith(wildcardEscape, x)) {
                                            ret = ret + wildcardEscape;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        return ret;
    }

    // end of second part WebDocWf extension for extended wildcard support
 /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value to search for.
     * @param nullOk   Whether a value of NULL is acceptable.
     * @return Whether a where-clause was generated.
     * false if value==null and nullOk==false.
     * @deprecated  Use addWhere(RDBColumn c1, BigDecimal value, String cmp_op) instead.
     */
    public boolean addWhereClause(String  column, BigDecimal  value,
                                  boolean nullOk) {
        storeTableNameForColumn(column);
        String  append = "";
        boolean added = false;

        if (value == null) {
            if (nullOk) {
                append = " " + column + " is null";
                added = true;
            }
        } else {
            append = " " + column + " = ?";
            added = true;
            parms.addElement(value);
        }
        if (added) {
            whereClauses.addElement(append);
        }
        return added;
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The int value to search for.
     * @deprecated  Use addWhere(RDBColumn column, int value) instead.
     */
    public void addWhereClause(String  column, int value) {
        _addWhereClause(column, new Integer (value), EQUAL);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The int value to search for.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     * @deprecated  Use addWhere(RDBColumn column, int value, String cmp_op ) instead.
     */
    public void addWhereClause(String  column, int value, String  cmp_op) {
        _addWhereClause(column, new Integer (value), cmp_op);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The long value to search for.
     * @deprecated  Use addWhere(RDBColumn column, long value) instead.
     */
    public void addWhereClause(String  column, long value) {
        _addWhereClause(column, new Long (value), EQUAL);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The long value to search for.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     * @deprecated  Use addWhere(RDBColumn column, long value, String cmp_op ) instead.
     */
    public void addWhereClause(String  column, long value, String  cmp_op) {
        _addWhereClause(column, new Long (value), cmp_op);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value to search for.
     * @deprecated  Use addWhere(RDBColumn column, float value) instead.
     */
    public void addWhereClause(String  column, float value) {
        _addWhereClause(column, new Float (value), EQUAL);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value to search for.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     * @deprecated  Use addWhere(RDBColumn column, float value, String cmp_op ) instead.
     */
    public void addWhereClause(String  column, float value, String  cmp_op) {
        _addWhereClause(column, new Float (value), cmp_op);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value to search for.
     * @deprecated  Use addWhere(RDBColumn column, double value) instead.
     */
    public void addWhereClause(String  column, double value) {
        _addWhereClause(column, new Double (value), EQUAL);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value to search for.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     * @deprecated  Use addWhere(RDBColumn column, double value, String cmp_op ) instead.
     */
    public void addWhereClause(String  column, double value, String  cmp_op) {
        _addWhereClause(column, new Double (value), cmp_op);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value to search for.
     * @deprecated  Use addWhere(RDBColumn column, boolean value) instead.
     */
    public void addWhereClause(String  column, boolean value) {
        _addWhereClause(column, new Integer (value ? 1 : 0), EQUAL);
    }

    /**
     * Add a where-clause to the SQL command.
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param value    The value to search for.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     * @deprecated  Use addWhere(RDBColumn column, boolean value, String cmp_op ) instead.
     */
    public void addWhereClause(String  column, boolean value, String  cmp_op) {
        _addWhereClause(column, new Integer (value ? 1 : 0), cmp_op);
    }

    /**
     * Add a trailing clause (order by, group by, etc.) to the SQL command.
     *
     * @param clause     The clause to append to the SQL command.
     */
    public void addEndClause(String  clause) {
        endClauses.put(clause, "");
        preventPrimaryKeySelect = true;
    }

    /**
     * Append to the 'select' portion of the SQL command.
     *
     * @param str     The string to append to the SQL command.
     * @deprecated  Use addEndClause() instead.
     */
    public void add(String  str) {
        addEndClause(str);
    }

    /**
     * Sets debug flag to print the SQL statement just before its execution.
     * Affects just this instance of QueryBuilder.
     */
    public void debug() {
        debugSQL = true;
    }

    /**
     * Sets debug flag to print the SQL statement just before its execution.
     * Affects all instances of QueryBuilder.
     */
    static public void debugAll() {
        debugAllSQL = true;
    }

    /**
     * Add DISTINCT keyword to query.
     */
    private String  distinct = "";
    public void distinct() {
        distinct = "distinct ";
    }

    public void addWhere(String  fullClause) {
        whereClauses.addElement(fullClause);
        multiTableJoin = true;
        preventPrimaryKeySelect = true;
    }

    // sinisa 14.06.2004. new addWhereIn methods, to be added in complex query cache


    /**
     * QueryBuilder.addWhereIn( RDBColumn, String[] )
     * allows queries like
     *   select Person.Age from Person
     *   where Person.Name in ( "Bob", "Joe" )
     * with code like
     *   QueryBuilder qb = new QueryBuilder();
     *   qb.select( PersonDO.Age );
     *   String[] values = { "Bob", "Joe" };
     *   qb.addWhereIn( PersonDO.Name, values );
     *
     * @param val : PersonDO.Name in the example above
     * @param values : array of Strings containing values for IN clause
     *
     * @exception QueryException if values is null or empty
     * author Sinisa Milosevic
     */
    public void addWhereIn(RDBColumn val, String [] values)
        throws QueryException {
        if (null == values || 0 == values.length) {
            throw new QueryException("IN clause must have at least one value.");
        }
        if (null == val) {
            throw new QueryException("IN clause left value cannot be null");
        }
        String  comma = "";
        String  qmarks = "";

        for (int i = 0; i < values.length; i++) {
            String  v = values[i];

            parms.addElement(values[i]);
            qmarks += comma + "?";
            comma = ", ";
        }

        storeTableName(val.getTableName());
        addWhere(val.getFullColumnName() + OPEN_IN + qmarks + CLOSE_IN);
    }


    /**
     * QueryBuilder.addWhereIn( RDBColumn, BigDecimal[] )
     * allows queries like
     *   select Person.Age from Person
     *   where Person.oid in ( 100, 101 )
     * with code like
     *   QueryBuilder qb = new QueryBuilder();
     *   qb.select( PersonDO.Age );
     *   BigDecimal[] values = { new BigDecimal("100") , new BigDecimal("101") };
     *   qb.addWhereIn( PersonDO.Name, values );
     *
     * @param val : PersonDO.Name in the example above
     * @param values : array of Strings containing values for IN clause
     *
     * @exception QueryException if values is null or empty
     * author Sinisa Milosevic
     */
    public void addWhereIn(RDBColumn val, BigDecimal [] values)
        throws QueryException {
        if (null == values || 0 == values.length) {
            throw new QueryException("IN clause must have at least one value.");
        }
        if (null == val) {
            throw new QueryException("IN clause left value cannot be null");
        }
        String  comma = "";
        String  qmarks = "";

        for (int i = 0; i < values.length; i++) {
            BigDecimal  v = values[i];
            if(v!=null) {
                //               parms.addElement(values[i]);
                qmarks += comma + v.toString();
                comma = ", ";
            }
        }

        storeTableName(val.getTableName());
        addWhere(val.getFullColumnName() + OPEN_IN + qmarks + CLOSE_IN);
    }



    /**
     * QueryBuilder.addWhereIn( RDBColumn, ObjectId[] )
     * allows queries like
     *   select Person.Age from Person
     *   where Person.oid in ( 100, 101 )
     * with code like
     *   QueryBuilder qb = new QueryBuilder();
     *   qb.select( PersonDO.Age );
     *   ObjectId[] values = { new ObjectId("100") , new ObjectId("101") };
     *   qb.addWhereIn( PersonDO.Name, values );
     *
     * @param val : PersonDO.Name in the example above
     * @param values : array of Strings containing values for IN clause
     *
     * @exception QueryException if values is null or empty
     * author Sinisa Milosevic
     */
    public void addWhereIn(RDBColumn val, ObjectId[] values)
        throws QueryException {
        if (null == values || 0 == values.length) {
            throw new QueryException("IN clause must have at least one value.");
        }
        if (null == val) {
            throw new QueryException("IN clause left value cannot be null");
        }
        String  comma = "";
        String  qmarks = "";

        for (int i = 0; i < values.length; i++) {
            ObjectId v = values[i];
            if(v!=null) {
                qmarks += comma + v.toString();
                comma = ", ";
            }
        }

        storeTableName(val.getTableName());
        addWhere(val.getFullColumnName() + OPEN_IN + qmarks + CLOSE_IN);
    }


    /**
     * QueryBuilder.addWhereIn( RDBColumn, GenericDO[] )
     * allows queries like
     *   select Person.Age from Person
     *   where Person.oid in ( 100, 101 )
     * with code like
     *   QueryBuilder qb = new QueryBuilder();
     *   qb.select( PersonDO.Age );
     *   GenericDO[] values = { new GenericDO("100") , new GenericDO("101") };
     *   qb.addWhereIn( PersonDO.Name, values );
     *
     * @param val : PersonDO.oid in the example above
     * @param values : array of Strings containing values for IN clause
     *
     * @exception QueryException if values is null or empty
     * author Sinisa Milosevic
     */
    public void addWhereIn(RDBColumn val, GenericDO[] values)
        throws QueryException {
        if (null == values || 0 == values.length) {
            throw new QueryException("IN clause must have at least one value.");
        }
        if (null == val) {
            throw new QueryException("IN clause left value cannot be null");
        }
        String  comma = "";
        String  qmarks = "";

        for (int i = 0; i < values.length; i++) {
            GenericDO v = values[i];
            if(v!=null) {
                try {
                    qmarks += comma + v.getHandle();
                    comma = ", ";
                } catch (DatabaseManagerException e) {
                    throw new QueryException("Error during preparing where clause in SQL statement");
                }

            }
        }

        storeTableName(val.getTableName());
        addWhere(val.getFullColumnName() + OPEN_IN + qmarks + CLOSE_IN);
    }

    //end sinisa 14.06.2004

    /**
     * QueryBuilder.addWhereIn( RDBColumn, QueryBuilder )
     * allows queries like
     *   select Person.Name from Person where Person.City
     *   in ( select City.oid from City where City.Size > 1000 )
     * with code like
     *   QueryBuilder mainQuery = new QueryBuilder();
     *   mainQuery.select( PersonDO.Name );
     *   QueryBuilder subQuery = new QueryBuilder();
     *   subQuery.select( CityDO.PrimaryKey );
     *   subQuery.addWhere( CityDO.Size, 1000, QueryBuilder.GREATER_THAN );
     *   mainQuery.addWhereIn( PersonDO.City, subQuery );
     *
     * @param field T1DO.T1Y in the example above
     * @param qb subQuery in the example above
     * author Jay Gunter
     */
    // following line is removed - no longer needed: rka
 // prev: private Vector subQueries = new Vector();
 public void addWhereIn(RDBColumn field, QueryBuilder qb)
        throws QueryException {
        if (this == qb) {
            throw new QueryException("Recursion detection: subquery cannot be "
                                         + "the same object as the main query.");
        }
        storeTableName(field.getTableName());
        addWhere(field.getFullColumnName() + OPEN_IN + qb.getSQL() + CLOSE_IN);
        // the following line is changed: rka
        parms.addElement(qb);
        // prev: subQueries.addElement( qb );
        multiTableJoin = true;
    }
  
    /**
     * Reset (remove) "UNION [ALL]" part of query.
     */
    public void resetUnion(){
        doUnionAll = new Vector ();  // 
        unionQuerys= new Vector ();  //
        unionTableJoin = false;
    }
    /**
     * QueryBuilder.addUnion(QueryBuilder qb, boolean all)
     *
     * @param qb  Additional query that will be added in "UNION" part of this query.
     * @param all If this parameter is set to "true" then union clausule will be set to "UNION ALL" instead of "UNION"
     */
    
    public void addUnion(QueryBuilder qb, boolean all)
    throws QueryException {
        if (this == qb) {
            throw new QueryException("Recursion detection: union query cannot be the same object as the main query.");
        }
        doUnionAll.add(new Boolean (all));
        unionQuerys.add(qb);
        unionTableJoin = true;
    }


    /**
     * QueryBuilder.addWhereExists( QueryBuilder )
     *
     * @param qb subQuery in the example above
     * author rkapusta@together.sk
     */
    public void addWhereExists(QueryBuilder qb)
        throws QueryException {
        if (this == qb) {
            throw new QueryException("Recursion detection: subquery cannot be "
                                         + "the same object as the main query.");
        }
        addWhere(OPEN_EXISTS + qb.getSQL() + CLOSE_IN);
        parms.addElement(qb);
        multiTableJoin = true;
    }

    /**
     * QueryBuilder.addWhereNotExists( QueryBuilder )
     *
     * @param qb subQuery in the example above
     * author rkapusta@together.sk
     */
    public void addWhereNotExists(QueryBuilder qb)
        throws QueryException {
        if (this == qb) {
            throw new QueryException("Recursion detection: subquery cannot be "
                                         + "the same object as the main query.");
        }
        addWhere(OPEN_NOT_EXISTS + qb.getSQL() + CLOSE_IN);
        parms.addElement(qb);
        multiTableJoin = true;
    }

    /**
     * QueryBuilder.addWhereNotIn( RDBColumn, QueryBuilder )
     * allows queries like
     *   select Person.Name from Person where Person.City
     *   not in ( select City.oid from City where City.Size > 1000 )
     * with code like
     *   QueryBuilder mainQuery = new QueryBuilder();
     *   mainQuery.select( PersonDO.Name );
     *   QueryBuilder subQuery = new QueryBuilder();
     *   subQuery.select( CityDO.PrimaryKey );
     *   subQuery.addWhere( CityDO.Size, 1000, QueryBuilder.GREATER_THAN );
     *   mainQuery.addWhereNotIn( PersonDO.City, subQuery );
     *
     * @param field T1DO.T1Y in the example above
     * @param qb subQuery in the example above
     * author Jay Gunter
     */
    public void addWhereNotIn(RDBColumn field, QueryBuilder qb)
        throws QueryException {
        if (this == qb) {
            throw new QueryException("Recursion detection: subquery cannot be "
                                         + "the same object as the main query.");
        }
        storeTableName(field.getTableName());
        addWhere(field.getFullColumnName() + OPEN_NOT_IN + qb.getSQL()
                     + CLOSE_IN);
        // the following line is changed
        parms.addElement(qb);
        // prev: subQueries.addElement( qb );
        multiTableJoin = true;
    }

    /**
     * QueryBuilder.addWhereIn( Object, Object[] )
     * allows queries like
     *   select Person.Age from Person
     *   where Person.Name in ( "Bob", "Joe" )
     * with code like
     *   QueryBuilder qb = new QueryBuilder();
     *   qb.select( PersonDO.Age );
     *   String[] values = { "Bob", "Joe" };
     *   qb.addWhereIn( PersonDO.Name, values );
     *
     * or query
     *   select Person.Age from Person
     *   where "Tom" in ( Person.Surname, Person.Nick )
     * with code
     *   QueryBuilder qb = new QueryBuilder();
     *   qb.select( PersonDO.Age );
     *   RDBColumn[] values = { PersonDO.Surname, PersonDO.Nick };
     *   qb.addWhereIn( "Tom", values );
     *
     * @param val : PersonDO.Name & "Tom" in the example above
     * @param values : array of Objects containing values for IN clause
     *              any of previous Object types can be RDBColumn, String,
     *              java.sql.Date, java.sql.Timestamp, BigDecimal, Integer,
     *              Long, Float, Double, Boolean
     * @exception QueryException if values is null or empty
     * author Jay Gunter
     */
    public void addWhereIn(Object  val, Object [] values)
        throws QueryException {
        if (null == values || 0 == values.length) {
            throw new QueryException("IN clause must have at least one value.");
        }
        if (null == val) {
            throw new QueryException("IN clause left value cannot be null");
        }
        String  comma = "";
        String  qmarks = "";

        for (int i = 0; i < values.length; i++) {
            Object  v = values[i];

            if (v instanceof RDBColumn) {
                RDBColumn f = (RDBColumn) v;

                storeTableName(f.getTableName());
                qmarks += comma + f.getFullColumnName();
            } else {
                parms.addElement(values[i]);
                qmarks += comma + "?";
            }
            comma = ", ";
        }
        if (val instanceof RDBColumn) {
            RDBColumn f = (RDBColumn) val;

            storeTableName(f.getTableName());
            addWhere(f.getFullColumnName() + OPEN_IN + qmarks + CLOSE_IN);
        } else {
            parms.addElement(val);
            addWhere("?" + OPEN_IN + qmarks + CLOSE_IN);
        }
    }

    /**
     * QueryBuilder.addWhereNotIn( Object, Object[] )
     * see addWhereIn( Object val, Object[] values ) for example
     *
     * @param val
     * @param values
     * @exception QueryException if values is null or empty
     * author Jay Gunter
     */
    public void addWhereNotIn(Object  val, Object [] values)
        throws QueryException {
        if (null == values || 0 == values.length) {
            throw new QueryException("NOT IN clause must have at least one value.");
        }
        if (null == val) {
            throw new QueryException("NOT IN clause left value cannot be null");
        }
        String  comma = "";
        String  qmarks = "";

        for (int i = 0; i < values.length; i++) {
            Object  v = values[i];

            if (v instanceof RDBColumn) {
                RDBColumn f = (RDBColumn) v;

                storeTableName(f.getTableName());
                qmarks += comma + f.getFullColumnName();
            } else {
                parms.addElement(values[i]);
                qmarks += comma + "?";
            }
            comma = ", ";
        }
        if (val instanceof RDBColumn) {
            RDBColumn f = (RDBColumn) val;

            storeTableName(f.getTableName());
            addWhere(f.getFullColumnName() + OPEN_NOT_IN + qmarks + CLOSE_IN);
        } else {
            parms.addElement(val);
            addWhere("?" + OPEN_NOT_IN + qmarks + CLOSE_IN);
        }
    }

    // WebDocWf extension for getting SQL command with parms
 public String  getSQLwithParms() {
        return recurseSQLwithParms(getSQL());
    }

    private String  recurseSQLwithParms(String  sSQL) {
        try {
            for (int iParm = 0; iParm < parms.size(); iParm++) {
                Object  o = parms.get(iParm);           
                int iIndex = sSQL.indexOf("?");
                
                if (o instanceof QueryBuilder) {
                    sSQL = ((QueryBuilder) o).recurseSQLwithParms(sSQL);
                } else {
                    StringBuffer  sbSQL = new StringBuffer (sSQL);
                    String  s = o.toString();
                    
                    sbSQL.replace(iIndex, iIndex + 1, s);
                    sSQL = sbSQL.toString();
                }                               
            }
        } catch (Exception  ex) {
            sSQL = sSQL + " : error occured during getting SQL command : " + ex;
        }
        return sSQL;
    }

    // end webdocwf 310.2.26 changes
 public String  getSQL() {
        // Query begins with SELECT <fields> FROM <tablelist>
 if (iQueryType == UPDATE_QUERY) {
            sql = "update ";
        } else if (iQueryType == DELETE_QUERY) {
            sql = "delete from ";
        } else {  
            
            if(selectClause!=null){
                boolean withPrefix = selectClause.startsWith("@T@_");
                boolean withNoPrefix = selectClause.startsWith("@F@_");
                String  fakeTableName;
                String  tableName="";
                if (withPrefix||withNoPrefix) {
                    tableName =selectClause.substring(selectClause.indexOf("_")+1,selectClause.indexOf("_@@."));
                    if(withPrefix) {
                        fakeTableName = "@T@_"+tableName+"_@@.";
                    } else {
                        fakeTableName = "@F@_"+tableName+"_@@.";
                    }
                    tableName=tableName+".";
                    if(withPrefix || isMultiTableJoin()) {
                        selectClause = Common.replaceAll(selectClause, fakeTableName, tableName);
                    }else{
                        selectClause = Common.replaceAll(selectClause, fakeTableName, "");
                    }
                }
            }
   
            String  glue = 0 == selectClause.length() ? "" : ", ";

            for (int i = 0; i < selectedFields.size(); i++) {
                RDBColumn field = (RDBColumn) selectedFields.elementAt(i);

                selectClause += glue + field.getFullColumnName();
                storeTableName(mainTableName = field.getTableName());
                glue = ", ";
            }
            sql = "select " + distinct + selectClause + " from ";
        }
        Enumeration  tables = tableNames.keys();

        for (int i = 0; tables.hasMoreElements(); i++) {
            if (i > 0) {
                sql += ", ";
            }
            sql += tables.nextElement();
        }
        if (iQueryType == UPDATE_QUERY) {
            sql += " set ";
            int nr_updateFields = updateFields.size();

            for (int i = 0; i < nr_updateFields; i++) {
                if (i > 0) {
                    sql += ", ";
                }
                sql += updateFields.elementAt(i);
            }
        }
        // Cleanup WHERE clauses.
 // Discard trailing ORs and OPEN_PARENs.
 String  chunk = null;
        int numWhereClauses = whereClauses.size();

        for (; numWhereClauses > 0; numWhereClauses--) {
            chunk = (String ) whereClauses.elementAt(numWhereClauses - 1);
            if (!(chunk.equals(OR) || chunk.equals(OPEN_PAREN))) {
                break;
            }
        }
        // Add WHERE clauses to query.
 int parenDepth = 0;
        boolean prevWas_OpenParen_or_OR = true;

        for (int i = 0; i < numWhereClauses; i++) {
            chunk = (String ) whereClauses.elementAt(i);

            /* System.err.println("chunk ="+chunk ); */
            if (0 == i) {
                sql += " WHERE ";
            }
            if (chunk.equals(OR)) {
                prevWas_OpenParen_or_OR = true;
            } else if (chunk.equals(OPEN_PAREN)) {
                if (!prevWas_OpenParen_or_OR) {
                    sql += " AND ";
                }
                prevWas_OpenParen_or_OR = true;
                parenDepth++;
            } else if (chunk.equals(CLOSE_PAREN)) {
                if (parenDepth > 0) {
                    prevWas_OpenParen_or_OR = false;
                    parenDepth--;
                } else {
                    chunk = "";
                }
            } else { // chunk is actual expression for where-clause
 if (!prevWas_OpenParen_or_OR) {
                    sql += " AND ";
                }
                prevWas_OpenParen_or_OR = false;
            }
            sql += chunk;
        }
        // Add any needed trailing CLOSE_PARENs.
 while (parenDepth-- > 0) {
            sql += CLOSE_PAREN;
        }
        if (iQueryType == SELECT_QUERY) {

            /*
             * ORDER BY code added by Chris Ryan (cryan@plugged.net.au)
             * The following code adds the ORDER BY 'xxx[, xxx]' to the
             * statement for querying.
             */
            int nr_OrderFields = orderFields.size();

            // ORDER BY required?
 if (nr_OrderFields > 0) {
                // yes, fill out the SQL...
                sql += " ORDER BY ";
                for (int i = 0; i < nr_OrderFields; i++) {
                    if (i > 0) {
                        sql += ", ";
                    }
                    sql += orderFields.elementAt(i) + " "
                        + orderDirections.elementAt(i);
                    // sql += orderFields.elementAt(i);
                }
            }
        }
        Enumeration  endings = endClauses.keys();

        while (endings.hasMoreElements()) {
            sql += endings.nextElement();
        }
        
        if (isUnionTableJoin()) {
            for (int i=0 ; i<unionQuerys.size() ; i++){
                QueryBuilder tmpQB=((QueryBuilder)unionQuerys.elementAt(i));
                if(((Boolean )doUnionAll.elementAt(i)).booleanValue()){
                    sql +="  UNION  ALL  "+tmpQB.getSQL();
                }else{
                    sql +="  UNION  "+tmpQB.getSQL();
                }
                parms.addElement(tmpQB);
            }
        }      
        
        return sql;
    }

    // WebDocWf fix, the following line has been changed
 public int applyParameters(PreparedStatement  ps, int paramCount)
        throws SQLException  {
        // before: public void applyParameters( PreparedStatement ps, int paramCount )
 for (int i = 0; i < parms.size(); i++) {
            int n = ++paramCount;
            Object     o = parms.elementAt(i);

            if (debugSQL || debugAllSQL) {
                System.out.print("Param: " + i + ", paramCount=" + paramCount);
                System.out.println(", Value: " + o);
            }
            if (o instanceof String ) {

                /*
                 if ( fuzzies.get( i ) ) {
                 if (likeKeyword.equals(DEFAULT_MATCHES_KEYWORD)) {
                 ps.setString(     n, getRegExpMatch( (String) o ) );
                 } else {
                 ps.setString(     n, (String)                 o );
                 }
                 } else {
                 ps.setString(     n, (String)                 o );
                 }
                 */
                ps.setString(n, (String ) o);
            } else if (o instanceof java.sql.Date ) { /* VB 2/2/2000 */
                ps.setDate(n, ((java.sql.Date ) o));
            } else if (o instanceof java.sql.Timestamp ) { /* VB 2/9/2000 */
                ps.setTimestamp(n, ((java.sql.Timestamp ) o));
            } else if (o instanceof BigDecimal ) {
                ps.setBigDecimal(n, (BigDecimal ) o);
            } else if (o instanceof Integer ) {
                ps.setInt(n, ((Integer ) o).intValue());
            } else if (o instanceof Long ) {
                ps.setLong(n, ((Long ) o).longValue());
            } else if (o instanceof Float ) {
                ps.setFloat(n, ((Float ) o).floatValue());
            } else if (o instanceof Double ) {
                ps.setDouble(n, ((Double ) o).doubleValue());
            } else if (o instanceof Boolean ) {
                ps.setBoolean(n, ((Boolean ) o).booleanValue());
            } else if (o instanceof GenericDO) {
                ps.setBigDecimal(n, ((GenericDO) o).get_OId().toBigDecimal());
            } // the following lines have been inserted: rka
 else if (o instanceof QueryBuilder) {
                QueryBuilder subQuery = (QueryBuilder) o;

                paramCount = subQuery.applyParameters(ps, --paramCount);
            }
            // end of insertion
        }
        // |the following lines have been removed due to change of subquery
 // |manipulation: rka
 /*
         for ( int i = 0; i < subQueries.size(); i++ ) {
         QueryBuilder subQuery = (QueryBuilder) subQueries.elementAt(i);
         // WebDocWf fix, the following line has been changed
         paramCount = subQuery.applyParameters( ps, paramCount );
         // before:     subQuery.applyParameters( ps, paramCount );
         }*/
        // |end of deletion
 // the following line has been inserted
 return paramCount;
    }

    /**
     * Generate a JDBC PreparedStatement using the values passed
     * for the where-clauses.
     *
     * @param conn   The DBConnection object to use for the search.
     */
    private String  sql = null;
    private void prepareStatement(DBConnection conn)
        throws SQLException  {
        getSQL();
        if (debugSQL || debugAllSQL) {
            System.out.println("\nQueryBuilder.prepareStatement sql=\n"
                                   + prettySQL(sql) + "\n");
        }
        try {
            if((iResultSetConcurrency == DEFAULT_RS_CONCURRENCY) && (iResultSetType == DEFAULT_RS_TYPE)) {
                stmt = conn.prepareStatement(sql);
            } else {
                stmt = ((ExtendedDBConnection)conn).prepareStatement(sql,iResultSetType,iResultSetConcurrency);
            }

            int qto = (0 < iCurrentQueryTimeout)
                ? iCurrentQueryTimeout
                : iDefaultQueryTimeout;

            if (qto > 0) {
                stmt.setQueryTimeout(qto);
            }
            // if ( maxRows > 0 )
 // WebDocWf extension cursor-handling
 // fetchsize is not defined for this query; try applicationfetchsize
 int ifs = (iCurrentFetchSize < 0)
                ? iDefaultFetchSize
                : iCurrentFetchSize;

            // check for queryfetchsize
 if (ifs > 0) {
                if (doCursorName(conn)){
                    stmt.setCursorName(getCountX());
                }
                if (doSetFetchSize(conn)){
                    stmt.setFetchSize(ifs);
                }
            }
            // WebDocWf extension cursor-handling end
            stmt.setMaxRows(maxRows);
        } catch (SQLException  se) {
            System.err.println("\n\nSQLException: " + se.getMessage() + ","
                                   + se.getErrorCode() + "," + se.getSQLState());
            throw se;
        }
        applyParameters(stmt, 0);
    }
    private int maxRows = -1;
    public void setMaxRows(int x) {
        if (0 > x) {
            x = 0;
        }
        maxRows = x;
    }
    private String []    breakWords = {
        "(", ")", "FROM", "from", "WHERE", "where", "AND", "and", "OR", "or" };
    private String  indent(int level) {
        String  ret = "";

        for (int i = 0; i < level; i++) {
            ret += "    ";
        }
        return ret;
    }

    private String  prettySQL(String  sql) {
        int next = 0;
        int sp = 0;
        int level = 0;
        int word = 0;
        String  out = "";
        int pastend = 9999999;

        sql = sql + " ";
        for (int offset = 0; offset < sql.length();) {
            int nearest = pastend;
            int i = 0;

            for (; i < breakWords.length; i++) {
                if (-1 != (next = sql.indexOf(breakWords[i], offset))) {
                    if (next < nearest) {
                        nearest = next;
                        word = i;
                    }
                }
            }
            next = nearest;
            i = word;
            if (i < breakWords.length && pastend != nearest) {
                int wordlen = breakWords[i].length();

                if (0 == i) {
                    level++;
                }
                out += sql.substring(offset, next);
                out += "\n";
                out += indent(level);
                out += sql.substring(next, next + wordlen) + " ";
                next += wordlen;
                if (1 == i) {
                    level--;
                }
                offset = next;
            } else {
                out += indent(level) + sql.substring(offset);
                break;
            }
        }
        return out;
    }

    /**
     * Return the JDBC PreparedStatement generated by prepareStatement().
     * @return The JDBC PreparedStatement.
     */
    public PreparedStatement  getStatement() {
        // Would anyone ever want this?
 return stmt;
    }

    /**
     * Return the JDBC PreparedStatement generated by prepareStatement().
     * @return The JDBC PreparedStatement.
     */
    public PreparedStatement  getStatement(DBConnection conn) throws SQLException  {
        prepareStatement(conn);
        return stmt;
    }

    /**
     * Execute the PreparedStatement.
     * @return The ResultSet object generated by JDBC for the search.
     */

    /*
     public ResultSet executeQuery() {
     conn = new DBConnection();
     return executeQuery( conn );
     }
     */

    /**
     * Execute the PreparedStatement and return a JDBC ResultSet.
     * @param conn   The DBConnection object to use for the search.
     * @return The ResultSet object generated by JDBC for the search.
     */
    public ResultSet  executeQuery(DBConnection conn)
        throws SQLException  {
        prepareStatement(conn);
        int maxExecuteTime = ((StandardDatabaseManager)DODS.getDatabaseManager()).getDatabaseManagerConfiguration().getMaxExecuteTime();
    
        boolean throwException = false;
        String   errMsg = "<no-defined>";
        
        Date  startQueryT = new Date ();
        try {
            rs = stmt.executeQuery();
        }catch(SQLException  esql){
            throwException=true;
            errMsg="<Replicated>:"+esql.getMessage();
            DODS.getLogChannel().write(Logger.ERROR," <SqlQueryExecute><SqlQueryException>("+ errMsg+")");
        }
        Date  stopQueryT = new Date ();
        
        int qTime = ( new Long (stopQueryT.getTime() - startQueryT.getTime())).intValue();
        int qto = (0 < iCurrentQueryTimeout)
        ? iCurrentQueryTimeout
        : iDefaultQueryTimeout;
        
        int queryTimeOutValue = qto*1000; //X(s)*1000 = (X*1000)(ms) 
 boolean overrun = (queryTimeOutValue>0)&&(qTime>=queryTimeOutValue);
        
        if (qTime>=maxExecuteTime)  {
            DODS.getLogChannel().write(Logger.DEBUG,
                            " <SqlQueryTime>" +
                            " :ExecuteTime="+qTime+" [MaxExecuteTime(ms)="+maxExecuteTime+"]"+
                            " :sql="+getSQLwithParms()+
                            " :execute time ="+qTime+
                            " :max execute time = "+maxExecuteTime);
        }
        if (overrun) {
            DODS.getLogChannel().write(Logger.DEBUG,
                            " <SqlQueryTimeOut>" +
                            " :ExecuteTime="+qTime+" [QueryTimeOut(ms)="+queryTimeOutValue+"]"+
                            " :sql="+getSQLwithParms()+
                            " :execute time ="+qTime+
                            " :QueryTimeOut ="+queryTimeOutValue);
        }
        
        if(throwException)throw new SQLException ("<SqlQueryExecute><SqlQueryException>("+errMsg+")");
        
        if(overrun) {
            try {
                if(rs!=null) {
                    rs.close();
                    rs=null;
                }
            }catch(SQLException  esql) {
                DODS.getLogChannel().write(Logger.DEBUG," <SqlQueryTimeOut> Forced ResultSet close error.");
            }
            throw new SQLException (" <SqlQueryTimeOut> Query timeout, forced close of ResultSet");
        }
        rsClosed = new Boolean (false);
        return rs;
    }

    /**
     * Reset the where-clause information.
     * The same fields specified in the QueryBuilder constructor,
     * can be retrieved for a different set of where-clauses.
     */
    public void reset() {
        setMaxRows(0);
        whereClauses = new Vector (); // keep the select, forget the where-clauses
        parms = new Vector ();   // discard parms for previous where-clause
        fuzzies = new BitSet (); // forget special string handling
        multiTableJoin = false;
        // Clear the ORDER BY Vectors
        orderFields = new Vector ();
        orderDirections = new Vector ();
        try {
            if (null != rs && null != rsClosed && !rsClosed.booleanValue()) {
                rsClosed = new Boolean (true);
                rs.close();
            }
            rs = null;
            rsClosed = new Boolean (true);
        } catch (SQLException  e) {
            DODS.getLogChannel().write(Logger.DEBUG,"QueryBuilder unable to close ResultSet: SQLException="
                                   + e.getMessage());
        }
        if (null != conn) {     // if this class created the connection, close it.
            conn.close();
        }
        conn = null;
    }

    public void close() {
        try {
            if (null != dbQuery) {
                dbQuery.release();
            }
            if (null != rs && null != rsClosed && !rsClosed.booleanValue()) {
                rsClosed = new Boolean (true);
                rs.close();
            }
            rsClosed = new Boolean (true);
            rs = null;
        } catch (SQLException  e) {
            DODS.getLogChannel().write(Logger.DEBUG,"QueryBuilder unable to close ResultSet: SQLException="
                                   + e.getMessage());
        }
    }

    /**
     * Returns a regular expression, as supported by "MATCHES"
     * that will will perform a case insensitive match of a substring.
     * @param base   The base String to convert to a regular expression.
     * @return       The generated regular expression.
     */
    private String  getRegExpMatch(String  base) {
        int len = base.length();
        char[] baseChars = new char[len];

        base.getChars(0, len, baseChars, 0);
        // The result will never exceed 5 * the length of
 // the original string.
 char[] result = new char[len * 5];
        int resultIdx = 0;

        for (int idx = 0; idx < baseChars.length; idx++) {
            if (Character.isSpaceChar(baseChars[idx])) {
                // Convert spaces to ' *'
 if ((resultIdx > 0) && (result[resultIdx - 1] != '*')) {
                    result[resultIdx++] = ' ';
                    // result[resultIdx++] = '*';
                    result[resultIdx++] = '%';
                }
            } else if (Character.isLetter(baseChars[idx])) {
                // Convert letters to [Aa]
                result[resultIdx++] = '[';
                result[resultIdx++] = Character.toUpperCase(baseChars[idx]);
                result[resultIdx++] = Character.toLowerCase(baseChars[idx]);
                result[resultIdx++] = ']';
            } else {
                // Leave everything else as is.
                result[resultIdx++] = baseChars[idx];
            }
        }
        // Match anything at end of line
        result[resultIdx++] = '*';
        return new String (result);
    }
    private String  mainTableName;                       // prepended to column names
 private Hashtable  tableNames = new Hashtable ();     // table names in FROM clause
 private Hashtable  endClauses = new Hashtable ();     // clauses appended to SQL
 private Vector  selectedFields = new Vector ();       // RDBColumns to return
 private String  selectClause = "";           // comma separated fields
 private Vector  whereClauses = new Vector (); // WHERE clauses
 private Vector  orderFields = new Vector ();  // fields for ORDER BY clauses
 private Vector  orderDirections = new Vector ();      // ascending/descending
 private PreparedStatement  stmt;                     // what we execute
 private Vector  parms;                               // parameters for stmt
 private BitSet  fuzzies;                             // which parms use regex match
 private int param = 0;                              // parameter index
 private DBConnection conn = null;           // for execution of stmt
 // [john] Update support {begin} -----------------------------------------------
 private Vector  updateFields = new Vector (); // fields for SET clauses
 public final static int SELECT_QUERY = 0;
    public final static int UPDATE_QUERY = 1;
    public final static int DELETE_QUERY = 2;
    private int iQueryType = SELECT_QUERY;
    public void setDeleteQuery() {
        iQueryType = DELETE_QUERY;
    }
    private Vector  doUnionAll = new Vector ();  // 
 private Vector  unionQuerys= new Vector ();  //
 /**
     * Add a column to set by update query
     *
     * @param column   The column (field) name to set.
     * @param value    The new value to set for.
     */
    public void addUpdateColumn(RDBColumn column, Object  value) {
        _addUpdateColumn(column.getFullColumnName(), value);
    }

    /**
     * Add a column to set by update query
     *
     * @param column   The column (field) name to set.
     * @param value    The new value to set for.
     * @deprecated  Use addUpdateColumn(RDBColumn column, Object value) instead.
     */
    public void addUpdateColumn(String  column, Object  value) {
        _addUpdateColumn(column, value);
    }

    /**
     * Add a column to set by update query
     *
     * @param column   The column (field) name to set.
     * @param value    The new value to set for.
     */
    private void _addUpdateColumn(RDBColumn column, Object  value) {
        _addUpdateColumn(column.getFullColumnName(), value);
    }

    /**
     * Add a column to set by update query
     *
     * @param column   The column (field) name to set.
     * @param value    The new value to set for.
     */
    private void _addUpdateColumn(String  column, Object  value) {
        iQueryType = UPDATE_QUERY;
        storeTableNameForColumn(column);
        if (value != null) {
            updateFields.addElement(column + " = ?");
            // parms.addElement(value);
            parms.add(updateFieldPos++, value);
        } else {
            updateFields.addElement(column + " = null");
        }
    }

    private int updateFieldPos = 0;

    /**
     * addUpdateSQL
     */
    public void addUpdateSQL(RDBColumn column, String  sqlValue) {
        addUpdateSQL(column.getFullColumnName(), sqlValue);
    }

    public void addUpdateSQL(String  column, String  sqlValue) {
        iQueryType = UPDATE_QUERY;
        storeTableNameForColumn(column);
        updateFields.addElement(column + " = " + sqlValue);
    }

    /**
     * Executes the SQL INSERT, UPDATE or DELETE statement
     *
     * @return either the row count for INSERT, UPDATE or DELETE statements;
     * or 0 for SQL statements that return nothing
     * @exception SQLException if a database access error occurs
     */
    public int executeUpdate() throws SQLException  {
        return executeUpdate(null);
    }

    public int executeUpdate(DBConnection dbConn) throws SQLException  {
        int rc = 0;

        if (tableNames.size() > 1) {
            throw new SQLException ("Too many tables specified for update/delete statement.");
        }
        try {
            if (null == dbConn) {
                if (conn == null) {
                    conn = DODS.getDatabaseManager().allocateConnection();
                }
                dbConn = conn;
            }
            prepareStatement(dbConn);
            rc = stmt.executeUpdate();
        } catch (SQLException  sqlExcept) {
            if (conn != null) {
                try {
                    conn.rollback();
                } catch (SQLException  rollbackExcept) {
                    System.out.println("Failed to rollback transaction."
                                           + "\n    SQLState = " + rollbackExcept.getSQLState()
                                           + "\n    SQLError = "
                                           + rollbackExcept.getErrorCode()
                                           + "\n    SQLMsg   = " + rollbackExcept.getMessage());
                }
                if (!conn.handleException(sqlExcept)) {
                    conn = null;
                }
            } else {
                throw sqlExcept;
            }
            System.out.println("ERR870: Exception in executeSQL " + sqlExcept);
        } catch (Exception  ex) {
            System.out.println("ERR871: Exception in executeSQL " + ex);
        }
        finally {
            if (conn != null) {
                try {
                    conn.reset();
                } catch (SQLException  sqlExcept) {
                    System.out.println("ObjectIdAllocator: "
                                           + "\n  Failed to reset connection."
                                           + "\n    SQLState = " + sqlExcept.getSQLState()
                                           + "\n    SQLError = " + sqlExcept.getErrorCode()
                                           + "\n    SQLMsg   = " + sqlExcept.getMessage());
                }
                finally {
                    conn.release();
                    conn = null;
                }
            }
        }
        return rc;
    }

    // [john] Update support {end} -------------------------------------------------
 /**
     * Set the string matching details for inexact string matches
     * See dods.conf for details.
     * Code added by Chris Ryan (cryan@plugged.net.au)
     *
     * @param match_keyword  Whether the DB can use "MATCHES" or "LIKE"
     * @param wildcard  The wildcard character for string searches
     */
    public void setStringMatchDetails(String  match_keyword, String  wildcard) {
        this.likeKeyword = match_keyword;
        this.wildcard = wildcard;
    }
    // WebDocWf extension for extended wildcard support
 // All following lines have been added:
 //
 private String  singleWildcard = DEFAULT_SINGLE_WILDCARD;
    private String  singleWildcardEscape = DEFAULT_SINGLE_WILDCARD_ESCAPE;
    private String  wildcardEscape = DEFAULT_WILDCARD_ESCAPE;
    private String  wildcardEscapeClause = DEFAULT_WILDCARD_ESCAPE_CLAUSE;

    /**
     * Initialize the extended wildcard support
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param match_keyword The SQL keyword "matches" or "like"
     * @param multi_wildcard The SQL string char for 0 to n characters ("*" or "%")
     * @param single_wildcard The SQL string char for 1 character ("?" or "_")
     * @param single_wildcard_escape The SQL string char for escaping a single wildcard
     * @param multi_wildcard_escape The SQL string char for escaping a multi wildcard
     * @param wildcard_escape_clause The SQL string for defining escape charcters
     */
    public void setStringMatchDetails(String  match_keyword, String  multi_wildcard, String  single_wildcard,
                                      String  single_wildcard_escape, String  multi_wildcard_escape,
                                      String  wildcard_escape_clause) {
        this.likeKeyword = match_keyword;
        this.wildcard = multi_wildcard;
        this.singleWildcard = single_wildcard;
        this.singleWildcardEscape = single_wildcard_escape;
        this.wildcardEscape = multi_wildcard_escape;
        this.wildcardEscapeClause = wildcard_escape_clause;
    }
    private String  userWildcard = userConfigWildcard;
    private String  userSingleWildcard = userConfigSingleWildcard;
    private String  userSingleWildcardEscape = userConfigSingleWildcardEscape;
    private String  userWildcardEscape = userConfigWildcardEscape;


    private void initDefaultWildCards(){
        userWildcard = userConfigWildcard;
        userSingleWildcard = userConfigSingleWildcard;
        userSingleWildcardEscape = userConfigSingleWildcardEscape;
        userWildcardEscape = userConfigWildcardEscape;
    }


    /**
     * Change the user wildcard
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param user_wildcard   The user wildcard (normally "*")
     */
    public void setUserStringWildcard(String  user_wildcard) {
        this.userWildcard = user_wildcard;
    }

    /**
     * Change the user single wildcard
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param user_single_wildcard   The user single wildcard (normally "?")
     */
    public void setUserStringSingleWildcard(String  user_single_wildcard) {
        this.userSingleWildcard = user_single_wildcard;
    }

    /**
     * Change the user single wildcard esacpe char
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param user_single_wildcard_escape   The user single wildcard escape char (normally "?")
     */
    public void setUserStringSingleWildcardEscape(String  user_single_wildcard_escape) {
        this.userSingleWildcardEscape = user_single_wildcard_escape;
    }

    /**
     * Change the user wildcard esacpe char
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param user_wildcard_escape   The user wildcard escape char (normally "?")
     */
    public void setUserStringWildcardEscape(String  user_wildcard_escape) {
        this.userWildcardEscape = user_wildcard_escape;
    }

    /**
     * Get the user wildcard
     *
     * WebDocWf extension for extended wildcard support
     *
     * @return   The user wildcard (normally "*")
     */
    public String  getUserStringWildcard() {
        return userWildcard;
    }

    /**
     * Get the user single wildcard
     *
     * WebDocWf extension for extended wildcard support
     *
     * @return   The user single wildcard (normally "?")
     */
    public String  getUserStringSingleWildcard() {
        return userSingleWildcard;
    }

    /**
     * Get the user single wildcard escape char
     *
     * WebDocWf extension for extended wildcard support
     *
     * @return   The user single wildcard escape char (normally "?")
     */
    public String  getUserStringSingleWildcardEscape() {
        return userSingleWildcardEscape;
    }

    /**
     * Get the user wildcard escape char
     *
     * WebDocWf extension for extended wildcard support
     *
     * @return   The user wildcard escape char (normally "?")
     */
    public String  getUserStringWildcardEscape() {
        return userWildcardEscape;
    }
    private boolean userAppendWildcard = DEFAULT_USER_APPEND_WILDCARD;
    private boolean userTrimString = DEFAULT_USER_TRIM_STRING;

    /**
     * Set whether Querybuilder should append a wildcard at the end of match-strings
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param user_append_wildcard Whether Querybuilder should append a wildcard at the end of match-strings
     */
    public void setUserStringAppendWildcard(boolean user_append_wildcard) {
        this.userAppendWildcard = user_append_wildcard;
    }

    /**
     * Set whether Querybuilder should trim the match strings
     *
     * WebDocWf extension for extended wildcard support
     *
     * @param user_trim_string boolean user_trim_string Whether Querybuilder should trim the match strings
     */
    public void setUserStringTrim(boolean user_trim_string) {
        this.userTrimString = user_trim_string;
    }

    /**
     * Get whether Querybuilder should append a wildcard at the end of match-strings
     *
     * WebDocWf extension for extended wildcard support
     *
     * @return Whether Querybuilder appends a wildcard at the end of match-strings
     */
    public boolean getUserStringAppendWildcard() {
        return userAppendWildcard;
    }

    /**
     * Get whether Querybuilder trims the match strings
     *
     * WebDocWf extension for extended wildcard support
     *
     * @return Whether Querybuilder trims the match strings
     */
    public boolean getUserStringTrim() {
        return userTrimString;
    }

    //
 // end of third part WebDocWf extension for extended wildcard support
 /**
     * Add a where-clause that compares a column against a date value
     * to the SQL command.
     * Contributed by: Victor Brilon
     *
     * @param column   The column (field) name to use as a search constraint.
     * @param date     Date.
     * @param cmp_op   The comparison operator to use:  QueryBuilder.EQUAL, ...
     * @deprecated  Use addWhere(RDBColumn c1, Date value, String cmp_op) instead.
     */
    public void addWhereClause(String  column, java.sql.Date  date,
                               String  cmp_op) {
        _addWhereClause(column, date, cmp_op);
    }

    public void addWhereClause(String  column, java.sql.Timestamp  date,
                               String  cmp_op) {
        _addWhereClause(column, date, cmp_op);
    }

    public void addWhereClause(String  column, java.sql.Time  date,
                               String  cmp_op) {
        _addWhereClause(column, date, cmp_op);
    }

    /**
     * Add another field to the ORDER BY list.
     * NOTE: is used by SourceGenerator_Query and writeMemberStuff.template.
     *
     * This method added by Chris Ryan (cryan@plugged.net.au) to
     * allow easier access to the ORDER BY part of the SQL query.
     *
     * @param column     The column to order by
     * @param direction  ASCENDING or DESCENDING order
     */
    public void addOrderByColumn(String  column, String  direction) {
        orderFields.addElement(column);
        orderDirections.addElement(direction);
        preventPrimaryKeySelect = true;
    }

    /**
     * Add another field to the ORDER BY list.
     * NOTE: is used by SourceGenerator_Query and writeMemberStuff.template.
     *
     * This method added by Chris Ryan (cryan@plugged.net.au) to
     * allow easier access to the ORDER BY part of the SQL query.
     *
     * @param column     The column to order by
     * @param direction  ASCENDING or DESCENDING order
     */
    public void addOrderByColumn(RDBColumn column, String  direction) {
        orderFields.addElement(column.getFullColumnName());
        orderDirections.addElement(direction);
        preventPrimaryKeySelect = true;
    }

    public void setDatabaseVendor(String  vendor) throws DatabaseManagerException {
        String  str = Common.getDodsConfProperty("LikeKeyword", vendor);

        if (str != null) {
            likeKeyword = str;
        }
        str = Common.getDodsConfProperty("Wildcard", vendor);
        if (str != null) {
            wildcard = str;
        }
        str = Common.getDodsConfProperty("SingleWildcard", vendor);
        if (str != null) {
            singleWildcard = str;
        }
        str = Common.getDodsConfProperty("SingleWildcardEscape", vendor);
        if (str != null) {
            singleWildcardEscape = str;
        }
        str = Common.getDodsConfProperty("WildcardEscape", vendor);
        if (str != null) {
            wildcardEscape = str;
        }
        str = Common.getDodsConfProperty("WildcardEscapeClause", vendor);
        if (str!=null) {
            if (str.equalsIgnoreCase("none")){
                wildcardEscapeClause = "";
            }else{
                wildcardEscapeClause = str;
            }
        }
    }

    /**
     * Set database vendor for the default logical database.
     */
    public void setDatabaseVendor() {
        try {
            String  vendor = DODS.getDatabaseManager().logicalDatabaseType();
            setDatabaseVendor(vendor);
        } catch (Exception  e) {
            e.printStackTrace();
        }
    }

    /**
     * Set database name to QueryBuilder.
     * @param dbName     Database name
     */
    public void setDatabaseName(String  dbName) {
        databaseName = dbName;
    }

    /**
     * Get database name.
     * @return String     Database name
     */
    public String  getDatabaseName() {
        if (databaseName == null) {
            try {
                databaseName = DODS.getDatabaseManager().getDefaultDB();
            } catch (Exception  e) {}
        }
        return databaseName;
    }

    static private final int op_EQUAL = 1;
    static private final int op_NOT_EQUAL = 2;
    static private final int op_LESS_THAN = 3;
    static private final int op_LESS_THAN_OR_EQUAL = 4;
    static private final int op_GREATER_THAN = 5;
    static private final int op_GREATER_THAN_OR_EQUAL = 6;
    static private final int op_IS_NULL = 7;
    static private final int op_IS_NOT_NULL = 8;
    static private final int op_CASE_SENSITIVE_CONTAINS = 9;
    static private final int op_CASE_INSENSITIVE_CONTAINS = 10;
    static private final int op_CASE_SENSITIVE_STARTS_WITH = 11;
    static private final int op_CASE_INSENSITIVE_STARTS_WITH = 12;
    static private final int op_CASE_SENSITIVE_ENDS_WITH = 13;
    static private final int op_CASE_INSENSITIVE_ENDS_WITH = 14;
    static private final int op_CASE_INSENSITIVE_EQUAL = 15;
    static Hashtable  cmp_ops = new Hashtable ();
    static {
        cmp_ops.put(EQUAL, new Integer (op_EQUAL));
        cmp_ops.put(NOT_EQUAL, new Integer (op_NOT_EQUAL));
        cmp_ops.put(LESS_THAN, new Integer (op_LESS_THAN));
        cmp_ops.put(LESS_THAN_OR_EQUAL, new Integer (op_LESS_THAN_OR_EQUAL));
        cmp_ops.put(GREATER_THAN, new Integer (op_GREATER_THAN));
        cmp_ops.put(GREATER_THAN_OR_EQUAL, new Integer (op_GREATER_THAN_OR_EQUAL));
        cmp_ops.put(IS_NULL, new Integer (op_IS_NULL));
        cmp_ops.put(IS_NOT_NULL, new Integer (op_IS_NOT_NULL));
        cmp_ops.put(CASE_SENSITIVE_CONTAINS,
                    new Integer (op_CASE_SENSITIVE_CONTAINS));
        cmp_ops.put(CASE_INSENSITIVE_CONTAINS,
                    new Integer (op_CASE_INSENSITIVE_CONTAINS));
        cmp_ops.put(CASE_SENSITIVE_STARTS_WITH,
                    new Integer (op_CASE_SENSITIVE_STARTS_WITH));
        cmp_ops.put(CASE_INSENSITIVE_STARTS_WITH,
                    new Integer (op_CASE_INSENSITIVE_STARTS_WITH));
        cmp_ops.put(CASE_SENSITIVE_ENDS_WITH,
                    new Integer (op_CASE_SENSITIVE_ENDS_WITH));
        cmp_ops.put(CASE_INSENSITIVE_ENDS_WITH,
                    new Integer (op_CASE_INSENSITIVE_ENDS_WITH));
        cmp_ops.put(CASE_INSENSITIVE_EQUAL,
                    new Integer (op_CASE_INSENSITIVE_EQUAL));
    }

    /**
     * Wrapper for compare() method that takes Objects for arguments.
     * This method takes doubles, and so will handle
     * shorts, ints, longs, floats, doubles, but not booleans.
     *
     * @return true if a compares to b according to cmp_op
     * @param a the DO member value
     * @param b the value to compare against
     * @exception QueryException if cmp_op is not a valid comparison operator.
     * @see #compare( Object a, Object b, String cmp_op )
     * author Jay Gunter
     */
    static public boolean compare(double a, double b, String  cmp_op)
        throws QueryException {
        return compare(new Double (a), new Double (b), cmp_op);
    }

    /**
     * Wrapper for compare() method that takes Objects for arguments.
     * This method takes booleans.
     *
     * @return true if a compares to b according to cmp_op
     * @param a the DO member value
     * @param b the value to compare against
     * @exception QueryException if cmp_op is not a valid comparison operator.
     * @see #compare( Object a, Object b, String cmp_op )
     * author Jay Gunter
     */
    static public boolean compare(boolean a, boolean b, String  cmp_op)
        throws QueryException {
        return compare(new Double (a ? 1 : 0), new Double (b ? 1 : 0), cmp_op);
    }

    /**
     * Wrapper for compare() method that takes Objects for arguments.
     * This method takes two arguments - first double, and second Object.
     *
     * @return true if a compares to b according to cmp_op
     * @param a the DO member value
     * @param b the value to compare against
     * @exception QueryException if cmp_op is not a valid comparison operator.
     * @see #compare( Object a, Object b, String cmp_op )
     */
    static public boolean compare(double a, Object  b, String  cmp_op)
        throws QueryException {
        return compare(new Double (a), b, cmp_op);
    }

    /**
     * Wrapper for compare() method that takes Objects for arguments.
     * This method takes two arguments - first boolean, and second Object.
     *
     * @return true if a compares to b according to cmp_op
     * @param a the DO member value
     * @param b the value to compare against
     * @exception QueryException if cmp_op is not a valid comparison operator.
     * @see #compare( Object a, Object b, String cmp_op )
     */
    static public boolean compare(boolean a, Object  b, String  cmp_op)
        throws QueryException {
        return compare(new Double (a ? 1 : 0), b, cmp_op);
    }

    /**
     * Compare two values according to a QueryBuilder comparison operator.
     * This method has some general applicability.
     * It is handy for writing sorting algorithms, etc.
     *
     * The Query classes generated by DODS use this method
     * to prune undesired values from the cache held by a DO class.
     *
     * @return true if a compares to b according to cmp_op
     * @param a the DO member value
     * @param b the value to compare against
     * @exception QueryException if cmp_op is not a valid comparison operator.
     * author Jay Gunter
     */
    static public boolean compare(Object  a, Object  b, String  cmp_op)
        throws QueryException {
        Integer  cmp = (Integer ) cmp_ops.get(cmp_op);

        if (null == cmp) {
            throw new QueryException("Unrecognized comparison operator: "
                                         + cmp_op);
        }
        int op = cmp.intValue();

        switch (op) {
            case op_IS_NULL:
                return null == b;

            case op_IS_NOT_NULL:
                return null != b;

            case op_EQUAL:
                if (null == a && null == b) {
                    return true;
                }
                if (null == a || null == b) {
                    return false;
                }
                byte[] z = {};

                if (a.getClass() == z.getClass()) {
                    byte[] ab = (byte[]) a;
                    byte[] bb = (byte[]) b;

                    if (ab.length != bb.length) {
                        return false;
                    }
                    for (int i = 0; i < ab.length; i++) {
                        if (ab[i] != bb[i]) {
                            return false;
                        }
                    }
                    return true;
                } else if (a instanceof GenericDO) {
                    String  aoid = ((GenericDO) a).get_OId().toString();
                    String  boid = ((GenericDO) b).get_OId().toString();
                    boolean equals = aoid.equals(boid);

                    // WebDocWf fix for cached lazyloading objects
 // lazyloading objects are only put in the cache if they are loaded
 // so there may be multiple instances for one row in memory
 // the following lines have been put under comments
 // if ( equals && a != b )
 // System.err.println(
 // "\n\nDATA OBJECT WARNING: Two DOs with the same OID: "
 // + aoid );
 // end of WebDocWf fix for cached lazyloading objects
 return equals;
                } else if (a instanceof BigDecimal &&b instanceof GenericDO) {
                    return ((GenericDO)b).get_OId().toString().equals(a.toString());
                } else if (a instanceof CoreDataStruct) {
                    String  aoid = ((CoreDataStruct) a).get_OId().toString();
                    String  boid = ((CoreDataStruct) b).get_OId().toString();
                    boolean equals = aoid.equals(boid);

                    // WebDocWf fix for cached lazyloading objects
 // lazyloading objects are only put in the cache if they are loaded
 // so there may be multiple instances for one row in memory
 // the following lines have been put under comments
 // if ( equals && a != b )
 // System.err.println(
 // "\n\nDATA OBJECT WARNING: Two DOs with the same OID: "
 // + aoid );
 // end of WebDocWf fix for cached lazyloading objects
 return equals;
                } else if (a instanceof BigDecimal &&b instanceof CoreDataStruct) {
                    return ((CoreDataStruct)b).get_OId().toString().equals(a.toString());
                }

                else {
                    return a.equals(b);
                }

            case op_NOT_EQUAL:
                return !compare(a, b, EQUAL);

            case op_GREATER_THAN:
                return !compare(a, b, LESS_THAN_OR_EQUAL);

            case op_GREATER_THAN_OR_EQUAL:
                return !compare(a, b, LESS_THAN);

            case op_LESS_THAN:
            case op_LESS_THAN_OR_EQUAL:
                if (null == a || null == b) {
                    return false;
                }
                if (a instanceof String ) {
                    String  aStr = (String ) a;
                    String  bStr = null;

                    if (!(b instanceof String )) {
                        bStr = b.toString();
                    }
                    int x = aStr.compareTo(bStr);

                    return (-1 == x || (0 == x && op_LESS_THAN_OR_EQUAL == op));
                } else if (a instanceof java.util.Date ) {
                    if (!(b instanceof java.util.Date )) {
                        throw new QueryException("Second arg for " + cmp_op
                                                     + " is not a Date");
                    }
                    java.util.Date  ad = (java.util.Date ) a;
                    java.util.Date  bd = (java.util.Date ) b;

                    return (ad.before(bd)
                                || (ad.equals(bd) && op_LESS_THAN_OR_EQUAL == op));
                } else if (a instanceof BigDecimal ) {
                    if (!(b instanceof BigDecimal )) {
                        throw new QueryException("First arg for " + cmp_op
                                                     + " is BigDecimal, " + "but second arg is not.");
                    }
                    int x = ((BigDecimal ) a).compareTo((BigDecimal )b);

                    return (-1 == x || (0 == x && op_LESS_THAN_OR_EQUAL == op));
                } else if (a instanceof Number ) {
                    if (!(b instanceof Number )) {
                        throw new QueryException("First arg for " + cmp_op
                                                     + " is Number, " + "but second arg is not.");
                    }
                    double ad = ((Number ) a).doubleValue();
                    double bd = ((Number ) b).doubleValue();

                    return (ad < bd || (ad == bd && op_LESS_THAN_OR_EQUAL == op));
                } else {
                    throw new QueryException("Comparison operator " + cmp_op
                                                 + " not supported for objects of type "
                                                 + a.getClass().getName());
                }

            case op_CASE_INSENSITIVE_EQUAL:
                if (null == a && null == b) {
                    return true;
                }
                if (null == a || null == b) {
                    return false;
                }
                String  aStr = (String ) a;
                String  bStr = (String ) b;
                return aStr.equalsIgnoreCase(bStr);

            case op_CASE_SENSITIVE_CONTAINS:
            case op_CASE_INSENSITIVE_CONTAINS:
            case op_CASE_SENSITIVE_STARTS_WITH:
            case op_CASE_INSENSITIVE_STARTS_WITH:
            case op_CASE_SENSITIVE_ENDS_WITH:
            case op_CASE_INSENSITIVE_ENDS_WITH:
                if (!(a instanceof String )) {
                    throw new QueryException("First arg for " + cmp_op
                                                 + " is not a string.");
                }
                if (!(b instanceof String )) {
                    throw new QueryException("Second arg for " + cmp_op
                                                 + " is not a string.");
                }
                String  as = (String ) a;
                String  bs = (String ) b;

                if (-1 != cmp_op.indexOf('x')) {
                    as = as.toLowerCase();
                    bs = bs.toLowerCase();
                }
                switch (op) {
                    case op_CASE_SENSITIVE_CONTAINS:
                    case op_CASE_INSENSITIVE_CONTAINS:
                        return -1 != as.indexOf(bs);

                    case op_CASE_SENSITIVE_STARTS_WITH:
                    case op_CASE_INSENSITIVE_STARTS_WITH:
                        return as.startsWith(bs);

                    case op_CASE_SENSITIVE_ENDS_WITH:
                    case op_CASE_INSENSITIVE_ENDS_WITH:
                        return as.endsWith(bs);
                }
                throw new QueryException("QueryBuilder.compare bug in CASE_ processing");

            default:
                throw new QueryException("Unchecked comparison operator: " + cmp_op);
        }
        // return false; // should never get here
    }

    // WebDocWf extension for row counters
 // All following lines have been added:
 /**
     * Set the Query select clause (e.g. &quot;*&quot;) to something different (e.g. &quot;count(*)&quot;)
     * Hack for implementing row counters
     *
     * @param newClause The new select clause string
     */
    public void setSelectClause(String  newClause) {
        selectClause = newClause;
    }

    /**
     * Get the Query select clause string
     * Hack for implementing row counters
     *
     * @return The current select clause string
     */
    public String  getSelectClause() {
        return selectClause;
    }

    // end of WebDocWf extension for row counters
 // WebDocWf extension cursor-handling
 /**
     * sets the fetchsize for this query, overwrites the applicationfetchsize
     * @param iCurrentFetchSizeIn integer representing the fetchsize for this query
     */
    public void setCurrentFetchSize(int iCurrentFetchSizeIn) {
        iCurrentFetchSize = iCurrentFetchSizeIn;
    }

    // WebDocWf extension cursor-handling
 /**
     * reads the current fetchsize for this query
     * @return the current fetchsize; if -1 the no fetchsize is defined, applicationfetchsize will be use if defined
     */
    public int getCurrentFetchSize() {
        return iCurrentFetchSize;
    }

    /**
     * sets the default fetchsize
     * @param iDefaultFetchSizeIn integer representing the fetchsize for this query
     */
    static public void setDefaultFetchSize(int iDefaultFetchSizeIn) {
        iDefaultFetchSize = iDefaultFetchSizeIn;
    }

    // WebDocWf extension cursor-handling
 /**
     * reads the default fetchsize
     * @return the current fetchsize; if -1 the no fetchsize is defined, applicationfetchsize will be use if defined
     */
    static public int getDefaultFetchSize() {
        return iDefaultFetchSize;
    }

    // WebDocWf extension cursor-handling end
 /**
     * sets the default queryTimeout
     * @param iQueryTimeout integer representing the queryTimeout for this query
     */
    static public void setDefaultQueryTimeout(int iQueryTimeout) {
        iDefaultQueryTimeout = iQueryTimeout;
    }

    /**
     * gets the default queryTimeout
     * @return the current queryTimeout;
     */
    static public int getDefaultQueryTimeout() {
        return iDefaultQueryTimeout;
    }

    /**
     * reads the current queryTimeout for this query
     * @return the current queryTimeout;
     */
    public int getCurrentQueryTimeout() {
        return iCurrentQueryTimeout;
    }

    /**
     * set the current queryTimeout for this query
     * @return the current queryTimeout;
     */
    public void setCurrentQueryTimeout(int iQueryTimeoutIn) {
        iCurrentQueryTimeout = iQueryTimeoutIn;
    }

    public static synchronized String  getCountX(){
        if (countX<Long.MAX_VALUE){
            countX++;
        }else{
            countX=0;
        }
        return "cursorID"+Long.toString(countX);
    }


    public static long countX = 0;


    private boolean doSetFetchSize(DBConnection conn){
        boolean disableFetchSizeWithMaxRows;
        try {
            disableFetchSizeWithMaxRows = ((StandardLogicalDatabase) DODS
                                               .getDatabaseManager()
                                               .findLogicalDatabase(conn.getDatabaseName()))
                .getDisableFetchSizeWithMaxRows();
        } catch (DatabaseManagerException e) {
            DODS.getLogChannel().write(Logger.DEBUG,"Error unknown logical database. Using default value for 'DisableFetchSizeWithMaxRows' parameter");
            disableFetchSizeWithMaxRows = StandardLogicalDatabase.DEFAULT_DISABLE_FETCH_SIZE_WITH_MAX_ROWS;
        }
        if (disableFetchSizeWithMaxRows & (maxRows>0)){
            return false;
        }else{
            return true;
        }
    }


    private boolean doCursorName(DBConnection conn){

        //if doSetFetchSize is false, do not perform check for cursor name
 if( !doSetFetchSize(conn) )
            return false;

        boolean cursorName;
        try {
            cursorName =((StandardLogicalDatabase) DODS
                             .getDatabaseManager()
                             .findLogicalDatabase(conn.getDatabaseName()))
                .getUseCursorName();
        } catch (DatabaseManagerException e) {
            DODS.getLogChannel().write(Logger.DEBUG,"Error unknown logical database. Using default value for 'UseCursorName' parameter");
            cursorName = StandardLogicalDatabase.DEFAULT_USE_CURSOR_NAME;
        }
        return cursorName;
    }

    //sinisa 15.06.2004

    /**
     * set the current cursor type - overrides default value from dbVendorConf file.
     * @param resultSetType a result set type; one of ResultSet.TYPE_FORWARD_ONLY, ResultSet.TYPE_SCROLL_INSENSITIVE, or ResultSet.TYPE_SCROLL_SENSITIVE.
     * @param resultSetConcurrency a concurrency type; one of ResultSet.CONCUR_READ_ONLY or ResultSet.CONCUR_UPDATABLE.
     * @return the current queryTimeout;
     */
    public void setCursorType(int resultSetType, int resultSetConcurrency) {

        iResultSetConcurrency = resultSetConcurrency;
        iResultSetType = resultSetType;
    }
    
} // end QueryBuilder class
