
/**
 * Title:        SecureDO<p>
 * Description:  SecureDO  is a superclass for all DODS DO's with access rights<p>
 * Copyright:    Copyright (c) Together Teamlösungen EDV-Dienstleistungen GmbH. under LGPL<p>
 * Company:      Together Teamlösungen EDV-Dienstleistungen GmbH.<p>
 * @author Alfred Madl (a.madl@together.at)
 * @version 1.0
 */
package org.webdocwf.dods.access;

import java.sql.ResultSet ;
import java.sql.SQLException ;
import com.lutris.appserver.server.sql.DBRowUpdateException;
import com.lutris.appserver.server.sql.DBTransaction;
import com.lutris.appserver.server.sql.DatabaseManagerException;
import com.lutris.appserver.server.sql.ObjectId;
import com.lutris.appserver.server.sql.ObjectIdException;
import com.lutris.dods.builder.generator.dataobject.GenericDO;
import com.lutris.dods.builder.generator.query.DataObjectException;
import com.lutris.dods.builder.generator.query.QueryException;
import com.lutris.dods.builder.generator.query.RefAssertionException;

public abstract class SecureDO extends GenericDO {

    /**
     * Deletes the DO from its table.
     *
     * @exception com.lutris.appserver.server.sql.DatabaseManagerException if a Transaction can not be created.
     * @exception RefAssertionException thrown by okTo method.
     * @exception java.sql.SQLException if any SQL errors occur.
     */
    public abstract void delete() throws SQLException , DatabaseManagerException, DataObjectException, RefAssertionException, DBRowUpdateException, QueryException;

    /**
     * Deletes the DO from its table.
     * The transaction is likely provided by the delete() method of another DO
     * which references this DO.
     *
     * @param dbt The transaction object to use for this operation.
     * @exception com.lutris.appserver.server.sql.DatabaseManagerException if a Transaction can not be created.
     * @exception com.lutris.appserver.server.sql.DBRowUpdateException if a version error occurs.
     * @exception RefAssertionException thrown by okTo method.
     * @exception java.sql.SQLException if any SQL errors occur.
     */
    public abstract void delete(DBTransaction dbt) throws SQLException , DatabaseManagerException, DataObjectException, RefAssertionException, DBRowUpdateException, QueryException;

    /**
     * Load the actual DO data if necessary.
     * Called by get/set methods.
     *
     * @exception DataObjectException If a data access error occurs.
     */
    protected abstract void checkLoad() throws DataObjectException;
    
    public abstract String   getHandle() throws DatabaseManagerException;

    public abstract String   get_Handle() throws DatabaseManagerException;

    public abstract boolean hasMatchingHandle(String  handle);

    /**
     * isReadOnly()
     * Returns true if the data for this object has been marked read-only.
     */
    public abstract boolean isReadOnly();

    /**
     * makeReadOnly()
     * Mark the object as readonly
     * 
     * WebDocWf extension
     * 
     */
    public abstract void makeReadOnly();

    /**
     * Mark the object as readwrite
     * 
     * WebDocWf extension
     * 
     */
    public abstract void makeReadWrite();

    public SecureDO() throws DatabaseManagerException, ObjectIdException {
        super();
    }

    public SecureDO(String  dbName) throws DatabaseManagerException, ObjectIdException {
        super(dbName);
    }
  
    public SecureDO(boolean isView) throws ObjectIdException, DatabaseManagerException {
        super(isView);
    }

    public SecureDO(String  dbName, boolean isView) throws ObjectIdException, DatabaseManagerException {
        super(dbName, isView);
    }

    public SecureDO(ObjectId id) throws ObjectIdException, DatabaseManagerException {
        super(id);
    }

    public SecureDO(String  dbName, ObjectId id) throws ObjectIdException, DatabaseManagerException {
        super(dbName, id);
    }

    public SecureDO(ResultSet  rs) throws SQLException , ObjectIdException, DatabaseManagerException {
        super(rs);
    }

    public SecureDO(String  dbName, ResultSet  rs) throws SQLException , ObjectIdException, DatabaseManagerException {
        super(dbName, rs);
    }

    /**
     * Deletes the DO from its table.
     *
     * @param usr The user for security checks
     *
     * @exception com.lutris.appserver.server.sql.DatabaseManagerException if a Transaction can not be created.
     * @exception RefAssertionException thrown by okTo method.
     * @exception java.sql.SQLException if any SQL errors occur.
     * @exception AccessException
     *   The user is not allowed to delete the DO
     *
     * This is a WebDocWf extension for DODS row instance security
     *
     */
    public void delete(org.webdocwf.dods.access.User usr) 
        throws SQLException , DatabaseManagerException, DataObjectException, RefAssertionException, DBRowUpdateException, QueryException, AccessException {
        assertDODeleteAccess(usr);
        delete();
    }

    /**
     * Ensure that the given user is allowed to delete the DO
     *
     * @param usr The user for security checks
     *
     * @exception AccessException
     *   The user is not allowed to delete the DO
     *
     * This is a WebDocWf extension for DODS row instance security
     *
     */
    public void assertDODeleteAccess(org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (!hasDODeleteAccess(usr)) {
            throw new AccessRightException("No access !", usr, "Delete", this,
                    null, null, null, null, null, null);
        }
    }

    /**
     * Check if the given user is allowed to delete the DO
     *
     * @param usr The user for security checks
     * @return Whether the user is allowed to delete this DO
     *
     * @exception AccessEvalException
     *   Error during access evaluation
     *
     * This is a WebDocWf extension for DODS row instance security
     *
     */
    public boolean hasDODeleteAccess(org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDODeleteAccess/checkLoad !",
                    e, usr, "Delete", this, null, null, null, null, null, null);
        }   
        return usr.hasDODeleteAccess(this);
    }

    /**
     * Deletes the DO from its table.
     *
     * @param usr The user for security checks
     *
     * @param dbt The transaction object to use for this operation.
     * @exception com.lutris.appserver.server.sql.DatabaseManagerException if a Transaction can not be created.
     * @exception RefAssertionException thrown by okTo method.
     * @exception java.sql.SQLException if any SQL errors occur.
     * @exception AccessException
     *   The user is not allowed to delete the DO
     *
     * This is a WebDocWf extension for DODS row instance security
     *
     */
    public void delete(DBTransaction dbt, org.webdocwf.dods.access.User usr) 
        throws SQLException , DatabaseManagerException, DataObjectException, RefAssertionException, DBRowUpdateException, QueryException, AccessException {
        assertDODeleteAccess(usr);
        delete(dbt);
    }

    /**
     * Ensure that the given user is allowed to copy the DO
     *
     * @param usr The user for security check
     *
     * @exception AccessException
     *   The user is not allowed to make a copy
     *
     * WebDocWf extension
     *
     */
    public void assertDOCopyAccess(org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (!hasDOCopyAccess(usr)) {
            throw new AccessRightException("No access !", usr, "Copy", this,
                    null, null, null, null, null, null);
        }
    }

    /**
     * Check whether the given user is allowed to copy the DO
     *
     * @param usr The user for security check
     * @return Whether the given user is allowed to copy the DO
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOCopyAccess(org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoCopyAccess/checkLoad !",
                    e, usr, "Copy", this, null, null, null, null, null, null);
        }
        return usr.hasDOCopyAccess(this);
    }

    /**
     * The get_Handle() method is used
     * to set the value for each GUI option,
     * and the hasMatchingHandle()
     * methods are used to lookup the Data Object when the selection has
     * been made.
     *
     * @param usr The user for security check
     * @return id of this DO as a string
     *
     * @exception DatabaseManagerException
     *   If a connection to the database cannot be established, etc.
     * @exception AccessException
     *   The user is not allowed to read the object existance
     *
     * WebDocWf extension
     *
     */
    public String   get_Handle(org.webdocwf.dods.access.User usr)
        throws DatabaseManagerException, AccessException {
        assertDOGetAccess(usr);
        return getHandle();
    }

    /**
     * The getHandle() method is used
     * to set the value for each GUI option,
     * and the hasMatchingHandle()
     * methods are used to lookup the Data Object when the selection has
     * been made.
     *
     * @param usr The user for security check
     * @return id of this DO as a string
     *
     * @exception DatabaseManagerException
     *   If a connection to the database cannot be established, etc.
     * @exception AccessException
     *   The user is not allowed to read the object existance
     *
     * WebDocWf extension
     * @deprecated Use get_Handle() instead.        
     */
    public String   getHandle(org.webdocwf.dods.access.User usr)
        throws DatabaseManagerException, AccessException {
        return get_Handle();
    }

    /**
     * hasMatchingHandle
     *
     * @param handle
     *   String version of DO id
     * @param usr The user for security check
     * @return boolean
     *   True if the string version of the id of this DO matches passed handle
     * @exception AccessException
     *   The user is not allowed to read the object existance
     *
     * WebDocWf extension
     *
     */
    public boolean hasMatchingHandle(String  handle, org.webdocwf.dods.access.User usr) 
        throws AccessException {
        assertDOGetAccess(usr);
        return hasMatchingHandle(handle);
    }

    // WebDocWf extension for DODS row instance security
 // The following lines have been added:
 /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, Object  value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null,
                    "Object", null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, boolean value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null,
                    "boolean", null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, byte value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null, "byte",
                    null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, short value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null, "short",
                    null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, int value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null, "int",
                    null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, long value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null, "long",
                    null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, float value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null, "float",
                    null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, double value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null,
                    "double", null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to read
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param value The current value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the attribute
     *
     * WebDocWf extension
     *
     */
    public boolean hasDOGetAttrAccess(String  attrName, byte[] value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAttrAccess/checkLoad !",
                    e, usr, "GetAttr", this, null, attrName, null, null,
                    "byte[]", null);
        }
        return usr.hasDOGetAttrAccess(this, attrName, value);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, Object  oldValue, Object  newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, boolean oldValue, boolean newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, byte oldValue, byte newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, short oldValue, short newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, int oldValue, int newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, long oldValue, long newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, float oldValue, float newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, double oldValue, double newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Check whether the given user is allowed to update
     * the attribute and the value
     *
     * @param attrName The name of the attribute
     * @param oldValue The current value of the attribute
     * @param newValue The new value of the attribute
     * @param usr The user for security  check
     * @return Whether the given user is allowed to update the attribute
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOSetAttrAccess(String  attrName, byte[] oldValue, byte[] newValue, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return usr.hasDOSetAttrAccess(this, attrName, oldValue, newValue);
    }

    /**
     * Ensure that the given user is allowed to read
     * the object in a given pointer
     *
     * @param attrName The name of the attribute
     * @param value The object in the pointer
     * @param usr The user for security  check
     *
     * @exception AccessException
     *    The user is not allowed to read the object
     *
     * WebDocWf extension
     *
     */
    public void assertDOGetDOValueAccess(String  attrName, SecureDO value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (value != null) {
            if (!value.hasDOGetAccess(usr)) {
                throw new AccessRightException("No access !", usr, "GETDOValue",
                        this, null, attrName, null, null, null, null);
            }
        }
    }

    /**
     * Check whether the given user is allowed to read
     * the object in a given pointer
     *
     * @param value The object in the pointer
     * @param usr The user for security  check
     * @return Whether the given user is allowed to read the object
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(SecureDO value, org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        return value.hasDOGetAccess(usr);
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(Object  value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(boolean value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(byte value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(short value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(int value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(long value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(float value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(double value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected boolean hasDOGetDOValueAccess(byte[] value, org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {    
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
        return true; 
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, Object  value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, boolean value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, byte value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, short value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, int value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, long value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, float value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, double value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Dummy method for generated datatypes
     *
     * WebDocWf extension
     *
     */
    protected void assertDOGetDOValueAccess(String  attrName, byte[] value, org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (false) {
            throw new AccessEvalException("Should never occur !");
        }
    }

    /**
     * Check whether the given user is allowed to
     * find the object using a query
     *
     * @param usr The user for security check
     * @return Whether the given user is allowed to find the object
     *
     * WebDocWf extension
     *
     */
    public boolean hasQueryFindAccess(org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {
        if (usr == null) {
            return false;
        }
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasQueryFindAccess/checkLoad !",
                    e, usr, "Query", this, null, null, null, null, null, null);
        }
        return usr.hasQueryFindAccess(this);
    }

    // end of WebDocWf extension for DODS row instance security
 // WebDocWf enhancement for public versionnumber
 // The following line has been added:
 /**
     * @deprecated Use get_Version()
     * @return this object's version.
     */
    public int getVersion() {
        return get_Version();
    }

    /**
     * Returns this object's version.
     * 
     * @return this object's version.
     * 
     * WebDocWf extension
     * 
     */
    public int get_Version() {
        return super.get_Version();
    }

    /**
     * Returns this object's version.
     * 
     * @param usr The user for security check
     * @return this object's version.
     * 
     * WebDocWf extension
     * 
     */
    public int get_Version(org.webdocwf.dods.access.User usr) 
        throws AccessException {
        assertDOGetVersionAccess(usr);
        return get_Version();
    }

    /**
     * Ensure that the given user is allowed to access the version number
     * 
     * @param usr The user for security check
     * 
     * WebDocWf extension
     * 
     */
    public void assertDOGetVersionAccess(org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (!hasDOGetVersionAccess(usr)) {
            throw new AccessRightException("No access !", usr, "GetVersion",
                    this, null, null, null, null, null, null);
        }
    }

    /**
     * Check whether the given user is allowed to access the version number
     * 
     * @param usr The user for security check
     * @return Whether the given user is allowed to access the version number
     * 
     * WebDocWf extension
     * 
     */
    public boolean hasDOGetVersionAccess(org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDOGetVersionAccess/checkLoad !",
                    e, usr, "GetVersion", this, null, null, null, null, null,
                    null);
        }
        return usr.hasDOGetVersionAccess(this);
    }

    /**
     * isReadOnly()
     * Returns true if the data for this object has been marked read-only.
     * 
     * @param usr The user for security check
     * @return Whether the data for this object has been marked read-only.
     * 
     * WebDocWf extension
     * 
     */
    public boolean isReadOnly(org.webdocwf.dods.access.User usr) 
        throws AccessException {
        assertDOIsReadOnlyAccess(usr);
        return isReadOnly();
    }

    /**
     * Ensure that the given user is allowed to read the readonly flag
     * 
     * @param usr The user for security check
     * 
     * WebDocWf extension
     * 
     */
    public void assertDOIsReadOnlyAccess(org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (!hasDOIsReadOnlyAccess(usr)) {
            throw new AccessRightException("No access !", usr, "IsReadOnly",
                    this, null, null, null, null, null, null);
        }
    }

    /**
     * Check whether the given user is allowed to  read the readonly flag
     * 
     * @param usr The user for security check
     * @return Whether the given user is allowed to  read the readonly flag
     * 
     * WebDocWf extension
     * 
     */
    public boolean hasDOIsReadOnlyAccess(org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDOIsReadOnlyAccess/checkLoad !",
                    e, usr, "IsReadOnly", this, null, null, null, null, null,
                    null);
        }
        return usr.hasDOIsReadOnlyAccess(this);
    }

    /**
     * Ensure that the given user is allowed to read the DO existance
     *
     * @param usr The user for security check
     *
     * @exception AccessException
     *   The user is not allowed to read the instance existance
     *   
     * WebDocWf extension
     *   
     */
    public void assertDOGetAccess(org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (!hasDOGetAccess(usr)) {
            throw new AccessRightException("No access", usr, "Get", this, null,
                    null, null, null, null, null);
        }
    }

    /**
     * Check whether the given user is allowed to read the DO existance
     *
     * @param usr The user for security check
     * @return Whether the given user is allowed to read the DO existance
     *   
     * WebDocWf extension
     *   
     */
    public boolean hasDOGetAccess(org.webdocwf.dods.access.User usr) 
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDoGetAccess/checkLoad !",
                    e, usr, "Get", this, null, null, null, null, null, null);
        }
        return usr.hasDOGetAccess(this);
    }

    // end of WebDocWf extension for DODS row instance security
 // WebDocWf extension for writeable fully cached obejcts
 /**
     * makeReadOnly()
     * Mark the object as readonly
     * 
     * @param usr The user for security check
     * 
     * WebDocWf extension
     * 
     */
    public void makeReadOnly(org.webdocwf.dods.access.User usr)
        throws AccessException {
        assertDOMakeReadOnlyAccess(usr);
        makeReadOnly();
    }

    /**
     * Ensure that the given user is allowed to set the object readonly
     * 
     * @param usr The user for security check
     * 
     * WebDocWf extension
     * 
     */
    public void assertDOMakeReadOnlyAccess(org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (!hasDOMakeReadOnlyAccess(usr)) {
            throw new AccessRightException("No access", usr, "MakeReadOnly",
                    this, null, null, null, null, null, null);
        }
    }

    /**
     * Check whether the given user is allowed to set the object readonly
     * 
     * @param usr The user for security check
     * @return Whether the given user is allowed to set the object readonly
     * 
     * WebDocWf extension
     * 
     */
    public boolean hasDOMakeReadOnlyAccess(org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDOMakeReadOnlyAccess/checkLoad !",
                    e, usr, "MakeReadOnly", this, null, null, null, null, null,
                    null);
        }
        return usr.hasDOMakeReadOnlyAccess(this);
    }

    /**
     * Mark the object as readwrite
     * 
     * @param usr The user for security check
     * 
     * WebDocWf extension
     * 
     */
    public void makeReadWrite(org.webdocwf.dods.access.User usr)
        throws AccessException {
        assertDOMakeReadWriteAccess(usr);
        makeReadWrite();
    }

    /**
     * Ensure that the given user is allowed to set the object readwrite
     * 
     * @param usr The user for security check
     * 
     * WebDocWf extension
     * 
     */
    public void assertDOMakeReadWriteAccess(org.webdocwf.dods.access.User usr)
        throws AccessException {
        if (!hasDOMakeReadWriteAccess(usr)) {
            throw new AccessRightException("No access", usr, "MakeReadWrite",
                    this, null, null, null, null, null, null);
        }
    }

    /**
     * Check whether the given user is allowed to set the object readwrite
     * 
     * @param usr The user for security check
     * @return Whether the given user is allowed to set the object readwrite
     * 
     * WebDocWf extension
     * 
     */
    public boolean hasDOMakeReadWriteAccess(org.webdocwf.dods.access.User usr)
        throws AccessEvalException {
        try {
            checkLoad();
        } catch (Exception  e) {
            throw new AccessEvalException("Error in hasDOMakeReadWriteAccess/checkLoad !",
                    e, usr, "MakeReadWrite", this, null, null, null, null, null,
                    null);
        }
        return usr.hasDOMakeReadWriteAccess(this);
    }
    // end of WebDocWf extension for writeable fully cached obejcts
}
