
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * $Id: StandardObjectIdAllocator.java,v 1.7 2005/05/26 08:08:10 predrag Exp $
 *
 */
package com.lutris.appserver.server.sql.standard;

import java.math.BigDecimal ;
import java.math.BigInteger ;
import java.sql.PreparedStatement ;
import java.sql.ResultSet ;
import java.sql.SQLException ;

import org.enhydra.dods.DODS;

import com.lutris.appserver.server.sql.DBConnection;
import com.lutris.appserver.server.sql.DatabaseManagerException;
import com.lutris.appserver.server.sql.LogicalDatabase;
import com.lutris.appserver.server.sql.ObjectId;
import com.lutris.appserver.server.sql.ObjectIdAllocationError;
import com.lutris.appserver.server.sql.ObjectIdAllocator;
import com.lutris.appserver.server.sql.ObjectIdException;
import com.lutris.logging.Logger;
import com.lutris.util.Config;
import com.lutris.util.ConfigException;
import com.lutris.util.KeywordValueException;

/**
 * Object ids can only be created via this manager.
 * Ensures that all object ids are unique across all
 * objects in this logical database. Also ensures good performance
 * for allocating object ids.
 * <P>
 * The configuration data is specified in the section:
 * <CODE>DatabaseManager.DB.<I>dbName</I>.ObjectId</CODE>
 * <P>
 * Configuration fields are:
 * <UL>
 * <LI> <CODE>CacheSize</CODE> -
 *      The number of object id's to cache between database queries. Optional,
 *  if not specified, then it defaults to 1024.
 * <LI> <CODE>MinValue</CODE> -
 *      The starting number of Object ID allocation. This will only be used
 *  if the Object ID table is empty and thus is useful in development
 *  and testing. Optional, if not specified it defaults to
 *  100000000000000000. Note that the largest number that can be
 *  associated with an OID in LBS is "database: DECIMAL(19,0)"
 * </UL>
 *
 * @author  Kyle Clark
 * @author  Paul Morgan
 * @since   LBS1.8
 * @version $Revision: 1.7 $
 */
public class StandardObjectIdAllocator implements ObjectIdAllocator {
    
    /**
     * The maximum object id in the cache.
     */
    private ObjectId max = null;
        
    /**
     * The object id mos recently allocated.
     */
    private ObjectId current = null;
        
    /**
     * The next object id that will be allocated.
     */
    private ObjectId next = null;
        
    /**
     * Default cache size if 1K.  Note that if the
     * application is started and stopped frequently
     * you should drop this value in order not
     * to waste object ids.
     */
    private long cacheSize = 1024;

   /**
     * Used in updateCache method in update statement because 
     * some databases has NEXT as a reserved word (HSQL)
     */
    private boolean columnWithPrefix = false;
                

   /**
     * Name of objectid table in database  
     */
    private String  oidTableName = "objectid";

   /**
     * Name of next column in objectid table  
     */
    private String  nextColumnName = "next";
    /**
     * Starting value for object id creation, if table is empty.
     * This is a string because very large numbers are possible.
     */
    private String  oidMinValue = "0";
        
    /**
     * Reference to the logical database for easy access to the
     * connection pool.
     */
    private LogicalDatabase logicalDatabase  = null;

    
    /**
     * Name of primary logical database for access to the
     * objectId table.
     */
    private String  primaryLogicalDatabaseName = null;
    
    /**
     * If allocation of object id's fails (because the contents of
     * the table are modified by another process) this object
     * id manager will keep trying to allocate object ids
     * until it succeeds.  If it fails more than LOG_THRESHOLD
     * times, then it will log a message to the system.
     */
    private static final int LOG_THRESHOLD = 100;
    
    
    /**
     * Initialize the object id manager.
     *
     * @param logicalDatabase LogicalDatabase.
     * @param objIdConfig LogicalDatabase Config object.
     * @exception ConfigException if bad configuration information is
     *  given in the config file.
     */
    protected StandardObjectIdAllocator(LogicalDatabase lDb, Config objIdConfig) 
        throws ConfigException {
        try {
            primaryLogicalDatabaseName = objIdConfig.getString("PrimaryLogicalDatabase", lDb.getName());
            cacheSize                  = objIdConfig.getInt("CacheSize", 1024);
            oidMinValue                = objIdConfig.getString("MinValue", "10000000000000000");
            columnWithPrefix           = objIdConfig.getBoolean("NextWithPrefix",false);
            oidTableName               = objIdConfig.getString("OidTableName","objectid");
            nextColumnName             = objIdConfig.getString("NextColumnName","next");
        } catch (KeywordValueException except) {
            throw new ConfigException("Bad DatabaseManager.DB."
                    + primaryLogicalDatabaseName
                    + ".ObjectId section defined in config file.");
        }
    }
    
    /**
     * Check if oid belongs to Object id's range [minOId, currentOId]
     *
     * @param oid
     *   oid which will be checked.
     * @exception ObjectIdException
     *   If a oid does't belong to range.
     */
    public synchronized void checkOId(ObjectId oid)  throws ObjectIdException {
        if (oid == null) { 
            throw new ObjectIdException("Object id is null");
        }
        BigDecimal  min = new BigDecimal (new BigInteger (oidMinValue));
        BigDecimal  id = new BigDecimal (oid.toString());

        if (id.compareTo(min) == -1) {
            throw new ObjectIdException("Object id out of range");
        }
        if (next == null || max == null) {
            updateCache();  
        }
        BigDecimal  bdNext = new BigDecimal (next.toString());

        if (id.compareTo(bdNext) == 0 || id.compareTo(bdNext) == 1) {
            throw new ObjectIdException("Object id out of range");
        }
    }
    
    /**
     * Allocates a new object id.
     */
    public synchronized ObjectId allocate() {
        try {
            if (next == null || max == null || next.equals(max)) {
                updateCache();
            }
            current = next; // next free
            next = next.increment(); 
            return  current;
        } catch (Exception  e) {
            throw new ObjectIdAllocationError("ObjectIdAllocator: "
                    + "\nFailed to allocate object id.  Caused by: " + "\n"
                    + e.getMessage());
        }
    }
    
    /**
     * Updates the cache of object id's.
     *
     * contains WebDocWf bug fix, concurrent applications should be allowed to generate oid's
     */
    private void updateCache() {
        final String  OID_TABLE = oidTableName;
        DBConnection conn = null;

        try {
            boolean tryAgain = false;
            int tryCount = 0;

            conn = allocateConnection();
            conn.setAutoCommit(false);
            do {
                BigDecimal  dbNext;
                ResultSet  rs;

                // Query next available object id from database.
                rs = conn.executeQuery("select * from " + OID_TABLE);
                if (!rs.next()) {
                    // Table is empty - initialize
                    conn.executeUpdate("insert into " + OID_TABLE + " values("
                            + oidMinValue + ")");
                    dbNext = new BigDecimal (oidMinValue);
                } else {
                    dbNext = rs.getBigDecimal(nextColumnName);
                }
                rs.close();             // In case there's more // JOHN
 // Sync this managers next oid w/ current value in db.
                next = new ObjectId(dbNext);
                // Update db contents
 // i.e. Allocate a block of oids of size cacheSize.
 BigDecimal  dbLast = dbNext;

                dbNext = dbNext.add(BigDecimal.valueOf(cacheSize));
                // This now updates all rows if there's more than one.
 String  sql = "update " + OID_TABLE + " "; // compliance with WEBDOCWF begin
 // WebDocWf bug fix, concurrent applications should be allowed to generate oid's
 // the "+ "where next = ?"" has been under comments before
 if(!columnWithPrefix)  
                        sql+= "set " +nextColumnName +" = ?" + " " + "where "+nextColumnName +"= ?";
                else
                        sql+= "set " + OID_TABLE + "."+nextColumnName+" = ?" + " " + "where " + OID_TABLE + "."+nextColumnName+" = ?";
                // end of first part of bugfix
 // original line
 // compliance with WEBDOCWF end
 PreparedStatement  stmt = conn.prepareStatement(sql);

                stmt.setBigDecimal(1, dbNext);
                // compliance with WEBDOCWF begin
 // WebDocWf bug fix, concurrent applications should be allowed to generate oid's
 // The following line has been under comments before:
                stmt.setBigDecimal(2, dbLast);
                // end of second part of bugfix
 // original line
 // compliance with WEBDOCWF end
 if (conn.executeUpdate(stmt, sql) < 1) {
                    // We were unable to update the table.
                    tryAgain = true;
                    tryCount++;
                    if (tryCount >= LOG_THRESHOLD) {
                              DODS.getLogChannel().write(Logger.EMERGENCY,
                                "ObjectIdAllocator: "
                                + "\n  Failed to allocate object ids."
                                + "\n  Trying again....");
                    }
                    // Transaction rollback
                    conn.rollback();
                    conn.reset();         // free resources
                    conn.setAutoCommit(false);
                    if (tryCount >= 50) {
                              DODS.getLogChannel().write(Logger.EMERGENCY,
                                "ObjectIdAllocator: "
                                + "\n  Failed to allocate object ids."
                                + "\n  Tried 50 times.  Giving up.");
                        throw new ObjectIdAllocationError("Failed to allocate object id."
                                + "\nTried 50 times.  Giving up.");
                    }
                } else {
                    tryAgain = false;
                    // Sync this managers max oid w/ db contents.
                    max = new ObjectId(dbNext);
                    // Transcation commit.
                    conn.commit();
                    if (tryCount >= LOG_THRESHOLD) {
                               DODS.getLogChannel().write(Logger.EMERGENCY,
                                "ObjectIdAllocator: "
                                + "\n  Succeeded in allocating object ids."
                                + "\n  Continuing...");
                    }
                }
            } while (tryAgain);
        } catch (SQLException  sqlExcept) {
            // An error occured.  Rollback all changes.
            max = next;               // The cache is not updated.
               DODS.getLogChannel().write(Logger.EMERGENCY,
                    "ObjectIdAllocator: "
                    + "\n  Failed to allocate object ids.  Giving up!"
                    + "\n    SQLState = " + sqlExcept.getSQLState()
                    + "\n    SQLError = " + sqlExcept.getErrorCode()
                    + "\n    SQLMsg   = " + sqlExcept.getMessage());
            if (conn != null) {
                try {
                    conn.rollback();
                } catch (SQLException  rollbackExcept) {
                         DODS.getLogChannel().write(Logger.EMERGENCY,
                            "ObjectIdAllocator: "
                            + "\n  Failed to rollback transaction."
                            + "\n    SQLState = " + rollbackExcept.getSQLState()
                            + "\n    SQLError = "
                            + rollbackExcept.getErrorCode()
                            + "\n    SQLMsg   = " + rollbackExcept.getMessage());
                }
                if (!conn.handleException(sqlExcept)) {
                    conn = null;
                }
            }
            throw new ObjectIdAllocationError("ObjectIdAllocator: "
                    + "\n  Failed to allocate object id.  Caused by SQLException:"
                    + "\n    SQLState = " + sqlExcept.getSQLState()
                    + "\n    SQLError = " + sqlExcept.getErrorCode()
                    + "\n    SQLMsg   = " + sqlExcept.getMessage());
        } catch (ObjectIdException oidEx) {
            throw new ObjectIdAllocationError("ObjectIdAllocator: "
                    + "\nFailed to allocate object id.  Caused by: " + "\n"
                    + oidEx.getMessage());
        }
        finally {
            if (conn != null) {
                try {
                    conn.reset();         // free existing resource if any
                } catch (SQLException  sqlExcept) {
                         DODS.getLogChannel().write(Logger.EMERGENCY,
                            "ObjectIdAllocator: "
                            + "\n  Failed to reset connection."
                            + "\n    SQLState = " + sqlExcept.getSQLState()
                            + "\n    SQLError = " + sqlExcept.getErrorCode()
                            + "\n    SQLMsg   = " + sqlExcept.getMessage());
                }
                finally {
                    conn.release();
                }
            }
        }
    }

    private DBConnection allocateConnection() throws SQLException  {
        if (logicalDatabase == null) {
            try {
                logicalDatabase = DODS.getDatabaseManager().findLogicalDatabase(primaryLogicalDatabaseName);
            } catch (DatabaseManagerException e) {
                e.printStackTrace();
                throw new SQLException (e.getMessage());
            }
        }
        return logicalDatabase.allocateConnection();
    }
}
