
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: StandardLogicalDatabase.java,v 1.10 2005/05/26 08:08:10 predrag Exp $
 */
package com.lutris.appserver.server.sql.standard;

import java.lang.reflect.Constructor ;
import java.sql.Driver ;
import java.sql.ResultSet ;
import java.sql.SQLException ;
import java.util.Date ;

import org.enhydra.dods.Common;
import org.enhydra.dods.CommonConstants;
import org.enhydra.dods.DODS;
import org.enhydra.dods.DriverDependencies;
import org.enhydra.dods.cache.CacheConstants;

import com.lutris.appserver.server.sql.AbstractDBTransactionFactory;
import com.lutris.appserver.server.sql.ConnectionAllocator;
import com.lutris.appserver.server.sql.DBConnection;
import com.lutris.appserver.server.sql.DBQuery;
import com.lutris.appserver.server.sql.DBTransaction;
import com.lutris.appserver.server.sql.DBTransactionFactoryCreator;
import com.lutris.appserver.server.sql.DatabaseManagerConfiguration;
import com.lutris.appserver.server.sql.ExtendedDBConnection;
import com.lutris.appserver.server.sql.LogicalDatabase;
import com.lutris.appserver.server.sql.ObjectId;
import com.lutris.appserver.server.sql.ObjectIdAllocator;
import com.lutris.appserver.server.sql.ObjectIdException;
import com.lutris.dods.builder.generator.query.RDBColumn;
import com.lutris.logging.Logger;
import com.lutris.util.Config;
import com.lutris.util.ConfigException;
import com.lutris.util.KeywordValueException;
/**
 * Represents a logical database. Each logical database has a connection
 * allocator and a object id allocator. The standard implementation of
 * a local database is used when the database is described as of
 * type <I>Standard</I> in
 * <CODE>DatabaseManager.DB.<I>dbname</I>.ClassType</CODE>
 * <P>
 * The configuration data is specified in the section:
 * <CODE>DatabaseManager.DB.<I>dbName</I></CODE>
 * <P>
 * The following is a description of the sub fields:
 * <UL>
 * <LI> <B><CODE>JdbcDriver</CODE></B> -
 *      The JDBC driver to use to access that database. Manditory.
 *      E.g. "intersolv.jdbc.sequelink.SequeLinkDriver"
 * <LI> <B><CODE>Connection</CODE></B> -
 *      This <I>section</I> is passed onto the connection allocator. See
 *     <A HREF="com.lutris.appserver.server.sql.StandardConnectionAllocator"</A>
 *      for details on connection allocator sub fields.
 * <LI> <B><CODE>ObjectId</CODE></B> -
 *      This <I>section</I> is passed onto the object id allocator. See
 *      <A HREF="com.lutris.appserver.server.sql.ObjectIdAllocator"</A>
 *      for details on object id allocator sub fields.
 * </UL>
 *
 * @author      Paul Morgan
 * @author      Kyle Clark
 * @since       LBS1.8
 * @version     $Revision: 1.10 $
 */
public class StandardLogicalDatabase implements LogicalDatabase, CacheConstants, DriverSpecificConstants  {

    public static final String  PARAMNAME_DRIV_DEP_CLASS = "DriverDependenciesClass";

    /**
     * Database Manager Transaction Factory.
     */
    protected AbstractDBTransactionFactory transactionFactory;

    /**
     * Database connection allocator.
     */
    protected ConnectionAllocator connectionAllocator;

    /**
     * Object id manager.
     */
    protected ObjectIdAllocator objectIdAllocator;
    


    /**
     * Driver object object.  Important: a reference to this must be
     * maintained or the driver class might be garbage collected out from
     * under us.
     */
    protected Driver  driver;

    /**
     * DriverClassName
     */
    protected String  driverClassName;


    /**
     * The symbolic name of this logical database.
     */
    protected String  dbName;

    /**
     * The database vendor type.
     */
    protected String  dbType;

    /**
     * The database configuration
     */
    private DatabaseConfiguration dbConf;
    private DriverDependencies drivDepsInstance;

    /**
     * Default constructor to configure a single logical database.
     * Note that the <CODE>init()</CODE> method must be called to
     * configure the database.
     */
    public StandardLogicalDatabase() {}

    /**
     * Creates and configures a single logical database.
     *
     * @param dbName
     *   The logical name of the database.  Used to
     * @param dbConfig
     *   The configuration object for this logical database.
     *   get config parameters.
     * @exception ConfigException
     *   If there is an error in the configuration file.
     * @exception SQLException
     *   If a SQL error occurs.
     */
    public StandardLogicalDatabase(String  dbName, Config dbConfig)
        throws ConfigException, SQLException  {
        this.init(dbName, dbConfig);
    }

    /**
     * Creates and configures a single logical database.
     *
     * @param dbName
     *   The logical name of the database.  Used to
     * @param dbConfig
     *   The configuration object for this logical database.
     *   get config parameters.
     * @param DbManagerConf
     *   The configuration object of DatabaseManager
     * @exception ConfigException
     *   If there is an error in the configuration file.
     * @exception SQLException
     *   If a SQL error occurs.
     */
    public StandardLogicalDatabase(String  dbName, Config dbConfig, DatabaseManagerConfiguration DbManagerConf)
        throws ConfigException, SQLException  {
        this.init(dbName, dbConfig, DbManagerConf);
    }


    public void init(String  dbName, Config dbConfig)
        throws ConfigException, SQLException  {
        dbConf = new DatabaseConfiguration(dbName);
        initConf( dbName, dbConfig);

    }

    public void init(String  dbName, Config dbConfig, DatabaseManagerConfiguration DbManagerConf)
        throws ConfigException, SQLException  {
        dbConf = new DatabaseConfiguration(dbName, DbManagerConf);
        initConf( dbName,  dbConfig);

    }


    /**
     * Initialize a single logical database. Used after the default
     * constructor is called.
     *
     * @param dbName The logical name of the database.
     * @param dbConfig The configuration object for this logical database.
     *
     * @exception ConfigException If there is an error in
     *   the configuration file.
     * @exception SQLException If a SQL error occurs.
     */
    private void initConf(String  dbName, Config dbConfig)
        throws ConfigException, SQLException  {

        this.dbName = dbName;


        // Read configuration information.
 Config objIdConfig, connectionConfig;

        try {
            String  changeAutocommit = dbConfig.getString("ChangeAutocommit", "true");
            if (changeAutocommit.equalsIgnoreCase("false")){
                Common.setChangeAutocommit(dbName, false);
            }else if (changeAutocommit.equalsIgnoreCase("true")){
                Common.setChangeAutocommit(dbName, true);
            }else{
                throw new ConfigException("invalid value for ChangeAutocommit parameter");
            }
        } catch (ConfigException e) {
            throw new ConfigException("Error reading ChangeAutocommit : "+e.getMessage());
        }

        try {
            objIdConfig = (Config) dbConfig.getSection("ObjectId");
            connectionConfig = (Config) dbConfig.getSection("Connection");
        } catch (KeywordValueException except) {
            throw new ConfigException("Invalid ObjectId or Connection sections in config file.");
        }
 
        Object  dataSource;
        try {
           dataSource = connectionConfig.getDataSource("DataSourceName");
        }
        catch (Exception  ex){  
            dataSource = null;
        }  
        if (dataSource==null){
        this.driverClassName = dbConfig.getString("JdbcDriver");
        String  jdbcDriver = this.driverClassName ;

        // Load JDBC driver; report an SQLException on failure.
 // No-op if already loaded.
 try {
            // We must use the applications class loader to load JDBC classes.
 Class  driverClass = Class.forName(jdbcDriver);

            driver = (Driver ) driverClass.newInstance();
        } catch (java.lang.ClassNotFoundException  except) {
            throw new SQLException ("can't load JDBC driver class: "
                                       + except.getMessage());
        } catch (java.lang.InstantiationException  except) {
            throw new SQLException ("can't instantiate JDBC driver class: "
                                       + except.getMessage());
        } catch (java.lang.IllegalAccessException  except) {
            throw new SQLException ("can't instantiate JDBC driver class: "
                                       + except.getMessage());
        }
            try {
                this.dbType = dbConfig.getString("ClassType");
            }catch(Exception  e1){
                this.dbType = Common.getDatabaseVendor(jdbcDriver);
            }
        }
        
    try {
        dbConf.setDBConnectionFactoryName(dbConfig.getString(CommonConstants.CONNECTION_FACTORY));
       } catch (Exception  ex) {}

       try {
       dbConf.setConnectionAllocatorName(dbConfig.getString(CommonConstants.CONNECTION_ALLOCATOR));
      } catch (Exception  ex) {}

        ConnectionAllocator connectionAllocator = loadConnectionAllocator(connectionConfig);
        ObjectIdAllocator objectIdAllocator = loadObjectIdAllocator(objIdConfig);

        this.connectionAllocator = connectionAllocator;
        this.objectIdAllocator = objectIdAllocator;
        if(dataSource!=null){
            String  databaseType=null;
            String  fullDriverClassName=null;
            try{
                ExtendedDBConnection tempConnection = (ExtendedDBConnection)connectionAllocator.allocate();
                String  driverName=tempConnection.getConnection().getMetaData().getDriverName();
                tempConnection.release();
                databaseType=Common.getDatabaseVendorFromDriverName(driverName);
                fullDriverClassName=Common.getDatabaseDriverClassFromDriverName(driverName);
            }catch(Exception  e){
                try {
                    databaseType = dbConfig.getString("ClassType");
                    fullDriverClassName = dbConfig.getString("JdbcDriver",null);
                }catch(Exception  e1){
                    databaseType = Common.getDatabaseVendor(dbConfig.getString("JdbcDriver",null));
                }               
            }
            this.dbType=databaseType;
            this.driverClassName = fullDriverClassName; 
        }

        this.initDisableFetchSizeWithMaxRows();
        this.initResultSetConcurrency();
        this.initResultSetType();

        
        try {
            dbConf.setDBTransactionFactoryName(dbConfig.getString(CommonConstants.TRANSACTION_FACTORY));
        } catch (Exception  ex) {}
//        try {
            transactionFactory = DBTransactionFactoryCreator.getDBTransactionFactory(dbConf.getDBTransactionFactoryName(),this);
//        } catch (RuntimeException e) {
//            throw new SQLException("can't create Transaction Factory Object: "+ e.getMessage());
//        }
 Config defaultsConfig = dbConfig;

        if (defaultsConfig != null) {
            try {
                dbConf.setLazyLoading(defaultsConfig.getBoolean(PARAMNAME_LAZY_LOADING));
            } catch (Exception  ex) {}
            try {
                dbConf.setCaseSensitive(defaultsConfig.getBoolean(PARAMNAME_CASE_SENSITIVE));
            } catch (Exception  ex) {}

            try {
                dbConf.setMaxExecuteTime(defaultsConfig.getInt(PARAMNAME_MAX_EXECUTE_TIME));
            } catch (Exception  ex) {}
            try {
                dbConf.setTransactionCheck(defaultsConfig.getBoolean(PARAMNAME_TRANSACTION_CHECK));
            } catch (Exception  ex) {}
            try {
                dbConf.setDeleteCheckVersion(defaultsConfig.getBoolean(PARAMNAME_DELETE_CHECK_VERSION));
            } catch (Exception  ex) {}
            try {
                dbConf.setAutoSave(defaultsConfig.getBoolean(PARAMNAME_AUTO_SAVE));
            } catch (Exception  ex) {}
            try {
                dbConf.setAutoSaveCreateVirgin(defaultsConfig.getBoolean(PARAMNAME_AUTO_SAVE_CREATE_VIRGIN));
            } catch (Exception  ex) {}
            try {
                dbConf.setAutoWrite(defaultsConfig.getBoolean(PARAMNAME_AUTO_WRITE));
            } catch (Exception  ex) {}
            try {
                dbConf.setTransactionCaches(defaultsConfig.getBoolean(PARAMNAME_TRANSACTION_CACHES));
            } catch (Exception  ex) {}
            try {
                dbConf.setDeadlockWaitTime(defaultsConfig.getInt(PARAMNAME_DEADLOCK_READ_TIME));
            } catch (Exception  ex) {}
            try {
                dbConf.setDeadlockRetryCount(defaultsConfig.getInt(PARAMNAME_DEADLOCK_RETRY_NUMBER));
            } catch (Exception  ex) {}
            try {
                dbConf.setReadOnly(defaultsConfig.getBoolean(PARAMNAME_ALL_READ_ONLY));
            } catch (Exception  ex) {}
            try {
                dbConf.setDefaultFetchSize(defaultsConfig.getInt(PARAMNAME_DEFAULT_FETCH_SIZE));
            } catch (Exception  ex) {}
            try {
                dbConf.setQueryTimeout(defaultsConfig.getInt(PARAMNAME_QUERY_TIMEOUT));
            } catch (Exception  ex) {}
            try {
                dbConf.setSelectOids(defaultsConfig.getBoolean(PARAMNAME_SELECT_OIDS));
            } catch (Exception  ex) {}
            try {
                dbConf.setIncrementVersions(defaultsConfig.getBoolean(PARAMNAME_INCREMENT_VERSIONS));
            } catch (Exception  ex) {}
            try {
                boolean tmpSCrN = defaultsConfig.getBoolean(PARAMNAME_USE_CURSOR_NAME);
                dbConf.setUseCursorName(tmpSCrN);
            } catch (Exception  ex) {}
            try {
                dbConf.setFullCacheCountLimit(defaultsConfig.getInt(CacheConstants.FULL_CACHE_COUNT_LIMIT));
            } catch (Exception  ex) {}
            try {
                dbConf.setQueryCacheImplClass(defaultsConfig.getString(CommonConstants.QUERY_CACAHE_IMPL_CLASS));
            } catch (Exception  ex) {}
            try {
                dbConf.setInitCachesResultSetType(defaultsConfig.getString(CommonConstants.INIT_CACHES_RESULT_SET_TYPE));
            } catch (Exception  ex) {}
            try {
                dbConf.setInitCachesResultSetConcurrency(defaultsConfig.getString(CommonConstants.INIT_CACHES_RESULT_SET_CONCURRENCY));
            } catch (Exception  ex) {}
            try {
                dbConf.setSqlBatch(defaultsConfig.getBoolean(CommonConstants.SQL_BATCH));
            } catch (Exception  ex) {}
            try {
                dbConf.setQueryTimeLimit(new Integer (defaultsConfig.getInt(CommonConstants.QUERY_TIME_LIMIT)));
            } catch (Exception  ex) {}

            // -XA
 
            try {
                dbConf.setXaDefaultTimeout(defaultsConfig.getInt(CommonConstants.XA_DEFAULT_TIMEOUT));
            } catch (Exception  ex) {}
            try {
                dbConf.setXaTransactonManagerLookupName(defaultsConfig.getString(CommonConstants.XA_TM_LOOKUP_NAME));
                
            } catch (Exception  ex) {}
            try {
                dbConf.setXaUsageCase(defaultsConfig.getInt(CommonConstants.XA_USAGE_CASE));
            } catch (Exception  ex) {}
            try {
                dbConf.setXaWrappedTransImplFactory(defaultsConfig.getString(CommonConstants.XA_WRAPPED_TRANS_IMPL_FACTORY));
            } catch (Exception  ex) {}
            try {
                dbConf.setXaUserTransactonLookupName(defaultsConfig.getString(CommonConstants.XA_USER_TRANSACTION_LOOKUP_NAME));
            } catch (Exception  e) { }
            try {
                dbConf.setXaJtaSupport(defaultsConfig.getString(CommonConstants.XA_JTA_SUPPORT));
            } catch (Exception  e) { }
            
            // -XA
        }
        try {
            defaultsConfig = (Config) dbConfig.getSection("cache");
        } catch (KeywordValueException except) {
            throw new ConfigException("No DatabaseManager.defaults defined in config file.");
        }
        if (defaultsConfig != null) {
            try {
                dbConf.setReserveFactor(defaultsConfig.getDouble(PARAMNAME_RESERVE_FACTOR));
            } catch (Exception  ex) {}
            try {
                dbConf.setCachePercentage(defaultsConfig.getDouble(PARAMNAME_CACHE_PERCENTAGE));
            } catch (Exception  ex) {}
            try {
                dbConf.setMaxCacheSize(defaultsConfig.getInt(PARAMNAME_MAX_CACHE_SIZE));
            } catch (Exception  ex) {}
            try {
                dbConf.setMaxSimpleCacheSize(defaultsConfig.getInt(PARAMNAME_MAX_SIMPLE_CACHE_SIZE));
            } catch (Exception  ex) {}
            try {
                dbConf.setMaxComplexCacheSize(defaultsConfig.getInt(PARAMNAME_MAX_COMPLEX_CACHE_SIZE));
            } catch (Exception  ex) {}
            try {
                dbConf.setMaxMultiJoinCacheSize(defaultsConfig.getInt(PARAMNAME_MAX_MULTI_JOIN_CACHE_SIZE));
            } catch (Exception  ex) {}
            try {
                dbConf.setInitialCacheFetchSize(defaultsConfig.getInt(PARAMNAME_INITIAL_CACHE_FETCH_SIZE));
            } catch (Exception  ex) {}
            try {
                dbConf.setInitialDSCacheSize(defaultsConfig.getInt(PARAMNAME_INITIAL_DS_CACHE_SIZE));
            } catch (Exception  ex) {}
            try {
                dbConf.setDodsCacheFactory(defaultsConfig.getString(PARAMNAME_DODS_CACHE_FACTORY));
            } catch (Exception  ex) {}
            

        }

    }

    /**
     * Check does oid belong to Object id's range [minOId, currentOId]
     *
     * @param oid
     *   oid which will be checked.
     * @exception ObjectIdException
     *   If a oid does't belong to range.
     */
    public void checkOId(ObjectId oid)  throws ObjectIdException {
        objectIdAllocator.checkOId(oid);
    }

    /**
     * Return the connection allocator.
     *
     * @param connectionConfig
     *   The configuration object for the connection allocator.
     * @exception ConfigException
     *   If there is an error in the configuration file.
     * @return
     *   The connection allocator.
     */
    public ConnectionAllocator loadConnectionAllocator(Config connectionConfig)
        throws ConfigException {
        Class  connectionAllocatorClass = null;
        Constructor  connectionAllocatorConstructor = null;
        Class [] methodTypes={LogicalDatabase.class,connectionConfig.getClass()};
        Object [] methodArgs={this,connectionConfig};
        ConnectionAllocator allocator = null;
        if (dbConf.getConnectionAllocatorName()!=null){         
            try{
                connectionAllocatorClass = Class.forName(dbConf.getConnectionAllocatorName());
                connectionAllocatorConstructor = connectionAllocatorClass.getConstructor(methodTypes);
                allocator = (ConnectionAllocator)connectionAllocatorConstructor.newInstance(methodArgs);
            }catch(Exception  e){
                DODS.getLogChannel().write(Logger.ERROR,"Failed to make Connection Allocator :"+dbConf.getConnectionAllocatorName()+" creating StandardConnectionAllocator insted");
                allocator = null;
            }
        }
        if (dbConf.getConnectionAllocatorName()==null || allocator == null){
            //allocator = new StandardConnectionAllocator(this, connectionConfig);
 try{ 
            connectionAllocatorClass = Class.forName("com.lutris.appserver.server.sql.standard.StandardConnectionAllocator");
            connectionAllocatorConstructor = connectionAllocatorClass.getConstructor(methodTypes);
            allocator = (ConnectionAllocator)connectionAllocatorConstructor.newInstance(methodArgs);
          }catch(Exception  e){
            String  str = "Failed to make Standard Connection Allocator : com.lutris.appserver.server.sql.standard.StandardConnectionAllocator";
                DODS.getLogChannel().write(Logger.CRITICAL,str);
            throw new Error (str);
         }
        }
        return allocator;
    }

    /**
     * Return the object id allocator.
     *
     * @param objIdConfig
     *   The configuration object for the object id allocator.
     * @exception ConfigException
     *   If there is an error in the configuration file.
     * @return
     *   The object id allocator.
     */
    public ObjectIdAllocator loadObjectIdAllocator(Config objIdConfig)
        throws ConfigException {
        return new StandardObjectIdAllocator(this, objIdConfig);
    }

    /**
     * Return the connection allocator object associated with this
     * logical database.
     *
     * @return
     *   The connection allocator.
     * @exception SQLException
     *   if SQL error occurs allocating connection.
     */
    public DBConnection allocateConnection()
        throws SQLException  {
        return connectionAllocator.allocate();
    }

    /**
     * Return an object id for this logical database.
     *
     * @return
     *   The object id.
     * @exception ObjectIdException
     *   if an error occurs allocating an object id.
     */
    public ObjectId allocateObjectId()
        throws ObjectIdException {
        return objectIdAllocator.allocate();
    }

    /**
     * Return a transaction for use on this logical database.
     *
     * @return The transaction object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    public DBTransaction createTransaction()
        throws SQLException  {
        return transactionFactory.getTransaction(connectionAllocator.allocate());
    }

    /**
     * Return a query for use on this logical database.
     *
     * @return The query object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    public DBQuery createQuery()
        throws SQLException  {
        return new StandardDBQuery(connectionAllocator.allocate());
    }

    /**
     * Immediately shutdown the logical database. This is
     * normally only called by the database manager on shutdown.
     */
    public void shutdown() {
        connectionAllocator.dropAllNow();
        driver = null;
    }

    /**
     * Return the symbolic name of this logical database.
     *
     * @return The symbolic name.
     */
    public String  getName() {
        return dbName;
    }

    /**
     * Return a description of the logical database type.
     *
     * @return The type.
     */
    public String  getType() {
        return dbType;
    }

    /**
     * Return a driver class of the logical database.
     *
     * @return Driver Class Name.
     */
    public String  getDriverClassName() {
        return this.driverClassName;
    }



    /**
     * Return a driver property from database vendor configutation file.
     *
     * @param  paramName Parameter Name
     * @return Driver Class Name.
     */
    public String  getDriverProperty(String  paramName) {
        String  key ="Drivers/"+driverClassName+"/"+paramName;
        return Common.getDodsConfProperty(key,dbType);
    }




    /**
     * Return the number of currently active connections.
     *
     * @return The number of connections.
     */
    public int getActiveConnectionCount() {
        return connectionAllocator.getActiveCount();
    }

    /**
     * Return the maximum number of connections active at one time.
     *
     * @return The number of connections.
     */
    public int getMaxConnectionCount() {
        return connectionAllocator.getMaxCount();
    }

    /**
     * Return the time when the maximum connection count occured.
     *
     * @return The <CODE>Date</CODE> when the maximum connection
     *  count occured.
     */
    public Date  getMaxConnectionCountDate() {
        return connectionAllocator.getMaxCountDate();
    }

    /**
     * Reset the maximum connection count and date.
     */
    public void resetMaxConnectionCount() {
        connectionAllocator.resetMaxCount();
    }

    /**
     * Return the number of database requests.
     *
     * @return The number of database requests (queries or transactions).
     */
    public long getRequestCount() {
        return connectionAllocator.getRequestCount();
    }

    /**
     * Return conf parameters for current database
     *
     * @return values for DODS default values for current database
     */
    public DatabaseConfiguration getDatabaseConfiguration() {
        return dbConf;
    }


    /**
     * Return value of DisableFetchSizeWithMaxRows property
     * @return Value of DisableFetchSizeWithMaxRows property
     */

    public boolean getDisableFetchSizeWithMaxRows(){
        return dbConf.getDisableFetchSizeWithMaxRows();
    }

    /**
     *  Init value of DisableFetchSizeWithMaxRows property
     *
     */
    private void  initDisableFetchSizeWithMaxRows() {
        String  tmpDFSWMR = getDriverProperty(PARAMNAME_DISABLE_FETCH_SIZE_WITH_MAX_ROWS);
        if (tmpDFSWMR==null){
            dbConf.setDisableFetchSizeWithMaxRows(DEFAULT_DISABLE_FETCH_SIZE_WITH_MAX_ROWS);
        }else{
            if (tmpDFSWMR.equalsIgnoreCase("true")){
                dbConf.setDisableFetchSizeWithMaxRows(true);
            }else if (tmpDFSWMR.equalsIgnoreCase("false")){
                dbConf.setDisableFetchSizeWithMaxRows(false);
            }else{
                DODS.getLogChannel().write(Logger.DEBUG,"Invalid value for DisableFetchSizeWithMaxRows parameter :"+tmpDFSWMR);
                dbConf.setDisableFetchSizeWithMaxRows(DEFAULT_DISABLE_FETCH_SIZE_WITH_MAX_ROWS);
            }
        }
    }


    /**
     * Return value of UseCursorName property
     * @return Value of UseCursorName property
     */
    public boolean getUseCursorName() {
        if (dbConf.getUseCursorName()==null){
            String  tmpUCrN = getDriverProperty(PARAMNAME_USE_CURSOR_NAME);
            if (tmpUCrN==null){
                return DEFAULT_USE_CURSOR_NAME;
            }else{
                if (tmpUCrN.equalsIgnoreCase("true")) {
                    setUseCursorName(true);
                    return true;
                }else if (tmpUCrN.equalsIgnoreCase("false")){
                    setUseCursorName(false);
                    return false;
                }else{
                    DODS.getLogChannel().write(Logger.DEBUG,"Invalid value for UseCursorName parameter :"+tmpUCrN);
                    return DEFAULT_USE_CURSOR_NAME;
                }
            }
        }else{
            return dbConf.getUseCursorName().booleanValue();
        }
    }


    /**
     * Change value of UseCursorName property
     * @param use
     */
    public void setUseCursorName(boolean use) {
        dbConf.setUseCursorName(use);
    }


    /**
     * Change value of UseCursorName property
     * @param use
     */
    public void setUseCursorName(Boolean  use) {
        dbConf.setUseCursorName(use);
    }


    /**
     * Return value of ResultSetType property
     * @return Value of ResultSetType property
     */
    private void initResultSetType() {
        String  tmpUCrN = getDriverProperty(PARAMNAME_RESULT_SET_TYPE);
        try {
            if (tmpUCrN==null)
                dbConf.setResultSetType(DEFAULT_RESULT_SET_TYPE);
            if(tmpUCrN.equalsIgnoreCase("TYPE_SCROLL_SENSITIVE"))
                dbConf.setResultSetType(ResultSet.TYPE_SCROLL_SENSITIVE);
            else if (tmpUCrN.equalsIgnoreCase("TYPE_SCROLL_INSENSITIVE"))
                dbConf.setResultSetType(ResultSet.TYPE_SCROLL_INSENSITIVE);
            else if (tmpUCrN.equalsIgnoreCase("TYPE_FORWARD_ONLY"))
                dbConf.setResultSetType(ResultSet.TYPE_FORWARD_ONLY);
            else {
                DODS.getLogChannel().write(Logger.DEBUG,"Invalid value for ResultSetType parameter :"+tmpUCrN);
                dbConf.setResultSetType(DEFAULT_RESULT_SET_TYPE);
            }
        }catch(Exception  ex) {
            DODS.getLogChannel().write(Logger.DEBUG,"Use default value for ResultSetType parameter ");
        }
    }


    /**
     * Return value of ResultSetType property
     * @return Value of ResultSetType property
     */

    public int getResultSetType(){
        return dbConf.getResultSetType();
    }


    /**
     * Return value of ResultSetConcurrency property
     * @return Value of ResultSetConcurrency property
     */
    private void initResultSetConcurrency() {
        String  tmpUCrN = getDriverProperty(PARAMNAME_RESULT_SET_CONCURRENCY);
        try{
            if (tmpUCrN==null)
                dbConf.setResultSetConcurrency(DEFAULT_RESULT_SET_CONCURRENCY);
            if(tmpUCrN.equalsIgnoreCase("CONCUR_READ_ONLY"))
                dbConf.setResultSetConcurrency(ResultSet.CONCUR_READ_ONLY);
            else if (tmpUCrN.equalsIgnoreCase("CONCUR_UPDATABLE"))
                dbConf.setResultSetConcurrency(ResultSet.CONCUR_UPDATABLE);
            else {
                DODS.getLogChannel().write(Logger.DEBUG,"Invalid value for ResultSetConcurrency parameter :"+tmpUCrN);
                dbConf.setResultSetConcurrency(DEFAULT_RESULT_SET_CONCURRENCY);
            }
        }catch(Exception  ex) {
            DODS.getLogChannel().write(Logger.DEBUG,"Use default value for ResultSetType parameter ");
        }
    }

    /**
     * Return value of ResultSetType property
     * @return Value of ResultSetType property
     */
    public int getResultSetConcurrency(){
        return dbConf.getResultSetConcurrency();
    }

    /**
     * Method getDriverDependencies
     *
     * @return   a DriverDependencies
     */
    public DriverDependencies getDriverDependencies() {
        if (null == drivDepsInstance) {
            String  className = getDriverProperty(PARAMNAME_DRIV_DEP_CLASS);
            try {
                if (null != className) {
                    drivDepsInstance = (DriverDependencies)
                        Class.forName(className)
                        .newInstance();
                }
            } catch (Exception  e) {
                DODS.getLogChannel()
                    .write(Logger.ERROR,
                           "Didn't instantiated DriverDepenencies "
                               + className,
                           e);
            } finally {
                if (null == drivDepsInstance)
                    drivDepsInstance = new DriverDependencies() {
                        public boolean isBlobAccessSpecial() { return false;}
                        public byte[] readBlob(ResultSet  rs, String  c){ return null;}
                        public void insertBlob(DBConnection conn,
                                               byte[] cont,
                                               RDBColumn primary,
                                               RDBColumn blobColumn,
                                               String  handle){}
                    };
            }
        }
        return drivDepsInstance;
    }
}
