
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: OracleLogicalDatabase.java,v 1.1 2004/09/03 13:42:52 sinisa Exp $
 */
package com.lutris.appserver.server.sql.oracle;

import java.sql.SQLException ;
import com.lutris.appserver.server.sql.ConnectionAllocator;
import com.lutris.appserver.server.sql.DBQuery;
import com.lutris.appserver.server.sql.DBTransaction;
import com.lutris.appserver.server.sql.ObjectIdAllocator;
import com.lutris.appserver.server.sql.standard.StandardLogicalDatabase;
import com.lutris.appserver.server.sql.DatabaseManagerConfiguration;
import com.lutris.util.Config;
import com.lutris.util.ConfigException;

/**
 * Represents a logical database with Oracle specific capabilities.
 *
 * @see     StandardLogicalDatabase
 * @author  Paul Morgan
 * @since   LBS1.8
 * @version $Revision: 1.1 $
 */
public class OracleLogicalDatabase extends StandardLogicalDatabase {

    /**
     * Creates and configures a single logical database with oracle
     * specific enhancements.
     *
     * @see StandardLogicalDatabase#StandardLogicalDatabase
     * @param dbName
     *   The logical name of the database.
     * @param dbConfig
     *   The configuration object for this logical database.
     * @exception ConfigException
     *   If there is an error in the configuration file.
     * @exception SQLException
     *   If a SQL error occurs.
     */
    public OracleLogicalDatabase(String  dbName, Config dbConfig)
        throws ConfigException, SQLException  {
        super.init(dbName, dbConfig);
    }
    
    /**
     * Creates and configures a single logical database with oracle
     * specific enhancements.
     *
     * @see StandardLogicalDatabase#StandardLogicalDatabase
     * @param dbName
     *   The logical name of the database.
     * @param dbConfig
     *   The configuration object for this logical database.
     * @param dbManagerConf
     *   The configuration object of DatabaseManager.
     * @exception ConfigException
     *   If there is an error in the configuration file.
     * @exception SQLException
     *   If a SQL error occurs.
     */
    public OracleLogicalDatabase(String  dbName, Config dbConfig, DatabaseManagerConfiguration dbManagerConf)
        throws ConfigException, SQLException  {
        super.init(dbName, dbConfig, dbManagerConf);
    }

    /**
     * Default constructor to configure a single logical database.
     * Note that the <CODE>init()</CODE> method must be called to
     * configure the database.
     * 
     * @see StandardLogicalDatabase#StandardLogicalDatabase
     */
    public OracleLogicalDatabase() {
        super();
    }
    
    /**
     * Return the connection allocator.
     *
     * @param connectionConfig
     *   The configuration object for the connection allocator.
     * @exception ConfigException
     *   If there is an error in the configuration file.
     * @return
     *   The connection allocator.
     */
    public ConnectionAllocator loadConnectionAllocator(Config connectionConfig)
        throws ConfigException {
        return new OracleConnectionAllocator(this, connectionConfig);
    }

    /**
     * Return the object id allocator.
     *
     * @param objIdConfig
     *   The configuration object for the object id allocator.
     * @exception ConfigException
     *   If there is an error in the configuration file.
     * @return
     *   The object id allocator.
     */
    public ObjectIdAllocator loadObjectIdAllocator(Config objIdConfig)
        throws ConfigException {
        return new OracleObjectIdAllocator(this, objIdConfig);
    }

    /**
     * Return a transaction for use on this oracle logical database.
     *
     * @return The transaction object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    public DBTransaction createTransaction()
        throws SQLException  {
        return new OracleDBTransaction((OracleDBConnection) connectionAllocator.allocate());
    }

    /**
     * Return a query for use on this logical database.
     *
     * @return The query object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    public DBQuery createQuery()
        throws SQLException  {
        return new OracleDBQuery(connectionAllocator.allocate());
    }

    /**
     * Return a description of the logical database type.
     *
     * @return The type.
     */
    public String  getType() {
        return "Oracle";
    }
}
