
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: DBTransaction.java,v 1.1 2004/09/03 13:42:37 sinisa Exp $
 *
 * formatted with JxBeauty (c) johann.langhofer@nextra.at
 */
package  com.lutris.appserver.server.sql;

import java.sql.SQLException ;

/**
 * Used to perform database transactions.
 *
 * <P>Example - adding a new user:
 * <PRE>
 *   import org.enhydra.dods.DODS;
 *   import com.lutris.appserver.server.sql.*;
 *
 *       DBTransaction transaction =
 *          DODS.getDatabaseManager().createTransaction(DATABASE_NAME);
 *
 *       // NOTE: class CustomerDO implements Transaction { ... }
 *       // NOTE: An Object ID is automatically calculated by the constructor.
 *       CustomerDO customer = new CustomerDO();
 *       customer.setFirstName("Santa");
 *       customer.setLastName("Claus");
 *
 *       // ... set all other CustomerFields ...
 *
 *       //
 *       // Now add the new object to the database.
 *       //
 *       try {
 *           transaction.insert(customer);
 *           transaction.commit();
 *           System.out.println("Object ID is " + customer.get_OId());
 *       }
 *       catch (SQLException e) {
 *           transaction.rollback();
 *           throw e;
 *       }
 *       finally {
 *           transaction.release();
 *       }
 * </PRE>
 *
 * @author  Kyle Clark
 * @since   LBS1.8
 * @version $Revision: 1.1 $
 */
public interface DBTransaction {

    /**
     * Method to update an object in the database.
     *
     * @param transaction
     *   Object that implements transaction interface.
     */
    public void update(Transaction transaction);

    /**
     * Method to delete an object in the database.
     *
     * @param transaction
     *   Object that implements transaction interface.
     */
    public void delete(Transaction transaction);

    /**
     * Method to insert an object in the database.
     *
     * @param transaction
     *   Object that implements transaction interface.
     */
    public void insert(Transaction transaction);

    /**
     * Method to commit upates.
     *
     * @exception java.sql.SQLException If a database access error occurs.
     */
    public void commit() throws SQLException ;

    /**
     * Method to rollback changes.
     *
     * @exception java.sql.SQLException
     *   If a database access error occurs.
     */
    public void rollback() throws SQLException ;

    /**
     * Frees all resources consumed by this transaction
     * Connections are returned to the connection pool.
     * Subsequent transactions via this object,
     * will allocate a new set of resources (i.e. connection).
     */
    public void release();

    /**
     * Exception handeler.  This object is should not be
     * used for subsequent queries if this method returns
     * false.
     *
     * @return boolean True if the exception can be handeled
     *   and the object is still valid, false otherwise.
     */
    public boolean handleException(SQLException  e);

    // compliance with WEBDOCWF begin
 /**
     * Method find a DO in the transaction
     *
     * @param transaction
     *   Object that implements transaction interface.
     * @return DO if the oid was in the transaction, null if it was not
     *
     * WebDocWf extension
     */
    public Transaction getDO (Transaction transaction);

    /**
     * Method find a DO in the transaction
     *
     * @param transaction
     *   Object that implements transaction interface.
     * @param action
     *   if not NONE=0, the DO is found only woth the matching action
     * @return DO if the oid was in the transaction, null if it was not
     *
     * WebDocWf extension
     */
    public Transaction getDO (Transaction transaction, int action);

    // original line
 //
 // compliance with WEBDOCWF end

    /**
     * Method return name of used database
     *
     * @return name of used database
     */
    public String  getDatabaseName();
  
    /**
     * Method set name of used database
     *
     * @param dbName name of used database
     */
    public void setDatabaseName(String  dbName);

    /**
     *
     */
    public void write() throws SQLException ;

    /**
     *
     */
    public void lockDO(Transaction cdo) throws SQLException ;

    /**
     * Return a query for use with this TRANSACTION please!!!
     *
     * @return The query object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    public DBQuery createQuery() throws SQLException ;
    
    /**
     *
     */
    public boolean preventCacheQueries();
}
