
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: DBConnection.java,v 1.1 2004/09/03 13:42:37 sinisa Exp $
 */
package com.lutris.appserver.server.sql;

import java.sql.CallableStatement ;
import java.sql.Connection ;
import java.sql.PreparedStatement ;
import java.sql.ResultSet ;
import java.sql.SQLException ;

/**
 * This interface defines a database connection.
 *
 * @author  Paul Morgan
 * @since   LBS1.8
 * @version $Revision: 1.1 $
 */
public interface DBConnection {

    /**
     * Validates this connection.  Check to see that it
     * is not closed and has been properly allocated.
     *
     * @exception java.sql.SQLException
     *   If it is not valid.
     */
    public void validate() throws SQLException ;

    /**
     * Closes down all query-specific resources.
     *
     * @exception java.sql.SQLException If a database error occurs.
     */
    public void reset() throws SQLException ;

    /**
     * Get a prepared statement given an SQL string.  If the statement is
     * cached, return that statement, otherwise prepare and save in the 
     * cache.
     *
     * @param sql The SQL statement to prepared.
     * @return The prepared statement, which is associated only with this
     *  connection and must not be used once the connection is released.
     * @exception java.sql.SQLException If a SQL error occured compiling the
     *  statement.
     */
    public PreparedStatement  prepareStatement(String  sql)
        throws SQLException ;

    /**
     * Creates a CallableStatement object for calling database 
     * stored procedures. Refer to jdk api refernece.
     * @param sql The SQL statement to be called.
     * @return a new CallableStatement object containing the 
     *  pre-compiled SQL statement.
     * @exception java.sql.SQLException If a database access error occurs
     *  statement.
     */
    public CallableStatement  prepareCall(String  sql)
        throws SQLException ;

    /**
     * Execute a prepared query statement.  Once the query has completed,
     * <CODE>reset()</CODE> should be called.
     *
     * @param preparedStmt The statement to execute.
     * @param msg for logging/debug purposes
     * @return Query result set, do not call close, use reset(),
     * @exception java.sql.SQLException If a SQL error occured executing the
     *  statement.
     */
    public ResultSet  executeQuery(PreparedStatement  preparedStmt,
            String  msg)
        throws SQLException ;

    /**
     * Execute a SQL query statement.  This is a wrapper that adds logging.
     * Once the query has completed, <CODE>reset()</CODE> should be called.
     *
     * @param sql The SQL query statement
     * @return Query result set, do not call close, use reset(),
     * @exception java.sql.SQLException If a SQL error occured executing the
     *  statement.
     */
    public ResultSet  executeQuery(String  sql) throws SQLException ;

    /**
     * Execute a SQL update statement.  This is a wrapper that adds logging.
     * Once the update has completed, <CODE>reset()</CODE> should be called.
     *
     * @param sql
     *   The SQL query statement
     * @return
     *   Either the row count for UPDATE, INSERT, DELETE or 0 for
     *   SQL statements that return nothing.
     * @exception java.sql.SQLException
     *   If a SQL error occured executing the update.
     */
    public int executeUpdate(String  sql) throws SQLException ;

    /**
     * Execute a prepared update statement.  Once the update has completed,
     * <CODE>reset()</CODE> should be called.
     *
     * @param preparedStmt The statement to execute.
     * @param msg for logging/debug purposes
     * @return Either the row count for UPDATE, INSERT, DELETE or 0 for
     *   SQL statements that return nothing.
     * @exception java.sql.SQLException If a SQL error occured executing the
     *  statement.
     */
    public int executeUpdate(PreparedStatement  preparedStmt,
            String  msg)
        throws SQLException ;

    /**
     * Execute a SQL statement that does not return a resultset.  This is a
     * wrapper that adds logging.  Once the query has completed,
     * <CODE>reset()</CODE> should be called.
     *
     * @param sql The SQL query statement
     * @return True if the next result is a ResultSet; false if it is
     *   an update count or there are no more results.
     * @exception java.sql.SQLException If a SQL error occured executing the
     *  statement.
     */
    public boolean execute(String  sql) throws SQLException ;

    /**
     * Check for warnings in a result set.
     *
     * @param resultSet The result set to check for warnings.
     * @exception java.sql.SQLException If a SQL error occured compiling the
     *  statement.
     */
    public void warningCheck(ResultSet  resultSet)
        throws SQLException ;

    /**
     * Return this connection to its allocator.  This object should not be
     * used after calling this function.
     */
    public void release();

    /**
     * Check if a connection is valid after an SQL exception
     * is thrown.  If it is not usable, then the connection is
     * dropped from the connection allocator and closed.  The connection
     * is then no longer usable.
     *
     * @param sqlExcept
     *   the SQL exception that occured.
     * @return true
     *   if the exception does not affect this
     *   connection object.  False otherwise - in which case
     *   this connection object is no longer usable.
     */
    public boolean handleException(SQLException  sqlExcept);

    /**
     * Commit a transaction.
     *
     * @exception java.sql.SQLException If a database access error occurs. 
     */
    public void commit() throws SQLException ;

    /**
     * Autocommit on/off.
     *
     * @param on
     *   false to disable auto commit mode.  True to enable.
     * @exception SQLException
     *   if a database access error occurs. 
     */
    public void setAutoCommit(boolean on) throws SQLException ;

    /**
     * Rollback a transaction.  Should only be used when
     * <A HREF=#setAutoCommit>auto-commit</A> mode
     * has been disabled.
     *
     * @exception SQLException
     *   if a database access error occurs. 
     */
    public void rollback() throws SQLException ;
    
    /**
     * Method called when this connection object is allocated from the
     * connection allocator.
     *
     * @exception SQLException
     *   if <CODE>reset</CODE> had no been called on the previous operation.
     */
    public void allocate() throws SQLException ;
    
    /**
     * Close this connection.  Use by the connection allocator when this
     * object is no longer used.  Errors are ignored.
     */
    public void close();
    
    /**
     * Get the generation number specified when the connection was created.
     *
     * @return The generation number.
     */
    public int getGeneration();

    /**
     * Increment the count of the number of requests against
     * this connection.
     */
    public void incrRequestCount();

    /**
     * Get the database URL.
     * @return The database URL.
     */
    public String  getUrl();

    /**
     * Get the database user name.  Normally user for error messages.
     * @return The database user name.
     */
    public String  getUser();

    /**
     * Get the underlying <code>Connection</code> object.
     * Use with extreme caution.
     * @return the connection object
     */
    public Connection  getConnection();

    /**
     * @return true if this connection is marked to be dropped out
     * of the connection pool and closed.
     */
    public boolean isMarkedForDrop();
    
    /**
     * @return database name of current connection
     *
     */
    public String  getDatabaseName();    
}
