
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: Transaction.java,v 1.1 2004/09/03 13:42:37 sinisa Exp $
 */
package com.lutris.appserver.server.sql;

import java.sql.SQLException ;

/**
 * Interface that must be implemented by objects accessed by DBTransaction.
 *
 * @author  Kyle Clark
 * @version $Revision: 1.1 $
 * @see     DBTransaction
 */
public interface Transaction {

    /**
     * Method to insert a new object into the database.
     *
     * @param conn Database connection.
     * @exception java.sql.SQLException If a database access error
     *  occurs.
     */
    public void executeInsert(DBConnection conn)
        throws SQLException ;

    /**
     * If this object's <code>executeInsert</code> method was
     * called then <code>finalizeInsert</code> is called with
     * the status of the database transaction.  This method
     * allows the data object to perform any post processing
     * if the transaction succeeded or failed.
     *
     * @param success true if the transaction succeeded
     *   and this object was successfully inserted into the database.
     */
    public void finalizeInsert(boolean success);

    /**
     * Method to update contents of object in database.
     *
     * @param conn Database connection.
     * @exception java.sql.SQLException If a database access error
     *  occurs.
     */
    public void executeUpdate(DBConnection conn)
        throws SQLException ;

    /**
     * If this object's <code>executeUpdate</code> method was
     * called then <code>finalizeUpdate</code> is called with
     * the status of the database transaction.
     * For instance the data object may want to
     * increment its version number once it has successfully
     * been commited to the database.  
     *
     * @param success true if the transaction succeeded
     *   and this object was successfully updated in the database.
     */
    public void finalizeUpdate(boolean success);

    /**
     * Method to delete an object from the database.
     *
     * @param conn Database connection.
     * @exception java.sql.SQLException If a database access error
     *  occurs.
     */
    public void executeDelete(DBConnection conn)
        throws SQLException ;

    /**
     * If this object's <code>executeDelete</code> method was
     * called then <code>finalizeDelete</code> is called with
     * the status of the database transaction.
     *
     * @param success true if the transaction succeeded
     *   and this object was successfully deleted from the
     *   database.
     */
    public void finalizeDelete(boolean success);
}
