
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: DBQuery.java,v 1.1 2004/09/03 13:42:37 sinisa Exp $
 */
package com.lutris.appserver.server.sql;

import java.sql.SQLException ;

/**
 * Utility for querying object from a database.  Allocates connections from
 * the database connection pool, ensures the integrity of those
 * connections, and manages result sets after a
 * <A HREF=#query>query</A> is performed.  Returns objects representing
 * data in the database.
 *
 * <P>Example - querying a user:
 * <PRE>
 * 
 *  import org.enhydra.dods.DODS;
 *  import com.lutris.appserver.server.sql.*;
 *
 *      DBQuery dbQuery =
 *          DODS.getDatabaseManager().createQuery(DATABASE_NAME);
 *          
 *      // NOTE: class CustomerQuery implements Query { ... }
 *      CustomerQuery
 *          customerQuery = new CustomerQuery();
 *      String [] loginIds = { "customer1", "customer2" };
 *
 *      try {
 *          for (int idx=0; idx &lt loginIds.length; idx++) {
 *
 *              customerQuery.setQueryLoginId(loginIds[idx]);
 *              dbQuery.query(customerQuery); // query the database
 *
 *              // Print all query results.
 *              CustomerDO customerResult;
 *              while ((customerResult =
 *                     (CustomerDO)dbQuery.next()) != null) {
 *                  System.out.println("Customer name for " + 
 *                                      loginIds[idx] + 
 *                                     " is " + customerResult.getName());
 *              }
 *
 *          }
 *      }
 *      finally {
 *          // 
 *          // Return database connections used by
 *          // this object back to the connection pool
 *          // and free any other resources consumed
 *          // by this object.
 *          // 
 *          dbQuery.release();
 *      }
 *
 * </PRE>
 *
 * @author  Kyle Clark
 * @since   LBS1.8
 * @version $Revision: 1.1 $
 */
public interface DBQuery {

    /**
     * Query the database.
     *
     * @param q
     *   Query interface via which the query will be executed.
     * @exception SQLException
     *   If a database access error occurs. 
     */
    public void query(Query q)
        throws SQLException ;

    /**
     * Returns a new object representing the next result form
     * the query.  
     *
     * @return
     *   New instance of object representing query results.
     *   <I>null</I> is returned when there are no more results.
     * @exception SQLException
     *   If a database access error occurs. 
     * @exception ObjectIdException
     *   If ObjectId was not found.
     */
    public Object  next()
        throws SQLException , ObjectIdException;

    /**
     * Frees all resources consumed by this query.
     * Connections are returned to the connection pool.
     * Subsequent queries via this object,
     * will throw an exception.
     */
    public void release();

    /**
     * Exception handler.  This object is should not be
     * used for subsequent queries if this method returns
     * false.
     *
     * @return
     *   True if the exception can be handled and the object is
     *   still valid, false otherwise.
     */
    public boolean handleException(SQLException  e);

    /**
     * Method to ensure this object is still valid.
     * Once this object has been released it cannot be
     * used any more.
     * 
     * @exception SQLException
     *   If a database access error occurs. 
     */
    public void validate()
        throws SQLException ;
}
