
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: LogicalDatabase.java,v 1.1 2004/09/03 13:42:37 sinisa Exp $
 */
package com.lutris.appserver.server.sql;

import java.sql.SQLException ;
import java.util.Date ;
import com.lutris.util.Config;
import com.lutris.util.ConfigException;

/**
 * Interface for a logical database. Each logical database must
 * provide services for connection and object id allocation.
 *
 * @author  Paul Morgan
 * @since   LBS1.8
 * @version $Revision: 1.1 $
 */
public interface LogicalDatabase {

    /**
     * Initialize the logical database.
     *
     * @exception ConfigException
     *  if there is an error in the configuration.
     * @exception SQLException
     *  if a SQL error occurs.
     */
    public void init(String  dbName, Config dbConfig)
        throws ConfigException, SQLException ;

    /**
     * Return a connection to this logical database.
     *
     * @return The connection.
     * @exception SQLException
     *  if a SQL error occurs.
     */
    public DBConnection allocateConnection()
        throws SQLException ;

    /**
     * Return an object id for this logical database.
     *
     * @return The object id.
     * @exception ObjectIdException
     *  if an object id cannot be allocated.
     */
    public ObjectId allocateObjectId()
        throws ObjectIdException;

    /**
     * Check does oid belong to Object id's range [minOId, currentOId]
     *
     * @param oid
     *   oid which will be checked.
     * @exception ObjectIdException
     *   If a oid does't belong to range.
     */
    public void checkOId(ObjectId oid)  throws ObjectIdException;

    /**
     * Return a transaction for use on this logical database.
     *
     * @return The transaction object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    public DBTransaction createTransaction()
        throws SQLException ;

    /**
     * Return a query for use on this logical database.
     *
     * @return The query object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    public DBQuery createQuery()
        throws SQLException ;

    /**
     * Immediately shutdown the logical database. All connections
     * should be closed and any other cleanup performed.
     */
    public void shutdown();
    
    // ===================================================================
 // The following are primaryily for management purposes...
 // ===================================================================
 /**
     * Return the symbolic name of this logical database.
     *
     * @return The symbolic name.
     */
    public String  getName();

    /**
     * Return a description of the logical database type.
     *
     * @return The type.
     */
    public String  getType();

    /**
     * Return the number of currently active connections.
     *
     * @return The number of connections.
     */
    public int getActiveConnectionCount();

    /**
     * Return the maximum number of connections active at one time.
     *
     * @return The number of connections.
     */
    public int getMaxConnectionCount();

    /**
     * Return the time when the maximum connection count occured.
     *
     * @return The <CODE>Date</CODE> when the maximum connection
     *  count occured.
     */
    public Date  getMaxConnectionCountDate();

    /**
     * Reset the maximum connection count and date.
     */
    public void resetMaxConnectionCount();

    /**
     * Return the number of database requests.
     *
     * @return The number of database requests (queries or transactions).
     */
    public long getRequestCount();
}
