
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: DatabaseManager.java,v 1.1 2004/09/03 13:42:37 sinisa Exp $
 */
package com.lutris.appserver.server.sql;

import java.sql.SQLException ;
import java.util.Date ;
import com.lutris.util.Config;

/**
 * The database management object interface. This class implementing this
 * interface manages the database connections for one application.
 *
 * @version $Revision: 1.1 $
 * @author  Paul Morgan
 */
public interface DatabaseManager {

    /**
     * Flag to enable debug logging of queries and transactions.
     */
    boolean debug = false;

    /**
     * Return main configuration class 
     *
     * @return
     *   main configuration class.
     */
    public Config getConfig();

    /**
     * Allocate a connection to the specified logic database. The
     * connection should be returned to the pool by calling its 
     * <a HREF=com.lutris.appserver.server.sql.DBConnection#release>
     * release()</a> function.  A thread will wait if no connections
     * are available.  Interupted exceptions are converted to errors.
     *
     * @param dbName
     *   Logical name of the database to allocate a connection to.
     * @return The allocated connection object.
     * @exception DatabaseManagerException 
     *   If a nonexistent logical database name is supplied.
     * @exception SQLException
     *   If a SQL error occures.
     */
    public DBConnection allocateConnection(String  dbName) 
        throws DatabaseManagerException, SQLException ;

    /**
     * Allocate a connection to the default logical database. The
     * connection should be returned to the pool by calling its 
     * <a HREF=com.lutris.appserver.server.sql.DBConnection#release>
     * release()</a> function.  A thread will wait if no connections
     * are available.  Interupted exceptions are converted to errors.
     *
     * @return The allocated connection object.
     * @exception DatabaseManagerException 
     *   If a nonexistent default logical database name is supplied.
     * @exception SQLException
     *   If a SQL error occures.
     * @see
     *   #setDefaultDatabase
     */
    public DBConnection allocateConnection() 
        throws DatabaseManagerException, SQLException ;

    /**
     * Allocate an object id from the specified logical database.
     *
     * @param dbName
     *   Logical name of the database from which to obtain an object id.
     * @return The allocated unique OID
     * @exception DatabaseManagerException 
     *   If a nonexistent logical database name is supplied.
     * @exception ObjectIdException 
     *   If a problem (e.g. SQL error) occured in obtaining the OID.
     */
    public ObjectId allocateObjectId(String  dbName) 
        throws DatabaseManagerException, ObjectIdException;

    /**
     * Allocate an object id from the specified logical database.
     *
     * @return The allocated connection object.
     * @exception DatabaseManagerException 
     *   If a nonexistent default logical database has been set.
     * @exception ObjectIdException 
     *   If a problem (e.g. SQL error) occured in obtaining the OID.
     * @see
     *   #setDefaultDatabase
     */
    public ObjectId allocateObjectId() 
        throws DatabaseManagerException, ObjectIdException;

    /**
     * Check does oid belong to Object id's range [minOId, currentOId]
     *
     * @param dbName
     *   Logical name of the database from which to check an object id.
     * @param oid
     *   oid which will be checked.
     * @exception DatabaseManagerException
     *   If a nonexistent logical database name is supplied.
     * @exception ObjectIdException
     *   If a oid does't belong to range.
     */
    public void checkOId(String  dbName, ObjectId oid)
        throws DatabaseManagerException, ObjectIdException;

    /**
     * Check does oid belong to Object id's range [minOId, currentOId] for default database
     *
     * @param oid
     *   oid which will be checked.
     * @exception DatabaseManagerException
     *   If a nonexistent default logical database has been set.
     * @exception ObjectIdException
     *   If a oid does't belong to range.
     */
    public void checkOId(ObjectId oid)
        throws DatabaseManagerException, ObjectIdException;

    /**
     * Create a transaction object for the specified logical database.
     *
     * @param dbName
     *   Logical name of the database from which to obtain a transaction.
     * @return The transaction
     * @exception DatabaseManagerException 
     *   If a nonexistent or invalid logical database name is supplied.
     * @exception SQLException 
     *   If a problem occured creating the transaction.
     */
    public DBTransaction createTransaction(String  dbName) 
        throws DatabaseManagerException, SQLException ;

    /**
     * Create a transaction object for the default logical database.
     *
     * @return The transaction
     * @exception DatabaseManagerException 
     *   If a nonexistent default logical database has been set.
     * @exception SQLException 
     *   If a problem occured creating the transaction.
     * @see
     *   #setDefaultDatabase
     */
    public DBTransaction createTransaction() 
        throws DatabaseManagerException, SQLException ;

    /**
     * Create a query object for the specified logical database.
     *
     * @param dbName
     *   Logical name of the database from which to obtain a query.
     * @return The query
     * @exception DatabaseManagerException 
     *   If a nonexistent or invalid logical database name is supplied.
     * @exception SQLException 
     *   If a problem occured creating the query.
     */
    public DBQuery createQuery(String  dbName) 
        throws DatabaseManagerException, SQLException ;

    /**
     * Create a query object for the default logical database.
     *
     * @return The query
     * @exception DatabaseManagerException 
     *   If a nonexistent default logical database has been set.
     * @exception SQLException 
     *   If a problem occured creating the query.
     * @see
     *   #setDefaultDatabase
     */
    public DBQuery createQuery() 
        throws DatabaseManagerException, SQLException ;

    /**
     * Return a logical database type for the default logical database.
     *
     * @param dbName
     *   Logical name of the database from which to obtain a query.
     * @return logical database type
     * @exception DatabaseManagerException
     *   If a nonexistent default logical database has been set.
     * @exception SQLException
     *   If a problem occured creating the query.
     * @see
     *   #setDefaultDatabase
     */
    public String  logicalDatabaseType(String  dbName)
        throws DatabaseManagerException, SQLException ;

    /**
     * Return a logical database type for the default logical database.
     *
     * @return logical database type
     * @exception DatabaseManagerException
     *   If a nonexistent default logical database has been set.
     * @exception SQLException
     *   If a problem occured creating the query.
     * @see
     *   #setDefaultDatabase
     */
    public String  logicalDatabaseType()
        throws DatabaseManagerException, SQLException ;

    /**
     * Set the default logical database. This should be used with
     * caution, but it makes allocating connections easier.
     *
     * @param dbName
     *   The default logical dabase name.
     * @exception DatabaseManagerException 
     *   If a nonexistent or illegal logical database name is supplied.
     */
    public void setDefaultDatabase(String  dbName)
        throws DatabaseManagerException;

    /**
     * Return default database name (given in config file)
     * @return
     *   default database name
     */
    public String  getDefaultDB();

    /**
     * Shutdown the database manager. All logical databases will be
     * shutdown and all connection closed.
     */
    public void shutdown();
    // ====================================================================
 // The following are primarily for management purposes...
 // ====================================================================
 /**
     * Returns the list of managed logical databases.
     * 
     * @return  List of logical database names.
     */
    public String [] getLogicalDatabaseNames();

    /**
     * Returns a description of the logical database type.
     * 
     * @param dbName
     *   The logical database name.
     * @return
     *   A text description of the logical database type.
     * @exception DatabaseManagerException 
     *   If a nonexistent logical database name is supplied.
     */
    public String  getType(String  dbName)
        throws DatabaseManagerException;
    
    /**
     * Returns the number of requests made to the database since startup time.
     * 
     * @param dbName
     *   The logical database name.
     * @return
     *   The number of database requests since the server started.
     * @exception DatabaseManagerException 
     *   If a nonexistent logical database name is supplied.
     */
    public long getRequestCount(String  dbName)
        throws DatabaseManagerException;
    
    /**
     * Returns the number of currently active connections for the
     * supplied logical database name.
     * If not implemented, then -1 is returned.
     * 
     * @param dbName
     *   The logical database name.
     * @return
     *   The number of currently active connections.
     * @exception DatabaseManagerException 
     *   If a nonexistent logical database name is supplied.
     */
    public int getActiveConnectionCount(String  dbName)
        throws DatabaseManagerException;
   
    /**
     * Returns the maximum number of concurent connections that existed
     * at any time since this object was created, or
     * <CODE>resetMaxConnectionCount()</CODE> was called.
     * This is a historical highwater mark. 
     * If not implemented, then -1 is returned.
     *
     * @param dbName
     *   The logical database name.
     * @return
     *   The highwater mark for number of connections, or -1.
     * @exception DatabaseManagerException 
     *   If a nonexistent logical database name is supplied.
     */
    public int getMaxConnectionCount(String  dbName)
        throws DatabaseManagerException;

    /**
     * Returns the time when the maximum refered to by 
     * <CODE>maxConnectionCount()</CODE> occured.
     * If not implemented, then null is returned.
     *
     * @param dbName
     *   The logical database name.
     * @return
     *   The Date of when the maximum number of connections occured.
     * @exception DatabaseManagerException 
     *   If a nonexistent logical database name is supplied.
     */
    public Date  getMaxConnectionCountDate(String  dbName)
        throws DatabaseManagerException;

    /**
     * Reset the maximum connection count. See
     * <CODE>maxConnectionCount()</CODE>. The highwater mark should be
     * reset to the current number of connections.
     *
     * @param dbName
     *   The logical database name.
     * @exception DatabaseManagerException 
     *   If a nonexistent logical database name is supplied.
     */
    public void resetMaxConnectionCount(String  dbName)
        throws DatabaseManagerException;
    
    public LogicalDatabase findLogicalDatabase(String  dbName)
        throws DatabaseManagerException;
}
