
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 */
package org.enhydra.dods.statistics;

import java.util.Date ;

/**
 * This interface provides information about table and its statistics.
 *
 * @author    Tanja Jovanovic
 * @author    Nenad Vico
 * @author    Zorica Suvajdzin
 * @version   2.0  15.06.2003.
 */
public interface Statistics {

    /**
     * When there are no caches (data object and query caches), statistics is
     * TABLE_STATISTICS (contains only information about tables).
     */
    public static final int TABLE_STATISTICS = 0;

    /**
     * When there is only data object cache (there are no query caches),
     * statistics is CACHE_STATISTICS (contains information about tables and
     * data object cache).
     */
    public static final int CACHE_STATISTICS = 1;

    /**
     * When there are both data object and query caches, statistics is
     * QUERY_CACHE_STATISTICS (contains information about tables, data object
     * and query caches).
     */
    public static final int QUERY_CACHE_STATISTICS = 2;

    /**
     * Returns type of the statistics (TABLE_STATISTICS, CACHE_STATISTICS, or
     * QUERY_CACHE_STATISTICS).
     *
     * @return Type of statistics (0 if statistics is TABLE_STATISTICS, 1 if
     * statistics is CACHE_STATISTICS, or 2 if statistics is
     * QUERY_CACHE_STATISTICS.
     */
    public int getStatisticsType();

    /**
     * Returns number of insert statements performed on the table.
     *
     * @return Number of insert statements performed on the table.
     */
    public int getInsertNum();

    /**
     * Sets number of insert statements performed on the table.
     *
     * @param newInsertNum New number of insert statements performed on the
     * table.
     */
    public void setInsertNum(int newInsertNum);

    /**
     * Increases number of insert statements performed on the table.
     */
    public void incrementInsertNum();

    /**
     * Returns number of update statements performed on the table.
     *
     * @return Number of update statements performed on the table.
     */
    public int getUpdateNum();

    /**
     * Sets number of update statements performed on the table.
     *
     * @param newUpdateNum New number of update statements performed on the
     * table.
     */
    public void setUpdateNum(int newUpdateNum);

    /**
     * Increases number of update statements performed on the table.
     */
    public void incrementUpdateNum();

    /**
     * Returns number of delete statements performed on the table.
     *
     * @return Number of delete statements performed on the table.
     */
    public int getDeleteNum();

    /**
     * Sets number of delete statements performed on the table.
     *
     * @param newDeleteNum New number of delete statements performed on the
     * table.
     */
    public void setDeleteNum(int newDeleteNum);

    /**
     * Increases number of delete statements performed on table.
     */
    public void incrementDeleteNum();

    /**
     * Returns number of DML operations (inserts, updates and deletes) performed
     * on the table.
     *
     * @return Number of DML operations performed on the table.
     */
    public int getDMLNum();

    /**
     * Returns number of lazy loadings performed on the table.
     *
     * @return Number of lazy loadings performed on the table.
     */
    public int getLazyLoadingNum();

    /**
     * Sets number of lazy loadings performed on the table.
     *
     * @param newLazyLoadingNum New number of lazy loadings performed on the
     * table.
     */
    public void setLazyLoadingNum(int newLazyLoadingNum);

    /**
     * Increases number of lazy loadings performed on the table for one.
     */
    public void incrementLazyLoadingNum();

    /**
     * Returns time when the statistics was started.
     *
     * @return Time when the statistics was started.
     */
    public Date  getStartTime();

    /**
     * Sets time when the statistics starts.
     *
     * @param startTime Time when the statistics starts.
     */
    public void setStartTime(Date  startTime);

    /**
     * Returns time when the statistics was stopped.
     *
     * @return Time when the statistics was stopped.
     */
    public Date  getStopTime();

    /**
     * Sets time when the statistics stops.
     *
     * @param stopTime time when the statistics stops.
     */
    public void setStopTime(Date  stopTime);

    /**
     * Sets stop time to current time.
     */
    public void stopTime();

    /**
     * Returns total number of non-oid queries performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     *
     * @return Total number of non-oid queries performed on the table.
     */
    public int getQueryNum();

    /**
     * Sets total number of non-oid queries performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     *
     * @param newQueryNum New total number of non-oid queries performed on
     * the table.
     */
    public void setQueryNum(int newQueryNum);
   
    /**
     * Increases total number of non-oid queries performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     */
    public void incrementQueryNum();
   
    /**
     * Returns total number of queries by oid performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     *
     * @return Total number of queries by oid performed on the table.
     */
    public int getQueryByOIdNum();

    /**
     * Sets total number of queries by oid performed on the table.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     *
     * @param newQueryByOIdNum New total number of queries by oid performed on
     * the table.
     */
    public void setQueryByOIdNum(int newQueryByOIdNum);

    /**
     * Increases total number of queries by oid performed on the table for one.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     */
    public void incrementQueryByOIdNum();

    /**
     * Returns average time needed for executing non-oid query.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     *
     * @return Average time needed for executing non-oid query.
     */
    public int getQueryAverageTime();

    /**
     * Updates average time needed for executing non-oid queries.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid. Non-oid query is any other query.
     *
     * @param newTime New query time in miliseconds.
     */
    public void updateQueryAverageTime(int newTime);

    /**
     * Returns average time needed for executing query by oid.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     *
     * @return Average time needed for executing oid query.
     */
    public int getQueryByOIdAverageTime();

    /**
     * Updates average time for executing OId queries and increments number
     * of them by paramether <code>no</code>.
     * Query by oid is query which "where" clause contains request for DO with
     * specified oid.
     *
     * @param newTime New query time in miliseconds for no queries by OId.
     * @param no Number of queries by OId.
     */
    public void updateQueryByOIdAverageTime(int newTime, int no);

    /**
     * Clears DO, simple query and complex query statistics.
     */
    public void clear();

    /**
     * Returns query statistics for :
     * DO (data object) cache when cache has value 0
     *      (value rg.enhydra.dods.cache.CacheConstants.DATA_CACHE)
     * simple query cache when cache has value 1
     *      (value org.enhydra.dods.cache.CacheConstants.SIMPLE_QUERY_CACHE)
     * complex query cache when cache has value 2
     *      (org.enhydra.dods.cache.CacheConstants.COMPLEX_QUERY_CACHE)
     *
     * @param type Value 0 (org.enhydra.dods.cache.CacheConstants.DATA_CACHE)
     * for DO (data object) cache,
     * value 1 (org.enhydra.dods.cache.CacheConstants.SIMPLE_QUERY_CACHE) for
     * simple query cache and value 2
     * (org.enhydra.dods.cache.CacheConstants.COMPLEX_QUERY_CACHE) for complex
     * query cache.
     * @return Query statistics for DO (data object) cache, simple query or
     * complex query cache.
     */
    public CacheStatistics getCacheStatistics(int type);
}
