
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: ObjectId.java,v 1.1 2004/09/03 13:42:37 sinisa Exp $
 */
package com.lutris.appserver.server.sql;

import java.io.Serializable ;
import java.math.BigDecimal ;
import java.math.BigInteger ;

/**
 * Represents an object id used by LBS data objects. The object id
 * must be unique. The combination of database URL and object id
 * constitutes a GUID. The maximum value of an object id is
 * DECIMAL(19,0)
 *
 * @version $Revision: 1.1 $
 * @author  Kyle Clark
 * @since   LBS1.7
 * @see     ObjectIdAllocator
 */
public class ObjectId implements Serializable  {

    /**
     * The value associated with this object id.
     */
    private BigDecimal  value;

    /**
     * The value of one.
     */
    static final public BigDecimal  ONE = new BigDecimal ("1");

    /**
     * The maximum value that can be associated with an object
     * id in LBS - DECIMAL(19,0).
     */
    static final public BigDecimal  MAX = new BigDecimal ("9999999999999999999");

    /**
     * Translates a string containing one or more integers
     * of the specified radix into an ObjectID.  The string
     * may not represet a negative number.
     *
     * @param val
     *   The string representation of the number.
     *   The character to digit mapping is provided by
     *   Character.digit()
     * @param radix
     *   Must be between Character.MIN_RADIX(2) and
     *   Character.MAX_RADIX(36).
     * @exception java.lang.NumberFormatException
     *   If the string representation contains invalid characters.
     * @exception ObjectIdException
     *   If val represents a negative number.
     */
    public ObjectId(String  val, int radix)
        throws ObjectIdException, NumberFormatException  {
        value = new BigDecimal (new BigInteger (val, radix), 0);
        if (value.signum() < 0) {
            throw new ObjectIdException("Object IDs cannot be negative.");
        }
        if (value.compareTo(MAX) > 0) {
            throw new ObjectIdException("Object IDs cannot exceed "
                    + MAX.toString());
        }
    }

    /**
     * Translates a string containing one or more decimal digits
     * into an ObjectID.  
     *
     * @param val
     *   The string representation of the decimal number that
     *   
     *   The character to digit mapping is provided by
     *   Character.digit()
     * @exception java.lang.NumberFormatException
     *   If the string representation contains invalid characters.
     * @exception ObjectIdException
     *   If val represents a negative number.
     */
    public ObjectId(String  val)
        throws ObjectIdException, NumberFormatException  {
        value = new BigDecimal (new BigInteger (val), 0);
        if (value.signum() < 0) {
            throw new ObjectIdException("Object IDs cannot be negative.");
        }
        if (value.compareTo(MAX) > 0) {
            throw new ObjectIdException("Object IDs cannot exceed "
                    + MAX.toString());
        }
    }

    /**
     * Translates a long into an ObjectID.  
     *
     * @param val
     *   The value to assign to the object id.
     * @exception ObjectIdException
     *   If val is a negative number.
     */
    public ObjectId(long val)
        throws ObjectIdException {
        if (val < 0) {
            throw new ObjectIdException("Object IDs cannot be negative.");
        }
        value = BigDecimal.valueOf(val);
        if (value.compareTo(MAX) > 0) {
            throw new ObjectIdException("Object IDs cannot exceed "
                    + MAX.toString());
        }
    }

    /**
     * Creates and object id whose value is the same as val.
     *
     * @param val
     *   The value to assign to the object id.
     * @exception ObjectIdException
     *   If val is a negative number or the scale of val
     *   is greater than zero.
     */
    public ObjectId(BigDecimal  val)
        throws ObjectIdException {
        if (val.signum() < 0) {
            throw new ObjectIdException("Object IDs cannot be negative.");
        }
        if (val.scale() > 0) {
            throw new ObjectIdException("Object IDs cannot have a scale "
                    + "greater than zero.");
        }
        if (val.compareTo(MAX) > 0) {
            throw new ObjectIdException("Object IDs cannot exceed "
                    + MAX.toString());
        }
        value = val;
    }

    /**
     * Returns an object id whose value is (this+val).
     *
     * @param val
     *   The value to add to this object.
     * @exception ObjectIdException
     *   If the result of the addition would result
     *   in an object id that exceeds the
     *   <a HREF=#MAX>maximum</a> object id
     *   size.
     */
    public ObjectId add(ObjectId val)
        throws ObjectIdException {
        return new ObjectId(value.add(val.toBigDecimal()));
    }

    /**
     * Returns an object id whose value is (this+val).
     *
     * @param val
     *   The value to add to this object.
     * @exception ObjectIdException
     *   If the result of the addition would result
     *   in an object id that exceeds the
     *   <a HREF=#MAX>maximum</a> object id
     *   size.
     */
    public ObjectId add(long val)
        throws ObjectIdException {
        return new ObjectId(value.add(BigDecimal.valueOf(val)));
    }

    /**
     * Returns an object id whose value is (this+1)
     *
     * @exception ObjectIdException
     *   If the result of the addition would result
     *   in an object id that exceeds the
     *   <a HREF=#MAX>maximum</a> object id
     *   size.
     */
    public ObjectId increment()
        throws ObjectIdException {
        return new ObjectId(value.add(ONE));
    }

    /**
     * Returns a big decimal representation of the object id.
     */
    public BigDecimal  toBigDecimal() {
        return value;
    }

    /**
     * Test if this object id is equal to another object id.
     */
    public boolean equals(ObjectId oid) {
        return (oid.toBigDecimal().compareTo(value) == 0);
    }

    /**
     * Returns a hash code for this object id. 
     * 
     * @return a hash code for this object id. 
     */
    public int hashCode() {
        return toString().hashCode();
    }

    /**
     * String representation of this object id.
     */
    public String  toString() {
        return value.toString();
    }
}
