
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: ConnectionAllocator.java,v 1.1 2004/09/03 13:42:35 sinisa Exp $
 */
package com.lutris.appserver.server.sql;

import java.util.Date ;

/**
 * Defines the connection allocator. A connection allocator is an integral
 * part of a logical database implementation. It manages a pool of database
 * connections.
 * 
 * @author  Paul Morgan
 * @since   LBS1.8
 * @version $Revision: 1.1 $
 */
public interface ConnectionAllocator {

    /**
     * Allocate a connection either from the managed pool or create a new.
     *
     * @exception java.sql.SQLException
     *   if a SQL error occures.
     */
    DBConnection allocate() throws java.sql.SQLException ;

    /**
     * Return a connection to the pool.  If it is of an old generation,
     * close and drop.
     *
     * @param dbConnection The connection object to return.
     */
    void release(DBConnection dbConnection);

    /**
     * Used to drop a connection from the pool. The connection
     * should not be re-allocated
     *
     * @param dbConnection
     *   The connection object to drop.
     */
    void drop(DBConnection dbConnection);
    
    /**
     * Drop all the connection in the pool immediately.
     */
    void dropAllNow();

    /**
     * Return the number of currently active (allocated) connections.
     *
     * @return The number of connections.
     */
    public int getActiveCount();

    /**
     * Return the maximum number of connections active at one time.
     *
     * @return The number of connections.
     */
    public int getMaxCount();

    /**
     * Return the time when the maximum connection count occured.
     *
     * @return The <CODE>Date</CODE> when the maximum connection
     *  count occured.
     */
    public Date  getMaxCountDate();

    /**
     * Reset the maximum connection count and date.
     */
    public void resetMaxCount();

    /**
     * Return the number of database requests made on any of the
     * connection allocated from this managed pool. Implementations
     * of the connection allocator must provide a mechanism for
     * <CODE>DBQuery</CODE> and <CODE>DBTransaction</CODE> objects
     * to count requests.
     *
     * @return The number of database requests (queries or transactions).
     */
    public long getRequestCount();
    
    /**
     * @return database name of current connection
     *
     */
    public String  getDatabaseName();    
    
}
