package live.attach.ui.roomactivity.participant;

import android.content.Context;
import android.graphics.Color;
import android.os.Build;
import android.support.annotation.AttrRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.support.annotation.StyleRes;
import android.util.AttributeSet;
import android.view.View;

import java.util.List;

import live.attach.repackaged.io.reactivex.disposables.Disposable;
import live.attach.lib.live.attach.repackaged.io.reactivex.android.schedulers.AndroidSchedulers;
import live.attach.sdk.AttachSdkInternal;
import live.attach.sdk.R;
import live.attach.domain.model.application.AttachProperties;
import live.attach.domain.model.participant.Participant;
import live.attach.ui.ViewUtils;
import live.attach.ui.roomactivity.participant.active.ActiveParticipantsView;
import live.attach.ui.roomactivity.participant.single.SingleParticipantView;
import live.attach.ui.view.AttachBrandedView;

public class ParticipantsView extends AttachBrandedView {
    private SingleParticipantView singleParticipantView;
    private ActiveParticipantsView activeParticipantsView;

    private AttachSdkInternal attachSdkInternal;
    private Disposable subscription;


    public static int getActiveParticipantsSize(Context context) {
        int maxSpace = ViewUtils.isPortrait(context) ? ViewUtils.getScreenWidth() : ViewUtils.getScreenHeight();
        float paddingRight = ViewUtils.convertDpToPixel(64f);
        maxSpace = (int) (maxSpace - paddingRight);
        int singleParticipantSize = (int) context.getResources().getDimension(R.dimen.participant_size);

        return maxSpace / singleParticipantSize - 1;
//        return 3;
    }

    public ParticipantsView(@NonNull Context context) {
        super(context);
        init(context, null);
    }

    public ParticipantsView(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }

    public ParticipantsView(@NonNull Context context, @Nullable AttributeSet attrs, @AttrRes int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context, attrs);
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public ParticipantsView(@NonNull Context context, @Nullable AttributeSet attrs, @AttrRes int defStyleAttr, @StyleRes int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        init(context, attrs);
    }

    private void init(Context context, AttributeSet attrs) {
        attachSdkInternal = (AttachSdkInternal) getAttachActivity().getAttachSdk();
        setClipChildren(false);
        setClipToPadding(false);
        View root = inflate(getContext(), R.layout.roomactivity_participants_view, this);
        singleParticipantView = root.findViewById(R.id.singleParticipantView);
        activeParticipantsView = root.findViewById(R.id.activeParticipantsView);
    }

    @Override
    protected void applyBranding(AttachProperties attachProperties) {
        int bgColor = Color.parseColor(attachProperties.getParticipantsFocusBackgroundColor().value);
        activeParticipantsView.setBackgroundColor(bgColor);
    }

    public void setOnParticipantClick(OnParticipantClickListener listener) {
        singleParticipantView.setOnParticipantClickListener(listener);
        activeParticipantsView.setOnParticipantClickListener(listener);
    }

    public void setOnShowMoreClickListener(View.OnClickListener listener) {
        activeParticipantsView.setOnShowMoreClickListener(listener);
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        subscription = attachSdkInternal
            .getRoomParticipants()
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(this::setParticipants);
    }

    @Override
    protected void onDetachedFromWindow() {
        subscription.dispose();
        super.onDetachedFromWindow();
    }

    public void setParticipants(List<Participant> participants) {
        if (participants.size() == 1) {
            activeParticipantsView.setVisibility(INVISIBLE);
            singleParticipantView.setVisibility(VISIBLE);
            singleParticipantView.setParticipant(participants.get(0));
        } else {
            activeParticipantsView.setVisibility(VISIBLE);
            singleParticipantView.setVisibility(INVISIBLE);

            int maxAmount = getActiveParticipantsSize(getContext());
            List<Participant> activeParticipants = participants.size() < maxAmount ? participants : participants.subList(0, maxAmount);

            activeParticipantsView.setParticipants(activeParticipants, participants.size() - maxAmount);
        }
    }
}