package live.attach.ui.roomactivity.layout;

import android.content.res.Configuration;
import android.support.annotation.NonNull;
import android.support.v4.view.ViewPager;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import org.webrtc.SurfaceViewRenderer;

import live.attach.sdk.AttachOverlayActivity;
import live.attach.sdk.R;
import live.attach.domain.model.video.VideoCall;
import live.attach.ui.ViewUtils;
import live.attach.ui.bottomsheet.viewpager.BottomSheetUtils;
import live.attach.ui.bottomsheet.viewpager.ViewPagerBottomSheetBehavior;
import live.attach.ui.roomactivity.video.CameraPermissionsView;
import live.attach.ui.roomactivity.video.RemoteVideoView;
import live.attach.ui.roomactivity.video.VideoOverlayView;
import live.attach.ui.view.common.SurfaceViewContainer;

public class RoomActivityLayoutManager {
    private AttachOverlayActivity activity;

    private SurfaceViewRenderer localSurface;
    private SurfaceViewRenderer remoteSurface;

    private ViewGroup portraitVideoContainer;
    private ViewGroup landscapeVideoContainer;

    private SurfaceViewContainer portraitMainSurfaceContainer;
    private SurfaceViewContainer portraitSmallSurfaceContainer;
    private SurfaceViewContainer landscapeLeftSurfaceContainer;
    private SurfaceViewContainer landscapeRightSurfaceContainer;

    private RemoteVideoView remoteVideoView;
    private CameraPermissionsView cameraPermissionsView;
    private VideoOverlayView videoOverlayView;
    private ViewPagerBottomSheetBehavior bottomSheetBehavior;

    public RoomActivityLayoutManager(AttachOverlayActivity activity) {
        this.activity = activity;

        remoteVideoView = activity.findViewById(R.id.remoteVideoView);
        cameraPermissionsView = activity.findViewById(R.id.cameraPermissionsView);
        videoOverlayView = activity.findViewById(R.id.videoOverlayControls);

        localSurface = activity.findViewById(R.id.videoLocal);
        remoteSurface = remoteVideoView.getSurfaceRenderer();

        portraitVideoContainer = activity.findViewById(R.id.videoLocalContainer);
        landscapeVideoContainer = activity.findViewById(R.id.landscapeVideoContainer);

        landscapeLeftSurfaceContainer = activity.findViewById(R.id.landscapeLeftSurfaceContainer);
        landscapeRightSurfaceContainer = activity.findViewById(R.id.landscapeRightSurfaceContainer);
        portraitMainSurfaceContainer = activity.findViewById(R.id.portraitMainSurfaceContainer);
        portraitSmallSurfaceContainer = activity.findViewById(R.id.portraitSmallSurfaceContainer);

        bottomSheetBehavior = ViewPagerBottomSheetBehavior.from(activity.findViewById(R.id.roomactivity_overlay));
        bottomSheetBehavior.setBottomSheetCallback(new ViewPagerBottomSheetBehavior.BottomSheetCallback() {
            @Override
            public void onStateChanged(@NonNull View bottomSheet, int newState) {
            }

            @Override
            public void onSlide(@NonNull View bottomSheet, float slideOffset) {
                if (activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
                    int height = landscapeVideoContainer.getHeight() - (int) ViewUtils.convertDpToPixel(130);
                    int offset = (int) (height * slideOffset);
                    landscapeVideoContainer.setTranslationY(-offset);
                }
            }
        });
        ViewPager viewPager = activity.findViewById(R.id.overlayContentView_viewPager);
        BottomSheetUtils.setupViewPager(viewPager);
    }

    public ViewPagerBottomSheetBehavior bottomSheetBehavior() {
        return bottomSheetBehavior;
    }

    public void onCreate() {
        if (activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            makeLandscape();
        }
    }

    public void onConfigurationChanged(Configuration newConfig) {
        ViewUtils.hideKeyboard(activity);
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            makeLandscape();
        } else if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
            makePortrait();
        }
    }

    public boolean isVideoVisible() {
        return portraitVideoContainer.getVisibility() == View.VISIBLE;
    }

    private void makeLandscape() {
        boolean localVideoVisible = localSurface.isShown();

        activity.findViewById(R.id.bottomSheetTop).setVisibility(View.GONE);
        bottomSheetBehavior.setPeekHeight((int) ViewUtils.convertDpToPixel(130));

        ((ViewGroup) localSurface.getParent()).removeView(localSurface);
        ((ViewGroup) remoteSurface.getParent()).removeView(remoteSurface);

        localSurface.setLayoutParams(new FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        ));
        remoteSurface.setLayoutParams(new FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        ));

        landscapeRightSurfaceContainer.addView(localSurface);
        landscapeLeftSurfaceContainer.addView(remoteSurface);

        portraitVideoContainer.setVisibility(View.GONE);

        ((ViewGroup) videoOverlayView.getParent()).removeView(videoOverlayView);
        landscapeVideoContainer.addView(videoOverlayView);

        ((ViewGroup) cameraPermissionsView.getParent()).removeView(cameraPermissionsView);
        landscapeVideoContainer.addView(cameraPermissionsView);

        if (localVideoVisible) {
            landscapeVideoContainer.setVisibility(View.VISIBLE);
        }
    }

    private void makePortrait() {
        boolean localVideoVisible = localSurface.isShown();
        boolean remoteVideoVisible = remoteSurface.isShown();

        activity.findViewById(R.id.bottomSheetTop).setVisibility(View.VISIBLE);
        bottomSheetBehavior.setPeekHeight((int) ViewUtils.convertDpToPixel(340));

        ((ViewGroup) localSurface.getParent()).removeView(localSurface);
        ((ViewGroup) remoteSurface.getParent()).removeView(remoteSurface);

        localSurface.setLayoutParams(new FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        ));
        remoteSurface.setLayoutParams(new FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        ));

        portraitMainSurfaceContainer.addView(localSurface);
        portraitSmallSurfaceContainer.addView(remoteSurface);

        landscapeVideoContainer.setVisibility(View.GONE);

        ((ViewGroup) videoOverlayView.getParent()).removeView(videoOverlayView);
        portraitVideoContainer.addView(videoOverlayView);

        ((ViewGroup) cameraPermissionsView.getParent()).removeView(cameraPermissionsView);
        portraitVideoContainer.addView(cameraPermissionsView);

        if (localVideoVisible) {
            portraitVideoContainer.setVisibility(View.VISIBLE);
        }
        if (remoteVideoVisible) {
            swapViews(localSurface, remoteSurface);
        }
    }

    public void showRemoteVideo(VideoCall call) {
        //LOCAL
        showLocalVideo();

        //REMOTE
        remoteVideoView.setVisibility(View.VISIBLE);
        remoteVideoView.setParticipant(call.participant);

        landscapeLeftSurfaceContainer.setVisibility(View.VISIBLE);
        landscapeVideoContainer.setVisibility(View.VISIBLE);

        // show remote as main view
        if (activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT) {
            if (localSurface.getParent() == portraitMainSurfaceContainer) {
                swapViews(localSurface, remoteSurface);
            }
        }
    }

    public void hideVideo() {
        portraitVideoContainer.setVisibility(View.GONE);
        remoteVideoView.setVisibility(View.GONE);

        landscapeVideoContainer.setVisibility(View.GONE);
        landscapeLeftSurfaceContainer.setVisibility(View.GONE);

        // bring local back to the main area
        if (activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT) {
            if (localSurface.getParent() != portraitMainSurfaceContainer) {
                swapViews(localSurface, remoteSurface);
            }
        }
    }

    public void showLocalVideo() {
        boolean showLogo;

        if (activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            showLogo = landscapeVideoContainer.getVisibility() != View.VISIBLE;
            landscapeVideoContainer.setVisibility(View.VISIBLE);
        } else {
            showLogo = portraitVideoContainer.getVisibility() != View.VISIBLE;
            portraitVideoContainer.setVisibility(View.VISIBLE);
        }
        if (showLogo) videoOverlayView.showLogo();
    }

    private static void swapViews(View view1, View view2) {
        ViewGroup parentView1 = ((ViewGroup) view1.getParent());
        ViewGroup parentView2 = ((ViewGroup) view2.getParent());

        parentView1.removeView(view1);
        parentView2.removeView(view2);

        view1.setLayoutParams(new FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        ));
        view2.setLayoutParams(new FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        ));

        parentView1.addView(view2);
        parentView2.addView(view1);
    }
}
