package live.attach.ui.roomactivity.chat;

import android.content.Context;
import android.os.Build;
import android.support.annotation.RequiresApi;
import android.support.v7.widget.AppCompatImageView;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.widget.EditText;

import java.util.ArrayList;
import java.util.List;

import live.attach.repackaged.io.reactivex.disposables.Disposable;
import live.attach.lib.live.attach.repackaged.io.reactivex.android.schedulers.AndroidSchedulers;
import live.attach.sdk.AttachSdkInternal;
import live.attach.sdk.R;
import live.attach.domain.model.application.AttachProperties;
import live.attach.domain.model.application.Color;
import live.attach.domain.model.chat.Message;
import live.attach.ui.view.AttachBrandedView;
import live.attach.ui.view.common.EmptyRecyclerView;

public class ChatView extends AttachBrandedView {
    private EmptyRecyclerView messagesList;
    private MessagesAdapter messagesAdapter;
    private EditText messageBody;
    private View sendMessageContainer;
    private AppCompatImageView sendButton;

    private AttachSdkInternal attachSdkInternal;
    private Disposable subscription;

    public ChatView(Context context) {
        super(context);
        init(context);
    }

    public ChatView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(context);
    }

    public ChatView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context);
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public ChatView(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        init(context);
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        subscription = attachSdkInternal
            .getRoomMessages()
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(this::setMessages);
    }

    @Override
    protected void onDetachedFromWindow() {
        subscription.dispose();
        super.onDetachedFromWindow();
    }

    private void init(Context context) {
        attachSdkInternal = (AttachSdkInternal) getAttachActivity().getAttachSdk();

        setFocusable(true);
        setFocusableInTouchMode(true);

        View root = inflate(getContext(), R.layout.roomactivity_chat_view, this);
        messagesList = root.findViewById(R.id.messagesList);
        sendButton = root.findViewById(R.id.sendBtn);
        messageBody = root.findViewById(R.id.messageBody);
        sendMessageContainer = root.findViewById(R.id.sendMessageContainer);

        sendButton.setOnClickListener(view -> sendMessage());

        messagesAdapter = new MessagesAdapter(getDefaultStyles());
        LinearLayoutManager linearLayoutManager = new LinearLayoutManager(context, LinearLayoutManager.VERTICAL, true);
        messagesList.setEmptyView(findViewById(R.id.emptyChatView));
        messagesList.setLayoutManager(linearLayoutManager);
        messagesList.setAdapter(messagesAdapter);

        EndlessRecyclerViewScrollListener scrollListener = new EndlessRecyclerViewScrollListener(linearLayoutManager) {
            @Override
            public void onLoadMore(int page, int totalItemsCount, RecyclerView view) {
                getMoreMessages();
            }
        };
        messagesList.addOnScrollListener(scrollListener);
    }

    @Override
    protected void applyBranding(AttachProperties attachProperties) {
        messagesAdapter.setBrandingProperties(attachProperties);

        Color editorBackground = attachProperties.getChatEditorBackgroundColor();
        Color textColor = Color.getForegroundColor(editorBackground);

        findViewById(R.id.emptyChatView).setBackgroundColor(android.graphics.Color.parseColor(attachProperties.getChatViewerBackgroundColor().value));
        messagesList.setBackgroundColor(android.graphics.Color.parseColor(attachProperties.getChatViewerBackgroundColor().value));
        sendMessageContainer.setBackgroundColor(android.graphics.Color.parseColor(attachProperties.getChatEditorBackgroundColor().value));

        Color editorBackgroundColor = Color.getForegroundColor(attachProperties.getChatEditorBackgroundColor());
        sendButton.setColorFilter(android.graphics.Color.parseColor(editorBackgroundColor.value));

        messageBody.setTextColor(android.graphics.Color.parseColor(textColor.value));
        messageBody.setHintTextColor(android.graphics.Color.parseColor(textColor.value));

        AppCompatImageView emptyListIcon = findViewById(R.id.chatView_emptyListIcon);
        if (attachProperties.getChatViewerBackgroundColor().isLightColor()) {
            emptyListIcon.setColorFilter(android.graphics.Color.parseColor(Color.GRAY.value));
        } else {
            emptyListIcon.setColorFilter(android.graphics.Color.parseColor(Color.WHITE.value));
        }
    }

    private void getMoreMessages() {
        attachSdkInternal.fetchMessages();
    }

    private void sendMessage() {
        if (TextUtils.isEmpty(messageBody.getText())) {
            return;
        }
        attachSdkInternal.sendMessage(messageBody.getText().toString());
        messageBody.setText("");

    }

    public void setMessages(List<Message> messages) {
        List<Message> chatMessages = new ArrayList<>();
        for (Message m : messages) {
            if (m.getCategory() == Message.Category.CHAT)
                chatMessages.add(m);
        }
        messagesAdapter.setMessages(chatMessages);
        messagesList.scrollToPosition(0);
    }
}
